/*
 * #%L
 * Lima Swing
 * 
 * $Id: FinancialTransactionTable.java 3202 2011-07-08 12:53:14Z vsalaun $
 * $HeadURL: http://svn.chorem.org/svn/lima/tags/lima-0.5/lima-swing/src/main/java/org/chorem/lima/ui/financialtransaction/FinancialTransactionTable.java $
 * %%
 * Copyright (C) 2008 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package org.chorem.lima.ui.financialtransaction;

import java.awt.Color;
import java.awt.Component;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.math.BigDecimal;
import java.util.Date;
import javax.swing.SwingWorker;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.lima.entity.Account;
import org.chorem.lima.entity.Entry;
import org.chorem.lima.entity.EntryBook;
import org.chorem.lima.entity.FinancialTransaction;
import org.chorem.lima.ui.celleditor.AccountTableCellEditor;
import org.chorem.lima.ui.celleditor.BigDecimalTableCellRenderer;
import org.chorem.lima.ui.celleditor.DateTableCellEditor;
import org.chorem.lima.ui.celleditor.EmptyCellRenderer;
import org.chorem.lima.ui.celleditor.EntryBookTableCellEditor;
import org.chorem.lima.ui.celleditor.BigDecimalTableCellEditor;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ColorHighlighter;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;


/**
 * Table des transaction qui ajoute des comportement (keys).
 * 
 * @author jpepin
 */
public class FinancialTransactionTable extends JXTable
        implements KeyListener, MouseListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = 3133690382049594727L;
    
    /** log. */
    private static final Log log = LogFactory
            .getLog(FinancialTransactionTable.class);

    protected FinancialTransactionViewHandler handler;

    private Highlighter colorTransaction;

    private ColorHighlighter colorBalance;
    
    private int x_tab;
    private int y_tab;

    public FinancialTransactionTable(FinancialTransactionViewHandler handler) {

        this.handler = handler;
        
        addKeyListener(this);
        addMouseListener(this);
        
        //Get new date editor
        setDefaultEditor(Date.class, new DateTableCellEditor());
        //Get new entry book editor
        setDefaultEditor(EntryBook.class, new EntryBookTableCellEditor());
        //Get new account editor
        setDefaultEditor(Account.class, new AccountTableCellEditor());
        //Get new amount editor
        setDefaultEditor(BigDecimal.class, new BigDecimalTableCellEditor());
        //Get new BigDecimal renderer
        setDefaultRenderer(BigDecimal.class, new BigDecimalTableCellRenderer());
        //get new String renderer for empty cells
        setDefaultRenderer(String.class, new EmptyCellRenderer());
        //get new Account renderer for empty cells
        setDefaultRenderer(Account.class, new EmptyCellRenderer());
        //get new EntryBook renderer for empty cells
        setDefaultRenderer(EntryBook.class, new EmptyCellRenderer());
                
        //highlight financial financial transactions
        addColorTransaction();
        // highlight unbalanced financial transactions
        addColorNonBalancedTransaction();
    }
    
    /**
     * Cette méthode permet de colorer toutes les transactions dans le tableau
     * afin de bien distinguer les transactions et entrées comptables.
     * On récupère la première cellule, on vérifie que c'est une date
     */
   protected void addColorTransaction() {
       if (colorTransaction != null) {
           removeHighlighter(colorTransaction);
       }
       HighlightPredicate predicate = new HighlightPredicate() {
           @Override
           public boolean isHighlighted(Component renderer,
                   ComponentAdapter adapter) {
               return adapter.getValueAt(adapter.row, 0) instanceof Date;
           }
       };
        colorTransaction =
            new ColorHighlighter(predicate, new Color(222,222,222), null);
        addHighlighter(colorTransaction);
    }

   
   
    /**
     * Permet de surligner une transaction dans le tableau lorsque
     * cette dernière n'est pas équilibrée.
     * On récupère la dernière cellule de la ligne
     * et on vérifie si la valeur est différente de 0
     */
    protected void addColorNonBalancedTransaction() {
        if (colorBalance != null) {
            removeHighlighter(colorBalance);
        }
        HighlightPredicate predicate = new HighlightPredicate() {
            @Override
            public boolean isHighlighted(Component renderer,
                    ComponentAdapter adapter) {
                boolean isHighlighted = false;
                Object value = adapter.getValueAt(adapter.row, 8);
                if (value instanceof BigDecimal) {
                    BigDecimal currentBalance = (BigDecimal) value;
                    // can compare two BigDecimals with different scales
                    // e.g: 3.1 == 3.10
                    if (currentBalance.compareTo(BigDecimal.ZERO) != 0) {
                        isHighlighted = true;
                    }
                }
                return isHighlighted;
            }
        };
        colorTransaction = 
            new ColorHighlighter(predicate, new Color(255, 198, 209), null);
        addHighlighter(colorTransaction);
    }

    /**
     * for each action combination key are think
     * for extend keyboard and laptop keyboard
     */
    @Override
    public void keyPressed(KeyEvent e) {

        //TODO combinaison de touches dans la config
        
        // delete selected row with the key : delete or ctrl + clear
        // ou de l'entree
        if ((e.getKeyCode() == KeyEvent.VK_DELETE )
           || (e.getKeyCode() == KeyEvent.VK_CLEAR
               && e.getModifiers() == KeyEvent.CTRL_MASK)){
            handler.deleteSelectedRow();
        }

        // add entry with the key combination : insert or ctrl + enter
        if ((e.getKeyCode() == KeyEvent.VK_INSERT )
           || (e.getKeyCode() == KeyEvent.VK_ENTER
               && e.getModifiers() == KeyEvent.CTRL_MASK)) {
            handler.addEmptyEntry();
        }
        
        // copy : ctrl + c

        // add financial transaction with the key combination : ctrl + c
        if  (e.getKeyCode() == KeyEvent.VK_C
            && e.getModifiers() == KeyEvent.CTRL_MASK) {
            handler.copyRow();
        }
        
        // paste : ctrl + v
        
        // add financial transaction with the key combination : ctrl + v
        if  (e.getKeyCode() == KeyEvent.VK_V
            && e.getModifiers() == KeyEvent.CTRL_MASK) {
            handler.pasteRow();
        }
        
        // add financial transaction with the key combination : ctrl + tab
        if  (e.getKeyCode() == KeyEvent.VK_TAB
            && e.getModifiers() == KeyEvent.CTRL_MASK) {
            handler.addFinancialTransaction();
        }
        
        // clear row selection with the key: escape
        if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
        	if (!this.isEditing()) {
                this.clearSelection();
        	}
        }
        
        /**
         * Touche tab
         * Incrémente le curseur de case tant que la case n'est pas editable
         * Ajoute une entrée comptable si tab est sur
         * la dernière cellule et si la transaction est non équilibré
         * sinon rajoute une transaction
         */
        
        if (e.getKeyChar() == KeyEvent.VK_TAB) {
            int max_x = this.getColumnCount();
            int max_y = this.getRowCount();
            x_tab = this.getSelectedColumn();
            y_tab = this.getSelectedRow();
            Boolean end = true;
            
            if (x_tab < max_x - 1){
                x_tab++;
            }
            //end of row
            else {
                x_tab=0;
                y_tab++;
            }
            
            //skip all cell while not editable or if end of table add entry or transaction or end of table
            while (!isCellEditable(y_tab, x_tab) && end ){
                //if end of row
                if (x_tab == max_x-1) {
                    Object object = this.handler.tableModel.getElementAt(y_tab);
                    
                    //if transaction, add entry
                    if (object instanceof FinancialTransaction){
                        FinancialTransaction financialTransaction = (FinancialTransaction) object;
                        if (financialTransaction.getEntry().size() == 0){
                            handler.addEmptyEntry();
                            this.setColumnSelectionInterval(1, 1);
                        }                        
                    }
                    //if entry
                    else {
                        //FIXME set value is doing after key pressed
                        // so update not terminated before get balanced
                        //Swing Worker stop the UI 500ms
                        // found best solution ?
                        final JXTable table = this;
                        final int y_t = y_tab;
                        final int m_t = max_y;
                        new SwingWorker<Void,Void>() {
                            @Override protected Void doInBackground() throws InterruptedException {
                                Thread.sleep(500);
                                return null;
                            }
                            @Override protected void done() {
                                Object object = handler.tableModel.getElementAt(y_tab-1);
                                FinancialTransaction financialTransaction = null;
                                if (object instanceof Entry){
                                    financialTransaction = ((Entry) object).getFinancialTransaction();
                                }
                                else if (object instanceof FinancialTransaction){
                                    financialTransaction = (FinancialTransaction) object;
                                }
                                BigDecimal amountC = financialTransaction.getAmountCredit();
                                BigDecimal amountD = financialTransaction.getAmountDebit();
                                if (amountC == amountD){
                                    if (y_t == m_t-1){
                                        handler.addFinancialTransaction();
                                        table.setColumnSelectionInterval(0, 0);
                                    }
                                }
                                else {
                                    handler.addEmptyEntry();
                                    table.setColumnSelectionInterval(1, 1);
                                    y_tab++;
                                    // positionne la sélection sur la nouvelle ligne créée
                                    table.setRowSelectionInterval(y_tab, y_tab);
                                    x_tab=0;
                                }
                            }
                        }.execute();
                    }
                    end = false;
                }
                else {
                    if (x_tab < max_x){
                        this.setRowSelectionInterval(y_tab, y_tab);
                        this.setColumnSelectionInterval(x_tab, x_tab);
                        x_tab++;
                    }
                }
            }
        }
        
        
    }


    @Override
    public void keyTyped(KeyEvent e) {
        
    }


    @Override
    public void keyReleased(KeyEvent e) {
    }

    @Override
    public void mouseClicked(MouseEvent e) {
    }

    @Override
    public void mousePressed(MouseEvent e) {
        if (this.rowAtPoint(e.getPoint()) == -1) {
            this.clearSelection();
        }
    }

    @Override
    public void mouseReleased(MouseEvent e) {
    }

    @Override
    public void mouseEntered(MouseEvent e) {
    }

    @Override
    public void mouseExited(MouseEvent e) {
    }

}
