package fr.ifremer.tutti.ui.swing.action;

/*
 * #%L
 * Tutti :: UI
 * $Id: CreateBenthosMelagAction.java 1581 2014-02-08 15:06:19Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/action/CreateBenthosMelagAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchTableModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency.BenthosFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.util.species.EnterMelagWeightUI;
import jaxx.runtime.SwingUtil;
import org.apache.commons.lang3.StringUtils;

import javax.swing.JOptionPane;
import java.util.Map;

import static org.nuiton.i18n.I18n.t;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.0.2
 */
public class CreateBenthosMelagAction extends AbstractTuttiAction<BenthosBatchUIModel, BenthosBatchUI, BenthosBatchUIHandler> {

    protected Map<BenthosBatchRowModel, Float> selectedRows = Maps.newHashMap();

    protected Float melagWeight = null;

    protected Float sampleMelagWeight = null;

    protected Float sortedWeight = null;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public CreateBenthosMelagAction(BenthosBatchUIHandler handler) {
        super(handler, false);
        weightUnit = getConfig().getBenthosWeightUnit();
    }

    @Override
    public boolean prepareAction() throws Exception {
        boolean result = super.prepareAction();

        int[] selectedRowIndexes = SwingUtil.getSelectedModelRows(handler.getTable());

        BenthosBatchTableModel tableModel = handler.getTableModel();

        // sum of the weights already entered by the user
        Float alreadyKnownWeights = 0f;
        sortedWeight = 0f;

        for (int selectedRowIndex : selectedRowIndexes) {
            BenthosBatchRowModel selectedRow =
                    tableModel.getEntry(selectedRowIndex);

            Float sampleWeight = null;
            if (selectedRow.getFinestCategory().getNotNullWeight() != null) {
                alreadyKnownWeights += selectedRow.getFinestCategory().getNotNullWeight();

            } else if (selectedRow.getWeight() != null) {
                sampleWeight = selectedRow.getWeight();

            } else {
                sampleWeight = 0f;
                if (selectedRow.getFrequency() != null) {
                    for (BenthosFrequencyRowModel frequency : selectedRow.getFrequency()) {
                        if (frequency.getWeight() == null) {
                            sampleWeight = 0f;
                            break;
                        }
                        sampleWeight += frequency.getWeight();
                    }
                }
            }

            if (sampleWeight != null) {
                if (sampleWeight == 0f) {
                    JOptionPane.showMessageDialog(
                            getUI(),
                            t("tutti.createBenthosMelag.error.message", selectedRowIndex + 1),
                            t("tutti.createBenthosMelag.error.title"),
                            JOptionPane.ERROR_MESSAGE);

                    SwingUtil.setSelectionInterval(handler.getTable(), selectedRowIndex);
                    result = false;
                    break;

                } else {
                    sortedWeight += sampleWeight;
                }
            }

            selectedRows.put(selectedRow, sampleWeight);
        }

        if (result) {
            melagWeight = openMelagWeightDialog();

            if (melagWeight != null) {
                // substract the weights that the user already entered
                // they must not be used to compute the other weights
                sampleMelagWeight = melagWeight - alreadyKnownWeights;
            }
        }

        return result && melagWeight != null;
    }

    protected Float openMelagWeightDialog() {
        EnterMelagWeightUI dialog = new EnterMelagWeightUI(getContext());
        Float result = dialog.openAndGetWeightValue(weightUnit);
        return result;
    }

    @Override
    public void doAction() throws Exception {
        String unitLabel = weightUnit.getShortLabel();
        String melagComment = t("tutti.createBenthosMelag.comment.part1") + "\n";
        for (BenthosBatchRowModel batch : selectedRows.keySet()) {
            Float sampleWeight = selectedRows.get(batch);
            if (sampleWeight != null) {
                Float weight = sampleMelagWeight * sampleWeight / sortedWeight;
                batch.getFinestCategory().setCategoryWeight(weight);
            }
            melagComment += t("tutti.createBenthosMelag.comment.part2", sampleWeight, unitLabel, decorate(batch.getSpecies())) + "\n";
        }
        melagComment += t("tutti.createBenthosMelag.comment.part3", melagWeight, unitLabel);

        for (BenthosBatchRowModel batch : selectedRows.keySet()) {
            String comment = batch.getComment();
            if (StringUtils.isBlank(comment)) {
                comment = "";

            } else {
                comment += "\n";
            }
            comment += melagComment;
            batch.setComment(comment);
        }
        getHandler().saveRows(selectedRows.keySet());
    }

    @Override
    public void releaseAction() {
        super.releaseAction();
        //FIXME-TC No release action is not design to refresh ui...
//        getHandler().getTable().repaint();
        selectedRows.clear();
        melagWeight = null;
        sortedWeight = null;
    }

    @Override
    public void postSuccessAction() {
        super.postSuccessAction();
        getHandler().getTable().repaint();

    }
}
