package fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesFrequencyTableModel.java 1385 2013-11-24 23:27:27Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-3.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/frequency/SpeciesFrequencyTableModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import fr.ifremer.shared.application.type.WeightUnit;
import fr.ifremer.shared.application.swing.table.AbstractApplicationTableModel;
import fr.ifremer.shared.application.swing.table.ColumnIdentifier;
import org.jdesktop.swingx.table.TableColumnModelExt;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;
import java.util.Map;

import static org.nuiton.i18n.I18n.n_;

/**
 * Model of the species frequency table.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesFrequencyTableModel extends AbstractApplicationTableModel<SpeciesFrequencyRowModel> {

    private static final long serialVersionUID = 1L;

    public static final ColumnIdentifier<SpeciesFrequencyRowModel> LENGTH_STEP = ColumnIdentifier.newId(
            SpeciesFrequencyRowModel.PROPERTY_LENGTH_STEP,
            n_("tutti.editSpeciesFrequencies.table.header.lengthStep"),
            n_("tutti.editSpeciesFrequencies.table.header.lengthStep"));

    public static final ColumnIdentifier<SpeciesFrequencyRowModel> NUMBER = ColumnIdentifier.newId(
            SpeciesFrequencyRowModel.PROPERTY_NUMBER,
            n_("tutti.editSpeciesFrequencies.table.header.number"),
            n_("tutti.editSpeciesFrequencies.table.header.number"));

    public static final ColumnIdentifier<SpeciesFrequencyRowModel> WEIGHT = ColumnIdentifier.newId(
            SpeciesFrequencyRowModel.PROPERTY_WEIGHT,
            n_("tutti.editSpeciesFrequencies.table.header.weight"),
            n_("tutti.editSpeciesFrequencies.table.header.weight"));

    private final SpeciesFrequencyUIModel uiModel;

    private final Map<Float, SpeciesFrequencyRowModel> rowCache;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public SpeciesFrequencyTableModel(WeightUnit weightUnit,
                                      TableColumnModelExt columnModel,
                                      SpeciesFrequencyUIModel uiModel) {
        super(columnModel, true, true);
        this.weightUnit = weightUnit;
        this.uiModel = uiModel;
        this.rowCache = Maps.newTreeMap();
        setNoneEditableCols();
    }

    @Override
    public SpeciesFrequencyRowModel createNewRow() {
        Float defaultStep = null;

        int rowCount = getRowCount();
        if (rowCount > 0) {

            SpeciesFrequencyRowModel rowModel = getEntry(rowCount - 1);
            Float lengthStep = rowModel.getLengthStep();
            if (lengthStep != null) {
                defaultStep = uiModel.getLengthStep(
                        lengthStep + uiModel.getStep());
            }
        }
        SpeciesFrequencyRowModel result =
                new SpeciesFrequencyRowModel(weightUnit);
        result.addPropertyChangeListener(SpeciesFrequencyRowModel.PROPERTY_LENGTH_STEP,
                                         new PropertyChangeListener() {
                                             @Override
                                             public void propertyChange(PropertyChangeEvent evt) {
                                                 Float oldValue = (Float) evt.getOldValue();
                                                 if (oldValue != null) {
                                                     rowCache.remove(oldValue);
                                                 }
                                                 SpeciesFrequencyRowModel row = (SpeciesFrequencyRowModel) evt.getSource();
                                                 Float lengthStep = row.getLengthStep();
                                                 if (lengthStep != null) {
                                                     rowCache.put(lengthStep, row);
                                                 }
                                             }
                                         });
        result.setLengthStepCaracteristic(uiModel.getLengthStepCaracteristic());
        result.setLengthStep(defaultStep);
        result.setValid(defaultStep != null);
        return result;
    }

    @Override
    public void setValueAt(Object aValue,
                           int rowIndex,
                           int columnIndex,
                           ColumnIdentifier<SpeciesFrequencyRowModel> propertyName,
                           SpeciesFrequencyRowModel entry) {
        super.setValueAt(aValue, rowIndex, columnIndex, propertyName, entry);
        // TODO Rebuild the computedWeight if possible...
    }

    public Map<Float, SpeciesFrequencyRowModel> getRowCache() {
        return rowCache;
    }

    @Override
    protected void onRowsChanged(List<SpeciesFrequencyRowModel> data) {

        // rebuild row cache
        rowCache.clear();

        for (SpeciesFrequencyRowModel row : data) {
            Float lengthStep = row.getLengthStep();
            if (lengthStep != null) {
                rowCache.put(lengthStep, row);
            }
        }
    }
}