package fr.ifremer.tutti.ui.swing.content.db;

/*
 * #%L
 * Tutti :: UI
 * $Id: ImportDbAction.java 1223 2013-09-25 13:31:19Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.7/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/db/ImportDbAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.TuttiBusinessException;
import fr.ifremer.tutti.persistence.ProgressionModel;
import fr.ifremer.tutti.persistence.TuttiPersistence;
import fr.ifremer.tutti.persistence.config.TuttiPersistenceConfig;
import fr.ifremer.tutti.ui.swing.content.AbstractMainUITuttiAction;
import fr.ifremer.tutti.ui.swing.content.MainUIHandler;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Version;

import javax.swing.JOptionPane;
import java.io.File;

import static org.nuiton.i18n.I18n._;

/**
 * To import a db and use it.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.1
 */
public class ImportDbAction extends AbstractMainUITuttiAction {

    /** Logger. */
    private static final Log log = LogFactory.getLog(ImportDbAction.class);

    protected File importFile;

    protected String jdbcUrl;

    protected boolean updateSchema;

    protected boolean closeDb;

    protected Version dbVersion;

    protected Version applicationVersion;

    public ImportDbAction(MainUIHandler handler) {
        super(handler, true);
        setActionDescription(_("tutti.dbManager.action.importDb.tip"));
    }

    public void setImportFile(File importFile) {
        this.importFile = importFile;
    }

    @Override
    public boolean prepareAction() throws Exception {

        jdbcUrl = null;
        dbVersion = applicationVersion = null;
        closeDb = updateSchema = false;

        boolean doAction = super.prepareAction();

        if (doAction) {

            jdbcUrl = TuttiPersistenceConfig.getInstance().getJdbcUrl();

            if (importFile == null) {

                // choose file to import
                importFile = chooseFile(
                        _("tutti.dbManager.title.choose.dbImportFile"),
                        _("tutti.dbManager.action.chooseDbFile"),
                        "^.*\\.zip", _("tutti.common.file.zip")
                );

                if (importFile == null) {

                    displayWarningMessage(
                            _("tutti.dbManager.title.choose.dbImportFile"),
                            _("tutti.dbManager.action.importdb.no.import.file.choosen")
                    );

                    doAction = false;
                }
            }

            if (doAction) {

                ProgressionModel progressionModel = new ProgressionModel();
                progressionModel.setTotal(3);
                setProgressionModel(progressionModel);

                getContext().getPersistenceService().checkImportStructure(importFile);
            }
        }
        return doAction;
    }

    @Override
    public void releaseAction() {
        importFile = null;
        super.releaseAction();
    }

    @Override
    public void doAction() throws Exception {
        Preconditions.checkNotNull(importFile);

        if (log.isInfoEnabled()) {
            log.info("Will import db: " + importFile);
        }

        ProgressionModel progressionModel = getProgressionModel();

        // ------------------------------------------------------------------ //
        // --- import db                                                      //
        // ------------------------------------------------------------------ //

        progressionModel.increments(_("tutti.importDb.step.unzipArchive"));

        getContext().getPersistenceService().importDb(importFile);

        // ------------------------------------------------------------------ //
        // --- open db                                                        //
        // ------------------------------------------------------------------ //

        progressionModel.increments(_("tutti.importDb.step.openDb", jdbcUrl));
        try {
            getContext().setDbExist(true);
            getContext().setDbLoaded(true);
            getHandler().reloadPersistenceService();
        } catch (Exception e) {

            if (log.isErrorEnabled()) {
                log.error("Could not open db", e);
            }
            // no more db
            getContext().setDbLoaded(false);
            getHandler().reloadPersistenceService();

            // could not load db
            throw new TuttiBusinessException(_("tutti.dbManager.action.importdb.couldNotOpen"), e);
        }

        // ------------------------------------------------------------------ //
        // --- check schema version                                           //
        // ------------------------------------------------------------------ //

        progressionModel.increments(_("tutti.importDb.step.checkSchemaVersion"));

        TuttiPersistence persistenceService =
                getHandler().getPersistenceService();

        dbVersion = persistenceService.getDbVersion();

        if (log.isInfoEnabled()) {
            log.info("Detected database version: " + dbVersion);
        }
        applicationVersion = persistenceService.getApplicationVersion();

        if (log.isInfoEnabled()) {
            log.info("Detected schema application version:" + applicationVersion);
        }

        if (dbVersion == null) {

            // no database version filled (can not migrate schema)
            String message = _("tutti.dbManager.action.upgradeDb.schema.version.not.found");
            displayWarningMessage(_("tutti.dbManager.title.schema.toupdate"), message);

            closeDb = true;
        } else if (dbVersion.equals(applicationVersion)) {

            // database schema is up to date
            if (log.isInfoEnabled()) {
                log.info("Database schema is up-to-date at version: " + dbVersion);
            }
        } else if (dbVersion.compareTo(applicationVersion) < 0) {

            // database schema need to migrate
            // ask user to migrate or not (if not will close db)

            String message = _("tutti.dbManager.action.upgradeDb.schema.to.update.message", dbVersion, applicationVersion);

            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    message,
                    _("tutti.dbManager.action.upgradeDb.schema.to.update.message.help"));
            int i = JOptionPane.showConfirmDialog(
                    getHandler().getTopestUI(),
                    htmlMessage,
                    _("tutti.dbManager.title.schema.toupdate"),
                    JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.QUESTION_MESSAGE);
            boolean continueAction = i == JOptionPane.OK_OPTION;

            if (continueAction) {

                // will migrate
                updateSchema = true;
            } else {

                //close db
                closeDb = true;
            }


        } else {

            // database schema version is higher than application one
            String message = _("tutti.dbManager.action.upgradeDb.schema.too.high", dbVersion, applicationVersion);

            String htmlMessage = String.format(
                    AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                    message,
                    _("tutti.dbManager.action.upgradeDb.schema.too.high.help"));
            int i = JOptionPane.showConfirmDialog(
                    getHandler().getTopestUI(),
                    htmlMessage,
                    _("tutti.dbManager.title.schema.toupdate"),
                    JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.QUESTION_MESSAGE);
            boolean continueAction = i == JOptionPane.OK_OPTION;

            if (!continueAction) {

                //close db
                closeDb = true;
            }
        }

        if (closeDb) {

            // ------------------------------------------------------------------ //
            // --- close db                                                       //
            // ------------------------------------------------------------------ //

            progressionModel.increments(_("tutti.importDb.step.closeDb"));
            TuttiActionHelper.runInternalAction(getHandler(), CloseDbAction.class);

            return;
        }

        if (updateSchema) {

            // need to migrate schema
            progressionModel.adaptTotal(progressionModel.getTotal() + 2);

            // ------------------------------------------------------------------ //
            // --- update schema                                                  //
            // ------------------------------------------------------------------ //

            String message = _("tutti.importDb.step.will.migrateSchema",
                               dbVersion, applicationVersion);

            progressionModel.increments(message);
            sendMessage(message);
            getHandler().getPersistenceService().updateSchema();

            sendMessage(_("tutti.flash.info.db.schema.updated", dbVersion, applicationVersion));
        }

        // ------------------------------------------------------------------ //
        // --- check db context                                               //
        // ------------------------------------------------------------------ //

        String message = _("tutti.importDb.step.check.dbContext",
                           dbVersion, applicationVersion);

        progressionModel.increments(message);

        if (log.isDebugEnabled()) {
            log.debug("Check db context");
        }
        getContext().checkDbContext();
    }

    @Override
    public void postSuccessAction() {
        handler.reloadDbManagerText();

        super.postSuccessAction();

        if (closeDb) {
            sendMessage(_("tutti.flash.info.db.imported.but.closed", jdbcUrl));
        } else {
            sendMessage(_("tutti.flash.info.db.imported", jdbcUrl));
        }

        // make sure title is reloaded
        getUI().getHandler().changeTitle();
    }

    @Override
    public void postFailedAction(Throwable error) {
        handler.reloadDbManagerText();
        super.postFailedAction(error);
    }

}