package fr.ifremer.tutti.ui.swing.content.operation.catches.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchUIHandler.java 694 2013-03-27 21:56:27Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/SpeciesBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.BatchContainer;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategory;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SampleCategoryComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.TableViewMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.create.CreateSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.create.CreateSpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyCellComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.editor.TuttiComputedOrNotDataTableCell;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUI;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUIModel;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.JOptionPane;
import javax.swing.RowFilter;
import javax.swing.UIManager;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n._;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SpeciesBatchUIHandler extends AbstractTuttiBatchTableUIHandler<SpeciesBatchRowModel, SpeciesBatchUIModel, SpeciesBatchUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesBatchUIHandler.class);

    public static final Set<String> SAMPLING_PROPERTIES = Sets.newHashSet(
            SpeciesBatchRowModel.PROPERTY_SAMPLE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SPECIES,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT);

    private final EnumMap<TableViewMode, RowFilter<SpeciesBatchTableModel, Integer>> tableFilters;

    public SpeciesBatchUIHandler(TuttiUI<?, ?> parentUi,
                                 SpeciesBatchUI ui) {
        super(parentUi, ui,
              SpeciesBatchRowModel.PROPERTY_SPECIES,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_NUMBER,
              SpeciesBatchRowModel.PROPERTY_COMMENT,
              SpeciesBatchRowModel.PROPERTY_ATTACHMENT,
              SpeciesBatchRowModel.PROPERTY_FREQUENCY,
              SpeciesBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM);
        tableFilters = new EnumMap<TableViewMode, RowFilter<SpeciesBatchTableModel, Integer>>(TableViewMode.class);

        tableFilters.put(TableViewMode.ALL, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                return true;
            }
        });

        tableFilters.put(TableViewMode.ROOT, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    SpeciesBatchTableModel model = entry.getModel();
                    SpeciesBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchRoot();
                }
                return result;
            }
        });

        tableFilters.put(TableViewMode.LEAF, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    SpeciesBatchTableModel model = entry.getModel();
                    SpeciesBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchLeaf();
                }
                return result;
            }
        });
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        SpeciesBatchUIModel model = getModel();

        List<SpeciesBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            model.removeAllAttachment(model.getAttachment());

            if (!TuttiEntities.isNew(bean)) {

                // get all batch species root (says the one with only a species sample category)
                BatchContainer<SpeciesBatch> rootSpeciesBatch =
                        persistenceService.getRootSpeciesBatch(bean.getId());

                model.setRootBatchId(rootSpeciesBatch.getId());

                List<Attachment> attachments =
                        persistenceService.getAllAttachments(Integer.valueOf(model.getObjectId()));
                model.addAllAttachment(attachments);

                if (log.isInfoEnabled()) {
                    log.info("species root batch id: " + model.getRootBatchId());
                }

                List<SpeciesBatch> catches = rootSpeciesBatch.getChildren();

                for (SpeciesBatch aBatch : catches) {

                    // root batch sample categroy is species
                    Preconditions.checkState(
                            aBatch.getSampleCategoryType() == SampleCategoryEnum.sortedUnsorted,
                            "Root species batch must be a sortedUnsorted sample " +
                            "category but was:" + aBatch.getSampleCategoryType());

                    SpeciesBatchRowModel rootRow =
                            loadBatch(aBatch, null, rows);

                    if (log.isDebugEnabled()) {
                        log.debug("Loaded root batch " +
                                  decorate(rootRow.getSpecies(), DecoratorService.FROM_PROTOCOL) + " - " +
                                  decorate(rootRow.getSortedUnsortedCategoryValue()));
                    }
                }
            }
        }

        model.setRows(rows);
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public SpeciesBatchTableModel getTableModel() {
        return (SpeciesBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SpeciesBatchRowModel row) {

        // a row is valid if species category is not empty and valid
        // then if any of none empty category is valid
        boolean result = row.getSpecies() != null;
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SpeciesBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        if (SAMPLING_PROPERTIES.contains(propertyName)) {

            // species has changed, recompute valid property
            recomputeRowValidState(row);
        }

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SpeciesBatchRowModel> rowMonitor,
                                             SpeciesBatchRowModel row) {
        if (row.isValid()) {
            // there is a valid bean attached to the monitor
            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                showInformationMessage(
                        "[ Captures - Espèces ] " +
                        "Sauvegarde des modifications de " + row + '.');

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
            }
        } else {

            //FIXME See how to delete rows ? Or moreover how to save tehem...
            if (log.isWarnEnabled()) {
                log.warn("Will not remove not valid speciesBatch: " + row.getId());
            }
        }
    }

    @Override
    protected void onModelRowsChanged(List<SpeciesBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        SpeciesBatchUIModel model = getModel();
        model.setRootNumber(0);

        for (SpeciesBatchRowModel row : rows) {
            updateTotalFromFrequencies(row);
            if (row.isBatchRoot()) {

                // update speciesUsed
                addToSpeciesUsed(row);
            }
        }
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          SpeciesBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             SpeciesBatchRowModel oldRow,
                                             int newRowIndex,
                                             SpeciesBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);

        // when selected row has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<SpeciesBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        SpeciesBatchUIModel model = new SpeciesBatchUIModel(catchesUIModel);
        model.setTableViewMode(TableViewMode.ALL);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("afterInit: " + ui);
        }

        initUI(ui);

//        SwingUtil.applyDataBinding(
//                ui,
//                SpeciesBatchUI.BINDING_CREATE_SPECIES_MELAG_MENU_ENABLED,
//                SpeciesBatchUI.BINDING_REMOVE_SPECIES_BATCH_MENU_ENABLED,
//                SpeciesBatchUI.BINDING_REMOVE_SPECIES_SUB_BATCH_MENU_ENABLED,
//                SpeciesBatchUI.BINDING_RENAME_SPECIES_BATCH_MENU_ENABLED);

        Map<Integer, SampleCategoryEnum> categoryEnumMap =
                SampleCategoryEnum.toIdMapping();

        List<SampleCategoryEnum> samplingOrder = Lists.newArrayList();

        List<Integer> samplingOrderIds = getConfig().getSamplingOrderIds();
        for (Integer id : samplingOrderIds) {
            samplingOrder.add(categoryEnumMap.get(id));
        }
        if (log.isInfoEnabled()) {
            log.info("Will use sampling order: " + samplingOrder);
        }

        JXTable table = getTable();

        // can show / hide some columns in model
        table.setColumnControlVisible(true);

        // create table column model
        TableCellRenderer defaultRenderer =
                table.getDefaultRenderer(Object.class);

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        Decorator<CaracteristicQualitativeValue> caracteristicDecorator =
                getDecorator(CaracteristicQualitativeValue.class, null);

        Color computedDataColor = getConfig().getColorComputedWeights();

        { // Species column

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Species.class, DecoratorService.FROM_PROTOCOL),
                             SpeciesBatchTableModel.SPECIES);
        }

        { // SortedUnsortedCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SORTED_UNSORTED_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        for (SampleCategoryEnum sampleCategoryType : samplingOrder) {
            switch (sampleCategoryType) {

                case size:

                { // SizeCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   SpeciesBatchTableModel.SIZE_CATEGORY,
                                                   caracteristicDecorator,
                                                   defaultRenderer);
                }
                break;
                case sex:

                { // SexCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   SpeciesBatchTableModel.SEX_CATEGORY,
                                                   caracteristicDecorator,
                                                   defaultRenderer);
                }
                break;
                case maturity:

                { // MaturityCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   SpeciesBatchTableModel.MATURITY_CATEGORY,
                                                   caracteristicDecorator,
                                                   defaultRenderer);
                }

                break;
                case age:


                { // AgeCategory column

                    addSampleCategoryColumnToModel(columnModel,
                                                   SpeciesBatchTableModel.AGE_CATEGORY,
                                                   getDecorator(Float.class, null),
                                                   defaultRenderer);
                }

                break;
            }
        }


        { // Weight column

            addColumnToModel(columnModel,
                             TuttiComputedOrNotDataTableCell.newEditor(
                                     Float.class, false, true, 3, computedDataColor),
                             TuttiComputedOrNotDataTableCell.newRender(
                                     defaultRenderer, true, 3, computedDataColor),
                             SpeciesBatchTableModel.WEIGHT);
        }

        { // Number column (from frequencies)

            addColumnToModel(columnModel,
                             SpeciesFrequencyCellComponent.newEditor(ui, computedDataColor),
                             SpeciesFrequencyCellComponent.newRender(computedDataColor),
                             SpeciesBatchTableModel.COMPUTED_NUMBER);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             SpeciesBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             SpeciesBatchTableModel.ATTACHMENT);
        }

        { // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    SpeciesBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        // create table model
        SpeciesBatchTableModel tableModel =
                new SpeciesBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);

        getModel().addPropertyChangeListener(SpeciesBatchUIModel.PROPERTY_TABLE_VIEW_MODE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                TableViewMode tableViewMode = (TableViewMode) evt.getNewValue();

                if (tableViewMode == null) {
                    tableViewMode = TableViewMode.ALL;
                }

                if (log.isDebugEnabled()) {
                    log.debug("Will use rowfilter for viewMode: " + tableViewMode);
                }
                RowFilter<SpeciesBatchTableModel, Integer> filter = tableFilters.get(tableViewMode);
                getTable().setRowFilter(filter);
            }
        });

        recomputeBatchActionEnable();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("Closing: " + ui);
        }
        ui.getSpeciesBatchAttachmentsButton().onCloseUI();
    }

    @Override
    protected void addHighlighters(JXTable table) {

        super.addHighlighters(table);

        Color toConfirmColor = getConfig().getColorRowToConfirm();

        // paint the cell in orange if the row is to confirm
        Highlighter confirmHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate() {

                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                        SpeciesBatchRowModel row = getTableModel().getEntry(adapter.row);
                        return row.getSpeciesToConfirm();
                    }

                }, toConfirmColor);
        table.addHighlighter(confirmHighlighter);

        // highlight only the species column if the row is selected
        Highlighter selectedHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        HighlightPredicate.IS_SELECTED,
                        new HighlightPredicate.IdentifierHighlightPredicate(SpeciesBatchTableModel.SPECIES)),
                UIManager.getColor("Table[Enabled+Selected].textBackground"));

        table.addHighlighter(selectedHighlighter);

        // paint the cell in dark orange if the row is to confirm and the cell is not editable
        Highlighter confirmNotEditableHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate() {

                    public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                        SpeciesBatchRowModel row = getTableModel().getEntry(adapter.row);
                        return row.getSpeciesToConfirm() && !adapter.isEditable();
                    }

                }, toConfirmColor.darker());
        table.addHighlighter(confirmNotEditableHighlighter);

    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createBatch() {

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        CreateSpeciesBatchUI createBatchEditor = parent.getSpeciesTabCreateBatch();

        createBatchEditor.getHandler().openUI(getModel());
        parent.getHandler().setSpeciesSelectedCard(EditCatchesUIHandler.CREATE_BATCH_CARD);
    }

    public void addBatch(CreateSpeciesBatchUIModel speciesBatchRootRowModel) {
        if (speciesBatchRootRowModel.isValid()) {

            SpeciesBatchTableModel tableModel = getTableModel();

            SpeciesBatchRowModel newRow = tableModel.createNewRow();
            Species species = speciesBatchRootRowModel.getSpecies();
            newRow.setSpecies(species);

            CaracteristicQualitativeValue sortedUnsortedCategory = speciesBatchRootRowModel.getSortedUnsortedCategory();
            SampleCategory<CaracteristicQualitativeValue> category = newRow.getSortedUnsortedCategory();
            category.setCategoryValue(sortedUnsortedCategory);
            category.setCategoryWeight(speciesBatchRootRowModel.getBatchWeight());
            newRow.setSampleCategory(category);

            recomputeRowValidState(newRow);

            saveRow(newRow);

            tableModel.addNewRow(newRow);
            AbstractSelectTableAction.doSelectCell(getTable(), tableModel.getRowCount() - 1, 0);

            // update speciesUsed
            addToSpeciesUsed(newRow);
        }

        recomputeBatchActionEnable();
    }

    public void splitBatch() {

        JXTable table = getTable();

        // get selected row
        int rowIndex = table.getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant split batch if no batch selected");

        SpeciesBatchTableModel tableModel = getTableModel();

        SpeciesBatchRowModel parentBatch = tableModel.getEntry(rowIndex);

        boolean split = true;
        if (parentBatch.getWeight() != null) {
            String htmlMessage = String.format(
                    CONFIRMATION_FORMAT,
                    _("tutti.editSpeciesBatch.split.weightNotNull.message"),
                    _("tutti.editSpeciesBatch.split.weightNotNull.help"));
            int i = JOptionPane.showConfirmDialog(
                    getTopestUI(),
                    htmlMessage,
                    _("tutti.editSpeciesBatch.split.weightNotNull.title"),
                    JOptionPane.OK_CANCEL_OPTION);

            if (i == JOptionPane.OK_OPTION) {
                parentBatch.setWeight(null);

            } else {
                split = false;
            }
        }

        if (split) {
            if (log.isInfoEnabled()) {
                log.info("Open split batch ui for row [" + rowIndex + ']');
            }

            EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
            SplitSpeciesBatchUI splitBatchEditor = parent.getSpeciesTabSplitBatch();

            splitBatchEditor.getHandler().editBatch(parentBatch);
            parent.getHandler().setSpeciesSelectedCard(EditCatchesUIHandler.SPLIT_BATCH_CARD);
        }
    }

    public void splitBatch(SampleCategoryEnum sampleCategoryEnum, List<SplitSpeciesBatchRowModel> rows) {
        JXTable table = getTable();

        // get selected row
        int insertRow = table.getSelectedRow();

        SpeciesBatchTableModel tableModel = getTableModel();
        SpeciesBatchRowModel parentBatch = tableModel.getEntry(insertRow);

        // Create rows in batch table model

        List<SpeciesBatchRowModel> newBatches = Lists.newArrayList();
        for (SplitSpeciesBatchRowModel row : rows) {
            if (row.isValid()) {

                // can keep this row
                SpeciesBatchRowModel newBatch = tableModel.createNewRow();

                loadBatchRow(parentBatch,
                             newBatch,
                             sampleCategoryEnum,
                             row.getCategoryValue(),
                             row.getWeight());

                recomputeRowValidState(newBatch);
                newBatches.add(newBatch);

                tableModel.addNewRow(++insertRow, newBatch);
                AbstractSelectTableAction.doSelectCell(getTable(), insertRow, 0);
            }
        }

        // add new batches to his parent
        parentBatch.setChildBatch(newBatches);

        //TODO Should only save parentBatch (will persist all his childs)
        //saveRow(parentBatch);

        // save new batches
        saveRows(newBatches);

        SpeciesBatchUIModel model = getModel();
        model.setLeafNumber(model.getLeafNumber() + newBatches.size() - 1);

        recomputeBatchActionEnable();

    }

    public void updateTotalFromFrequencies(SpeciesBatchRowModel row) {
        List<SpeciesFrequencyRowModel> frequency = row.getFrequency();

        Integer totalNumber = 0;
        boolean onlyOneFrequency = false;
        if (CollectionUtils.isNotEmpty(frequency)) {
            for (SpeciesFrequencyRowModel frequencyModel : frequency) {
                if (frequencyModel.getNumber() != null) {
                    totalNumber += frequencyModel.getNumber();
                }
            }
            onlyOneFrequency = frequency.size() == 1;
        }
        row.setComputedNumber(totalNumber);
        row.getFinestCategory().setOnlyOneFrequency(onlyOneFrequency);

    }

    public void saveRows(Iterable<SpeciesBatchRowModel> rows) {
        for (SpeciesBatchRowModel row : rows) {
            saveRow(row);
        }
    }

    protected void saveRow(SpeciesBatchRowModel row) {

        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        Preconditions.checkNotNull(row.getSpecies());
        SampleCategory<?> sampleCategory = row.getSampleCategory();
        Preconditions.checkNotNull(sampleCategory);
        Preconditions.checkNotNull(sampleCategory.getCategoryType());
        Preconditions.checkNotNull(sampleCategory.getCategoryValue());

        SpeciesBatch catchBean = row.toBean();
        catchBean.setFishingOperation(fishingOperation);

        SpeciesBatchRowModel parent = row.getParentBatch();
        if (parent != null) {
            catchBean.setParentBatch(parent.toBean());
        }

        // apply sample category
        catchBean.setSampleCategoryType(sampleCategory.getCategoryType());
        catchBean.setSampleCategoryValue(sampleCategory.getCategoryValue());
        catchBean.setSampleCategoryWeight(sampleCategory.getCategoryWeight());

        if (TuttiEntities.isNew(catchBean)) {

            String parentBatchId = null;
            if (parent != null) {
                parentBatchId = parent.getId();
            }

            if (log.isInfoEnabled()) {
                log.info("Persist new species batch with parentId: " +
                         parentBatchId);
            }
            catchBean = persistenceService.createSpeciesBatch(catchBean,
                                                              parentBatchId);
            row.setId(catchBean.getId());
        } else {
            if (log.isInfoEnabled()) {
                log.info("Persist existing species batch: " + catchBean.getId() + " (parent : " + catchBean.getParentBatch() + ")");
            }
            persistenceService.saveSpeciesBatch(catchBean);
        }

        List<SpeciesFrequencyRowModel> frequencyRows = row.getFrequency();

        List<SpeciesBatchFrequency> frequency =
                SpeciesFrequencyRowModel.toBeans(frequencyRows, catchBean);

        if (log.isInfoEnabled()) {
            log.info("Will save " + frequency.size() + " frequencies.");
        }
        frequency = persistenceService.saveSpeciesBatchFrequency(
                catchBean.getId(), frequency);

        // push it back to row model
        frequencyRows = SpeciesFrequencyRowModel.fromBeans(frequency);
        row.setFrequency(frequencyRows);
    }

    public String getFilterSpeciesBatchRootButtonText(int rootNumber) {
        return _("tutti.editSpeciesBatch.filterBatch.mode.root", rootNumber);
    }

    protected void recomputeBatchActionEnable() {

        int rowIndex = getTable().getSelectedRow();

        //TODO Improve this test
        boolean enableAdd = true;
//                CollectionUtils.isNotEmpty(getModel().getAvailableSpecies());

        boolean enableRename = false;
        boolean enableSplit = false;
        boolean enableRemove = false;
        boolean enableRemoveSub = false;
        boolean enableCreateMelag = false;

        if (rowIndex != -1) {

            // there is a selected row

            //TODO If there is some sub-batch, can remove them
            //TODO If there is no sub-batch, can split current batch

            SpeciesBatchTableModel tableModel = getTableModel();
            SpeciesBatchRowModel row = tableModel.getEntry(rowIndex);
            int selectedRowCount = getTable().getSelectedRowCount();

            if (row.isValid()) {

                // must have at least species filled in row
                // otherwise nothing can be done

                enableSplit = true;
                enableRemove = true;
                enableRemoveSub = true;
                enableRename = true;
                enableCreateMelag = true;
            }

            if (enableSplit) {

                // can split if selected batch is a leaf
                enableSplit = row.isBatchLeaf()
                              && selectedRowCount == 1
                              && row.getNumber() == null
                              && (row.getComputedNumber() == null
                                  || row.getComputedNumber() == 0);
            }

            if (enableRename) {

                // can rename if selected batch is a parent
                enableRename = row.isBatchRoot()
                               && selectedRowCount == 1;
            }

            if (enableRemove) {

                // can always remove the batch
                enableRemove = selectedRowCount == 1;
            }

            if (enableRemoveSub) {

                // can remove sub batch if selected batch is not a leaf
                enableRemoveSub = !row.isBatchLeaf()
                                  && selectedRowCount == 1;
            }

            if (enableCreateMelag) {

                JXTable table = getTable();

                // can add species to a melag if several root are selected

                if (selectedRowCount < 2) {
                    enableCreateMelag = false;

                } else {
                    int[] selectedRows = table.getSelectedRows();
                    for (int selectedRowIndex : selectedRows) {
                        SpeciesBatchRowModel selectedRow =
                                tableModel.getEntry(selectedRowIndex);

                        if (!selectedRow.isBatchRoot()) {
                            enableCreateMelag = false;
                            break;
                        }
                    }
                }
            }
        }

        SpeciesBatchUIModel model = getModel();
        model.setCreateBatchEnabled(enableAdd);
        model.setSplitBatchEnabled(enableSplit);
        model.setRemoveBatchEnabled(enableRemove);
        model.setRemoveSubBatchEnabled(enableRemoveSub);
        model.setRenameSpeciesBatchEnabled(enableRename);
        model.setCreateSpeciesMelagEnabled(enableCreateMelag);
    }

    public void collectChildren(SpeciesBatchRowModel row,
                                Set<SpeciesBatchRowModel> collectedRows) {

        if (!row.isBatchLeaf()) {

            for (SpeciesBatchRowModel batchChild : row.getChildBatch()) {
                collectedRows.add(batchChild);
                collectChildren(batchChild, collectedRows);
            }
        }
    }

    protected SpeciesBatchRowModel loadBatch(SpeciesBatch aBatch,
                                             SpeciesBatchRowModel parentRow,
                                             List<SpeciesBatchRowModel> rows) {

        String id = aBatch.getId();

        List<SpeciesBatchFrequency> frequencies =
                persistenceService.getAllSpeciesBatchFrequency(id);

        List<Attachment> attachments =
                persistenceService.getAllAttachments(Integer.valueOf(id));

        SpeciesBatchRowModel newRow =
                new SpeciesBatchRowModel(aBatch, frequencies, attachments);
        // set the surveycode, do it only on the parent,
        // the species of the parent is set to the children in loadBatchRow
        if (parentRow == null && context.isProtocolFilled()) {
            // get the surveycode from the species list of the model
            List<Species> speciesList = getDataContext().getReferentSpeciesWithSurveyCode();
            int i = speciesList.indexOf(newRow.getSpecies());
            if (i > -1) {
                newRow.setSpecies(speciesList.get(i));
            }
        }

        SampleCategoryEnum sampleCategoryEnum = aBatch.getSampleCategoryType();

        Preconditions.checkNotNull(
                sampleCategoryEnum,
                "Can't have a batch with no sample category, but was: " + aBatch);

        loadBatchRow(parentRow,
                     newRow,
                     sampleCategoryEnum,
                     aBatch.getSampleCategoryValue(),
                     aBatch.getSampleCategoryWeight());

        rows.add(newRow);

        if (!aBatch.isChildBatchsEmpty()) {

            // create batch childs rows

            List<SpeciesBatchRowModel> batchChilds = Lists.
                    newArrayListWithCapacity(aBatch.sizeChildBatchs());

            for (SpeciesBatch childBatch : aBatch.getChildBatchs()) {
                SpeciesBatchRowModel childRow = loadBatch(childBatch, newRow, rows);
                batchChilds.add(childRow);
            }
            newRow.setChildBatch(batchChilds);
        }

        return newRow;
    }

    protected void loadBatchRow(SpeciesBatchRowModel parentRow,
                                SpeciesBatchRowModel newRow,
                                SampleCategoryEnum sampleCategoryEnum,
                                Serializable categoryValue,
                                Float categoryWeight) {

        // get sample category from his type
        SampleCategory sampleCategory =
                newRow.getSampleCategory(sampleCategoryEnum);

        // fill it
        sampleCategory.setCategoryValue(categoryValue);
        sampleCategory.setCategoryWeight(categoryWeight);

        // push it back to row as his *main* sample category
        newRow.setSampleCategory(sampleCategory);

        if (parentRow != null) {

            // copy back parent data (mainly other sample categories)

            newRow.setSpecies(parentRow.getSpecies());
            newRow.setSpeciesToConfirm(parentRow.getSpeciesToConfirm());
            newRow.setParentBatch(parentRow);

            newRow.setSpecies(parentRow.getSpecies());
            if (sampleCategoryEnum != SampleCategoryEnum.sortedUnsorted) {
                newRow.setSortedUnsortedCategory(parentRow.getSortedUnsortedCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.size) {
                newRow.setSizeCategory(parentRow.getSizeCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.sex) {
                newRow.setSexCategory(parentRow.getSexCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.maturity) {
                newRow.setMaturityCategory(parentRow.getMaturityCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.age) {
                newRow.setAgeCategory(parentRow.getAgeCategory());
            }
        }
    }

    protected <C extends Serializable> void addSampleCategoryColumnToModel(TableColumnModel columnModel,
                                                                           ColumnIdentifier<SpeciesBatchRowModel> columnIdentifier,
                                                                           Decorator<C> decorator,
                                                                           TableCellRenderer defaultRenderer) {
        addColumnToModel(
                columnModel,
                SampleCategoryComponent.newEditor(decorator),
                SampleCategoryComponent.newRender(defaultRenderer,
                                                  decorator,
                                                  getConfig().getColorComputedWeights()),
                columnIdentifier);
    }

    public void removeFromSpeciesUsed(SpeciesBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isInfoEnabled()) {
            log.info("Remove from speciesUsed: " + decorate(row.getSortedUnsortedCategoryValue()) + " - " + decorate(row.getSpecies()));
        }
        SpeciesBatchUIModel model = getModel();
        model.getSpeciesUsed().remove(row.getSortedUnsortedCategoryValue(),
                                      row.getSpecies());

        if (row.isBatchRoot()) {
            model.setRootNumber(model.getRootNumber() - 1);
        }
    }

    protected void addToSpeciesUsed(SpeciesBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isDebugEnabled()) {
            log.debug("Add to speciesUsed: " +
                      decorate(row.getSortedUnsortedCategoryValue()) +
                      " - " + decorate(row.getSpecies()));
        }
        SpeciesBatchUIModel model = getModel();
        model.getSpeciesUsed().put(row.getSortedUnsortedCategoryValue(),
                                   row.getSpecies());

        model.setRootNumber(model.getRootNumber() + 1);
    }

    public Species openAddSpeciesDialog(String title, List<Species> species) {
        SelectSpeciesUI dialogContent = new SelectSpeciesUI(ui);
        SelectSpeciesUIModel model = dialogContent.getModel();
        model.setSelectedSpecies(null);
        model.setSpecies(species);

        openDialog(dialogContent, title, new Dimension(400, 130));

        return model.getSelectedSpecies();
    }
}
