package fr.ifremer.tutti.ui.swing.util.attachment;

/*
 * #%L
 * Tutti :: UI
 * $Id: AttachmentEditorUIHandler.java 629 2013-03-15 09:48:31Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/attachment/AttachmentEditorUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JRootPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.IOException;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * @author kmorin <morin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class AttachmentEditorUIHandler {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AttachmentEditorUIHandler.class);

    public static final String CLOSE_DIALOG_ACTION = "closeDialog";

    public static final String SHOW_DIALOG_ACTION = "showDialog";

    protected final AttachmentEditorUI ui;

    protected final PersistenceService persistenceService;

    protected final TuttiUIContext context;

    public AttachmentEditorUIHandler(TuttiUIContext context,
                                     AttachmentEditorUI ui) {
        this.context = context;
        this.ui = ui;
        this.persistenceService = context.getPersistenceService();
    }

    protected void beforeInitUI() {
    }

    protected void afterInitUI() {

        ui.getFile().setDialogOwner(ui);
        ui.pack();
        ui.setResizable(true);
//        ui.setSize(800, 300);

        ComponentResizer cr = new ComponentResizer();
        cr.registerComponent(ui);
        ComponentMover cm = new ComponentMover();
        cm.setDragInsets(cr.getDragInsets());
        cm.registerComponent(ui);

        JRootPane rootPane = ui.getRootPane();

        KeyStroke shortcutClosePopup = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                shortcutClosePopup, CLOSE_DIALOG_ACTION);

        closeAction = new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                ui.dispose();
                ui.setVisible(false);
            }
        };

        openAction = new AbstractAction() {

            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                ui.setVisible(true);
            }
        };

        ImageIcon actionIcon = SwingUtil.createActionIcon("close-dialog");
        closeAction.putValue(Action.SMALL_ICON, actionIcon);
        closeAction.putValue(Action.LARGE_ICON_KEY, actionIcon);
        closeAction.putValue(Action.ACTION_COMMAND_KEY, "close");
        closeAction.putValue(Action.NAME, "close");
        closeAction.putValue(Action.SHORT_DESCRIPTION, _("tutti.attachmentEditor.action.closeAttachment.tip"));

        rootPane.getActionMap().put(CLOSE_DIALOG_ACTION, closeAction);
        rootPane.getActionMap().put(SHOW_DIALOG_ACTION, openAction);

        JButton closeButton = new JButton(closeAction);
        closeButton.setText(null);
        closeButton.setFocusPainted(false);
        closeButton.setRequestFocusEnabled(false);
        closeButton.setFocusable(false);

        JToolBar jToolBar = new JToolBar();
        jToolBar.setOpaque(false);
        jToolBar.add(closeAction);
        jToolBar.setBorderPainted(false);
        jToolBar.setFloatable(false);
        ui.getAttachmentBody().setRightDecoration(jToolBar);
    }

    protected Action closeAction;

    protected Action openAction;

    public void closeEditor() {

        closeAction.actionPerformed(null);
    }

    public void openEditor(JComponent component) {

        if (component != null) {
            place(component);
        }
        openAction.actionPerformed(null);
    }

    public void init() {
        resetFields();
        ui.getAttachments().removeAll();
        AttachmentModelAware bean = ui.getBean();
        if (bean != null) {
            List<Attachment> list = bean.getAttachment();
            if (list != null) {
                for (Attachment attachment : list) {
                    addAttachment(attachment);
                }
            }
        }
    }

    public void place(JComponent component) {
        // Computes the location of bottom left corner of the cell
        Component comp = component;
        int x = 0;
        int y = component.getHeight();
        while (comp != null) {
            x += comp.getX();
            y += comp.getY();
            comp = comp.getParent();
        }

        ui.pack();
        // if the editor is too big on the right,
        // then align its right side to the right side of the cell
        if (x + ui.getWidth() > ui.getOwner().getX() + ui.getOwner().getWidth()) {
            x = x - ui.getWidth() + component.getWidth();
        }
        ui.setLocation(x, y);
    }

    public void addAttachment() {
        AttachmentModelAware bean = ui.getBean();
        Attachment attachment = new Attachment();

        File file = ui.getFile().getSelectedFile();
        if (file != null) {
            String name = ui.getFileName().getText();
            if (StringUtils.isEmpty(name)) {
                name = file.getName();
            }
            attachment.setObjectType(bean.getObjectType());
            attachment.setObjectId(Integer.valueOf(bean.getObjectId()));
            attachment.setName(name);
            attachment.setComment(ui.getFileComment().getText());

            attachment = persistenceService.createAttachment(attachment, file);
            bean.addAttachment(attachment);

            resetFields();
            addAttachment(attachment);

            ui.pack();
        }
    }

    protected void addAttachment(final Attachment attachment) {
        AttachmentItem item = new AttachmentItem();
        item.setAttachment(attachment);
        item.setHandler(this);
        ui.getAttachments().add(item);
    }

    protected void resetFields() {
        ui.getFile().setSelectedFile((File) null);
        ui.getFile().setSelectedFile((String) null);
        ui.getFileName().setText("");
        ui.getFileComment().setText("");
    }

    public void openAttachment(Attachment attachment) {
        File file = persistenceService.getAttachmentFile(attachment.getId());
        SwingUtil.openLink("file://" + file.getAbsolutePath());
    }

    public void saveAttachment(Attachment attachment) {
        boolean hackDialog = ui.isAlwaysOnTop();
        if (hackDialog) {
            ui.setAlwaysOnTop(false);
        }
        try {
            File attachmentFile = persistenceService.getAttachmentFile(attachment.getId());
            File file = TuttiUIUtil.chooseFile(ui,
                                               _("tutti.attachmentEditor.saveAttachment.title"),
                                               _("tutti.attachmentEditor.saveAttachment.button"));

            if (file != null) {
                if (file.exists()) {
                    String htmlMessage = String.format(
                            AbstractTuttiUIHandler.CONFIRMATION_FORMAT,
                            _("tutti.attachmentEditor.saveAttachment.warning.message"),
                            _("tutti.attachmentEditor.saveAttachment.warning.help"));

                    int answer = JOptionPane.showConfirmDialog(ui,
                                                               htmlMessage,
                                                               _("tutti.attachmentEditor.saveAttachment.warning.title"),
                                                               JOptionPane.YES_NO_OPTION,
                                                               JOptionPane.WARNING_MESSAGE);

                    if (answer == JOptionPane.NO_OPTION) {
                        return;
                    }
                }
                FileUtils.copyFile(attachmentFile, file);
                TuttiUIUtil.showSuccessMessage(ui,
                                               _("tutti.attachmentEditor.saveAttachment.success.title"),
                                               _("tutti.attachmentEditor.saveAttachment.success.message", file.getName()));
            }
            if (hackDialog) {
                ui.setAlwaysOnTop(true);
            }

        } catch (IOException e) {
            if (log.isErrorEnabled()) {
                log.error("Error while saving attachment", e);
            }
            JOptionPane.showMessageDialog(
                    ui,
                    _("tutti.attachmentEditor.saveAttachment.fail.message"),
                    _("tutti.attachmentEditor.saveAttachment.fail.title"),
                    JOptionPane.ERROR_MESSAGE
            );

        } finally {
            if (hackDialog) {
                ui.setAlwaysOnTop(true);
            }
        }
    }

    public void removeAttachment(AttachmentItem attachmentItem) {
        Attachment attachment = attachmentItem.getAttachment();
        boolean hackDialog = ui.isAlwaysOnTop();
        if (hackDialog) {
            ui.setAlwaysOnTop(false);
        }
        int answer = JOptionPane.showConfirmDialog(ui,
                                                   _("tutti.attachmentEditor.deleteAttachment.message", attachment.getName()),
                                                   _("tutti.attachmentEditor.deleteAttachment.title"),
                                                   JOptionPane.YES_NO_OPTION);
        if (hackDialog) {
            ui.setAlwaysOnTop(true);
        }

        if (answer == JOptionPane.YES_OPTION) {
            AttachmentModelAware bean = ui.getBean();

            persistenceService.deleteAttachment(attachment.getId());
            bean.removeAttachment(attachment);
            ui.getAttachments().remove(attachmentItem);

            ui.pack();
        }
    }
}
