package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiAction.java 432 2013-02-15 20:28:55Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/AbstractTuttiAction.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.TuttiService;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUI;
import fr.ifremer.tutti.ui.swing.util.action.TuttiActionUIModel;
import jaxx.runtime.SwingUtil;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.util.decorator.Decorator;

import javax.swing.AbstractAction;
import java.awt.event.ActionEvent;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * TODO
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public abstract class AbstractTuttiAction<M extends AbstractBean, UI extends TuttiUI<M, ?>, H extends AbstractTuttiUIHandler<M, UI>> extends AbstractAction {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractTuttiAction.class);

    private static final Timer t = new Timer();

    private static final ExecutorService executorService =
            Executors.newSingleThreadExecutor();

    private final H handler;

    private boolean failed;

    private final boolean hideBody;

    private final static Object lock = new Object();

    protected abstract void doAction(ActionEvent event) throws Exception;

    protected AbstractTuttiAction(H handler,
                                  String name,
                                  String icon,
                                  String text,
                                  String tip,
                                  boolean hideBody) {

        this.handler = handler;
        this.hideBody = hideBody;
        putValue(SMALL_ICON, SwingUtil.createActionIcon(icon));
        putValue(LARGE_ICON_KEY, SwingUtil.createActionIcon(icon));
        putValue(ACTION_COMMAND_KEY, name);
        putValue(NAME, text);
        putValue(SHORT_DESCRIPTION, tip);
    }

    public final UI getUI() {
        return handler.getUI();
    }

    protected String getActionName() {
        return (String) getValue("actionName");
    }

    public String getActionDescription() {
        return (String) getValue("actionDescription");
    }

    protected boolean prepareAction(ActionEvent event) {
        putValue("actionName", getValue(NAME));
        putValue("actionDescription", getValue(SHORT_DESCRIPTION));
        return true;
    }

    protected void releaseAction(ActionEvent event) {
        putValue("actionName", null);
        putValue("actionDescription", null);
    }

    public boolean isFailed() {
        return failed;
    }

    public void setFailed(boolean failed) {
        this.failed = failed;
    }

    @Override
    public final void actionPerformed(final ActionEvent event) {

        if (log.isInfoEnabled()) {
            log.info("Task [" + this + "] starting");
        }

        setFailed(false);

        boolean doAction = prepareAction(event);

        if (doAction) {

            boolean useTimer;

            synchronized (lock) {
                useTimer = actions.isEmpty();

                mainAction = AbstractTuttiAction.this;
                actions.add(AbstractTuttiAction.this);
            }

            if (useTimer) {

                // a new thread action

                executorService.submit(new Runnable() {
                    @Override
                    public void run() {

                        runAction(true, event);
                    }
                });

            } else {

                runAction(false, event);
            }
        } else {
            releaseAction(event);
        }
    }

    protected void runAction(boolean useTimer, ActionEvent event) {

        TimerTask timer = null;
        if (useTimer) {

            // there is already anohter action in pool, no timer

            timer = new ActionTimerTask(this);

            t.schedule(timer, 1000);
        }

        try {

            this.doAction(event);
        } catch (Throwable e) {
            setFailed(true);

            if (log.isErrorEnabled()) {
                log.error("Task [" + this + "] error: " + e.getMessage(), e);
            }
            throw new TuttiExceptionHandler.TuttiActionException(this, e);
        } finally {
            if (log.isInfoEnabled()) {
                log.info("Task [" + this + "] done");
            }
            if (timer != null) {

                timer.cancel();
            }

            try {
                this.releaseAction(event);
            } finally {

                synchronized (lock) {

                    TuttiActionUI actionUI = AbstractTuttiAction.this.getContext().getActionUI();

                    TuttiActionUIModel actionUIModel = actionUI.getModel();

                    if (ObjectUtils.equals(AbstractTuttiAction.this, actionUIModel.getAction())) {

                        // same action, then remove it (will close dialog as a side effect)
                        actionUIModel.clear();
                    }
                    actions.remove(AbstractTuttiAction.this);

                    if (AbstractTuttiAction.this.equals(getMainAction())) {
                        mainAction = null;
                    }
                }
            }
        }
    }

    public TuttiUIContext getContext() {
        return handler.getContext();
    }

    public H getHandler() {
        return handler;
    }

    public M getModel() {
        return handler.getModel();
    }

    protected <S extends TuttiService> S getService(Class<S> serviceType) {
        return getContext().getService(serviceType);
    }

    protected TuttiApplicationConfig getConfig() {
        return getContext().getConfig();
    }

    protected void setMnemonic(int key) {
        putValue(MNEMONIC_KEY, key);
    }

    public boolean isHideBody() {
        return hideBody;
    }

    private final static Set<Object> actions = Sets.newHashSet();

    private static AbstractTuttiAction mainAction;

    public static AbstractTuttiAction getMainAction() {
        return mainAction;
    }

    protected static class ActionTimerTask<A extends AbstractTuttiAction> extends TimerTask {

        private final Object lock = new Object();

        protected boolean canceled;

        private final A action;

        protected TuttiActionUI actionUI;

        public ActionTimerTask(A action) {
            this.action = action;
            this.actionUI = action.getContext().getActionUI();
        }

        @Override
        public void run() {

            if (canceled) {

                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] was already canceled, do nothing");
                }
            } else {

                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] is started, show waiting dialog");
                }

                if (log.isDebugEnabled()) {
                    log.debug("Try to open dialog (was canceled ? " + canceled + ")");
                }
                if (!canceled || !action.isFailed()) {

                    updateBusyState(true);

                    actionUI.getModel().setAction(action);
                }
            }
        }


        @Override
        public boolean cancel() {
            synchronized (lock) {
                canceled = true;
                if (log.isDebugEnabled()) {
                    log.debug("Task [" + action + "] canceled.");
                }
                boolean cancel = super.cancel();

                updateBusyState(false);

                if (ObjectUtils.equals(action, actionUI.getModel().getAction())) {

                    // same action, then remove it (will close dialog as a side effect)
                    actionUI.getModel().clear();
                }
                return cancel;
            }
        }

        protected void updateBusyState(boolean busy) {

            MainUI ui = action.getContext().getMainUI();
            if (busy) {
                // ui bloquee
                if (action.isHideBody()) {
                    ui.getBody().setVisible(false);
                }
            } else {
                // ui debloquee
                if (action.isHideBody()) {
                    ui.getBody().setVisible(true);
                }
            }
        }
    }

    protected void sendMessage(String message) {
        getContext().showInformationMessage(message);
    }

    protected <O> Decorator<O> getDecorator(Class<O> type, String name) {
        DecoratorService decoratorService =
                getContext().getService(DecoratorService.class);

        Preconditions.checkNotNull(type);

        Decorator decorator = decoratorService.getDecoratorByType(type, name);
        if (decorator == null) {

            if (DecoratorService.LabelAware.class.isAssignableFrom(type)) {
                decorator = getDecorator(DecoratorService.LabelAware.class, null);
            }
        }
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected String decorate(Object object) {
        return handler.getDecorator(object.getClass(), null).toString(object);
    }
}
