package fr.ifremer.tutti.ui.swing.content.operation.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesFrequencyUIHandler.java 54 2012-12-12 22:50:50Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/species/SpeciesFrequencyUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.SwingUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.JDialog;
import javax.swing.table.TableColumnModel;
import java.awt.Color;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesFrequencyUIHandler extends AbstractTuttiTableUIHandler<SpeciesFrequencyRowModel, SpeciesFrequencyUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesFrequencyUIHandler.class);

    private final SpeciesFrequencyUI ui;

    public SpeciesFrequencyUIHandler(TuttiUIContext context,
                                     SpeciesFrequencyUI ui) {
        super(context,
              SpeciesFrequencyRowModel.PROPERTY_LENGTH_STEP,
              SpeciesFrequencyRowModel.PROPERTY_NUMBER,
              SpeciesFrequencyRowModel.PROPERTY_WEIGHT,
              SpeciesFrequencyRowModel.PROPERTY_COMPUTED_WEIGHT);
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected SpeciesFrequencyUIModel getModel() {
        return ui.getModel();
    }

    @Override
    protected SpeciesFrequencyTableModel getTableModel() {
        return (SpeciesFrequencyTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected TableColumnModel createTableColumnModel() {

        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        { // LengthStep

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.LENGTH_STEP,
                                  TuttiUI.DECIMAL1_PATTERN);
        }

        { // Number

            addIntegerColumnToModel(columnModel,
                                    SpeciesFrequencyTableModel.NUMBER,
                                    TuttiUI.INT_6_DIGITS_PATTERN);
        }

        { // Weight

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // ComputedWeight

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.COMPUTED_WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }
        return columnModel;
    }

    @Override
    protected void onRowModified(SpeciesFrequencyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        if (SpeciesFrequencyRowModel.PROPERTY_NUMBER.equals(propertyName)) {

            // Need to recompute the computedWeight
            computeComputedWeight(row);
        }
    }

    @Override
    protected void onRowValidStateChanged(SpeciesFrequencyRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
    }

    @Override
    protected void onRowModifyStateChanged(SpeciesFrequencyRowModel row,
                                           Boolean oldValue,
                                           Boolean newValue) {
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {

        SpeciesFrequencyUIModel model = new SpeciesFrequencyUIModel();

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        JXTable table = getTable();

        // create table column model
        TableColumnModel columnModel = createTableColumnModel();

        SpeciesFrequencyUIModel model = getModel();

        // create table model
        SpeciesFrequencyTableModel tableModel =
                new SpeciesFrequencyTableModel(columnModel, model);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        installTableKeyListener(columnModel, table);

        table.getTableHeader().setReorderingAllowed(false);

        table.addHighlighter(TuttiUIUtil.newBackgroundColorHighlighter(
                HighlightPredicate.READ_ONLY, Color.LIGHT_GRAY));

        // when model datas change let's propagate it table model
        listenRowsFromModel();

        //TODO Should it come from PROTOCOL or config ?
        model.setStep(.5f);

        model.setMinStep(10f);
        model.setMaxStep(20f);

        //TODO Configure this ?
        model.setConfigurationMode(SpeciesFrequencyUIModel.ConfigurationMode.SIMPLE);

        // always scroll to selected row
        SwingUtil.scrollToTableSelection(getTable());

        ui.getRafaleStepField().getTextField().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    e.consume();
                    Float step = (Float) ui.getRafaleStepField().getModel();

                    applyRafaleStep(step);
                }
            }
        });
    }

    @Override
    public void onCloseUI() {
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void generateLengthSteps() {

        SpeciesFrequencyUIModel model = getModel();

        Map<Float, SpeciesFrequencyRowModel> rowsByStep =
                getTableModel().getRowCache();

        Float minStep = model.getLengthStep(model.getMinStep());
        Float maxStep = model.getLengthStep(model.getMaxStep());

        for (float i = minStep, step = model.getStep(); i <= maxStep; i += step) {
            if (!rowsByStep.containsKey(i)) {

                // add it
                SpeciesFrequencyRowModel newRow = new SpeciesFrequencyRowModel();
                newRow.setLengthStep(i);
                rowsByStep.put(i, newRow);
            }
        }

        List<SpeciesFrequencyRowModel> rows =
                Lists.newArrayList(rowsByStep.values());

        model.setRows(rows);
    }

    public void applyRafaleStep(Float step) {

        if (log.isInfoEnabled()) {
            log.info("Will apply rafale step: " + step);
        }
        SpeciesFrequencyUIModel model = getModel();
        SpeciesFrequencyTableModel tableModel = getTableModel();

        Map<Float, SpeciesFrequencyRowModel> rowsByStep = tableModel.getRowCache();

        float aroundLengthStep = model.getLengthStep(step);

        SpeciesFrequencyRowModel row = rowsByStep.get(aroundLengthStep);

        int rowIndex;

        if (row != null) {

            // increments current row
            Integer number = row.getNumber();
            row.setNumber((number == null ? 0 : number) + 1);
            rowIndex = tableModel.updateRow(row);

        } else {

            // create a new row
            row = new SpeciesFrequencyRowModel();
            row.setLengthStep(aroundLengthStep);
            row.setNumber(1);

            // get new index
            List<Float> steps = Lists.newArrayList(rowsByStep.keySet());
            steps.add(aroundLengthStep);

            Collections.sort(steps);

            rowIndex = steps.indexOf(aroundLengthStep);

            tableModel.addNewRow(rowIndex, row);
        }

        getTable().scrollRowToVisible(rowIndex);
    }

    public void editBatch(SpeciesBatchRowModel row) {

        List<SpeciesFrequencyRowModel> frequency = null;
        if (row != null) {
            frequency = row.getFrequency();
        }

        List<SpeciesFrequencyRowModel> editFrequency;
        if (CollectionUtils.isEmpty(frequency)) {

            // new list
            editFrequency = Lists.newArrayList();
        } else {

            editFrequency = Lists.newArrayList(frequency);
        }

        if (log.isInfoEnabled()) {
            log.info("Will edit batch row: " + row + " with " +
                     editFrequency.size() + " frequency");
        }
        getModel().setRows(editFrequency);

        // keep batch (will be used to push back editing entry)
        getModel().setBatch(row);
    }

    public void close() {

        if (log.isInfoEnabled()) {
            log.info("Will close UI " + ui);
        }

        // transfer rows to editor

        List<SpeciesFrequencyRowModel> frequency = Lists.newArrayList();
        for (SpeciesFrequencyRowModel row : getModel().getRows()) {
            if (row.isValid()) {

                // can keep this row
                frequency.add(row);
            }
        }

        if (log.isInfoEnabled()) {
            log.info("Push back " + frequency.size() +
                     " frequency to batch " + getModel().getBatch());
        }

        // push back to batch
        getModel().getBatch().setFrequency(frequency);

        ui.editBatch(null);

        SwingUtil.getParentContainer(ui, JDialog.class).setVisible(false);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeComputedWeight(SpeciesFrequencyRowModel row) {

        if (log.isInfoEnabled()) {
            log.info("Will recompute computed weight for frequency: " + row);
        }
    }

}