package fr.ifremer.tutti.ui.swing.content.operation.accidental;

/*
 * #%L
 * Tutti :: UI
 * $Id: AccidentalBatchUIHandler.java 82 2012-12-17 09:18:52Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/accidental/AccidentalBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.AccidentalBatch;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.editor.AttachmentCellComponent;
import fr.ifremer.tutti.ui.swing.util.editor.LongTextCellComponent;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.table.TableColumnModel;
import java.util.List;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class AccidentalBatchUIHandler extends AbstractTuttiBatchTableUIHandler<AccidentalBatchRowModel, AccidentalBatchUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AccidentalBatchUIHandler.class);

    /**
     * UI.
     *
     * @since 0.2
     */
    private final AccidentalBatchUI ui;

    public AccidentalBatchUIHandler(FishingOperationsUI parentUi,
                                    AccidentalBatchUI ui) {
        super(parentUi, ArrayUtils.EMPTY_STRING_ARRAY,
              AccidentalBatchRowModel.PROPERTY_SPECIES,
              AccidentalBatchRowModel.PROPERTY_WEIGHT,
              AccidentalBatchRowModel.PROPERTY_COMMENT);
        this.ui = ui;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected AccidentalBatchUIModel getModel() {
        return ui.getModel();
    }

    @Override
    protected AccidentalBatchTableModel getTableModel() {
        return (AccidentalBatchTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected FishingOperation getFishingOperation() {
        return getModel().getFishingOperation();
    }

    @Override
    protected TableColumnModel createTableColumnModel() {

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        List<Species> allSpecies = persistenceService.getAllSpecies();

        {
            // Species (by code) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, DecoratorService.SPECIES_BY_CODE);

            addComboDataColumnToModel(columnModel,
                                      AccidentalBatchTableModel.SPECIES_BY_CODE,
                                      decorator, allSpecies);
        }

        {
            // Species (by genusCode) column

            Decorator<Species> decorator = getDecorator(
                    Species.class, DecoratorService.SPECIES_BY_GENUS);

            addComboDataColumnToModel(columnModel,
                                      AccidentalBatchTableModel.SPECIES_BY_GENUS_CODE,
                                      decorator, allSpecies);

        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  AccidentalBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             LongTextCellComponent.newEditor(ui.getLongTextEditor()),
                             LongTextCellComponent.newRender(n_("tutti.tooltip.comment.none")),
                             AccidentalBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellComponent.newEditor(ui.getAttachmentEditor()),
                             AttachmentCellComponent.newRender(
                                     getDecorator(Attachment.class, null),
                                     n_("tutti.tooltip.attachment.none")),
                             AccidentalBatchTableModel.ATTACHMENTS);
        }
        return columnModel;
    }

    @Override
    protected void onRowModified(AccidentalBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        boolean wasValid = row.isValid();

        boolean valid = isRowValid(row);

        if (log.isInfoEnabled()) {
            log.info("Was valid / Is valid: " + wasValid + " / " + valid);
        }
        row.setValid(valid);
    }

    @Override
    protected void onRowValidStateChanged(AccidentalBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        int rowIndex = getTableModel().getRowIndex(row);

        if (rowIndex > -1) {
            getTableModel().fireTableRowsUpdated(rowIndex, rowIndex);
        }
    }

    @Override
    protected void onRowModifyStateChanged(AccidentalBatchRowModel row,
                                           Boolean oldValue,
                                           Boolean newValue) {
    }

    @Override
    protected void onModelRowsChanged(List<AccidentalBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        // set valid flag
        for (AccidentalBatchRowModel row : rows) {
            boolean valid = isRowValid(row);
            row.setValid(valid);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired() {

        TuttiBeanMonitor<AccidentalBatchRowModel> rowMonitor = getRowMonitor();

        AccidentalBatchRowModel bean = rowMonitor.getBean();
        if (bean != null) {

            if (bean.isValid()) {
                // there is a valid bean attached to the monitor

                if (rowMonitor.wasModified()) {

                    // monitored bean was modified, save it
                    if (log.isInfoEnabled()) {
                        log.info("Row " + bean + " was modified, will save it");
                    }

                    saveRow(bean);

                    // clear modified flag on the monitor
                    rowMonitor.clearModified();
                }
            } else {

                // row is not valid can not save it

                AccidentalBatch catchBean = bean.toBean();

                if (!TuttiEntities.isNew(catchBean)) {

                    // remove this
                    persistenceService.deleteAccidentalBatch(catchBean.getId());
                }
            }
        }
    }

    @Override
    protected boolean isRowValid(AccidentalBatchRowModel row) {
        boolean result = row.getSpecies() != null && row.getWeight() != null;
        return result;
    }

    @Override
    protected void saveRow(AccidentalBatchRowModel row) {

        AccidentalBatch catchBean = row.toBean();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        catchBean.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(catchBean)) {

            catchBean = persistenceService.createAccidentalBatch(catchBean);
            row.setId(catchBean.getId());
        } else {
            persistenceService.saveAccidentalBatch(catchBean);
        }
    }

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        JXTable table = getTable();

        if (table.isEditing()) {

            // make sure to stop editor
            table.editingCanceled(null);
        }

        if (fishingOperationMonitor.wasModified()) {

            // previous fishingOperation was modified, let's save it
            AccidentalBatchUIModel beanToSave = fishingOperationMonitor.getBean();

            FishingOperation fishingOperation = beanToSave.toBean();

            if (log.isInfoEnabled()) {
                log.info("FishingOperation " + fishingOperation.getId() +
                         " was modified, will save it.");
            }

            persistenceService.saveFishingOperation(fishingOperation);
        }

        // make sure selection is empty (will remove bean from monitor)
        table.clearSelection();

        boolean empty = bean == null;

        AccidentalBatchUIModel model = getModel();

        List<AccidentalBatchRowModel> rows;

        String fishingoperationText;

        if (empty) {
            rows = null;
            bean = new FishingOperation();
            model.setFishingOperation(null);
            fishingoperationText = "";
        } else {

            Decorator<FishingOperation> decorator =
                    getDecorator(FishingOperation.class, null);

            fishingoperationText =
                    _("tutti.label.traitReminder", decorator.toString(bean));

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId() + " - " + fishingoperationText);
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<AccidentalBatch> catches =
                        persistenceService.getAllAccidentalBatch(bean.getId());
                for (AccidentalBatch aBatch : catches) {
                    AccidentalBatchRowModel entry =
                            new AccidentalBatchRowModel(aBatch);
                    rows.add(entry);
                }
            }
            model.setFishingOperation(bean);
        }

        model.fromBean(bean);
        model.setRows(rows);

        fishingOperationMonitor.clearModified();

        ui.getFishingOperationReminderLabel().setText(fishingoperationText);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        AccidentalBatchUIModel model = new AccidentalBatchUIModel();
        ui.setContextValue(model);

        fishingOperationMonitor.setBean(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // create table column model
        TableColumnModel columnModel = createTableColumnModel();

        // create table model
        AccidentalBatchTableModel tableModel =
                new AccidentalBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

}