package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: AbstractTuttiUIHandler.java 77 2012-12-15 16:05:45Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/AbstractTuttiUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.IdAware;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.config.TuttiConfig;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.editor.SimpleTimeEditor;
import jaxx.runtime.swing.editor.NumberEditor;
import jaxx.runtime.swing.editor.bean.BeanComboBox;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.renderer.DecoratorListCellRenderer;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXDatePicker;
import org.nuiton.util.decorator.Decorator;
import org.nuiton.util.decorator.JXPathDecorator;
import org.nuiton.validator.bean.simple.SimpleBeanValidator;

import javax.swing.JList;
import javax.swing.ListCellRenderer;
import javax.swing.ListModel;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.text.JTextComponent;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Contract of any UI handler.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public abstract class AbstractTuttiUIHandler<M> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AbstractTuttiUIHandler.class);

    public abstract void beforeInitUI();

    public abstract void afterInitUI();

    public abstract void onCloseUI();

    protected abstract M getModel();

    protected final TuttiUIContext context;

    protected AbstractTuttiUIHandler(TuttiUIContext context) {
        this.context = context;
    }

    public TuttiUIContext getContext() {
        return context;
    }

    public TuttiConfig getConfig() {
        return context.getConfig();
    }

    public void setText(KeyEvent event, String property) {
        JTextComponent field = (JTextComponent) event.getSource();
        String value = field.getText();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setBoolean(ItemEvent event, String property) {
        boolean value = event.getStateChange() == ItemEvent.SELECTED;
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void setDate(ActionEvent event, String property) {
        JXDatePicker field = (JXDatePicker) event.getSource();
        Date value = field.getDate();
        TuttiUIUtil.setProperty(getModel(), property, value);
    }

    public void selectListData(ListSelectionEvent event, String property) {
        if (!event.getValueIsAdjusting()) {
            JList list = (JList) event.getSource();
            ListSelectionModel selectionModel = list.getSelectionModel();

            selectionModel.setValueIsAdjusting(true);
            try {
                List selectedList = Lists.newLinkedList();

                for (int index : list.getSelectedIndices()) {
                    Object o = list.getModel().getElementAt(index);
                    selectedList.add(o);
                }
                TuttiUIUtil.setProperty(getModel(), property, selectedList);
            } finally {
                selectionModel.setValueIsAdjusting(false);
            }
        }
    }

    protected void initUI(TuttiUI ui) {

        for (Map.Entry<String, Object> entry : ui.get$objectMap().entrySet()) {
            Object component = entry.getValue();
            if (component instanceof NumberEditor) {
                initNumberEditor((NumberEditor) component);
            } else if (component instanceof JXDatePicker) {
                initDatePicker((JXDatePicker) component);
            } else if (component instanceof SimpleTimeEditor) {
                initTimeEditor((SimpleTimeEditor) component);
            }
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param comboBox le component graphique à initialiser
     */
    protected <E extends IdAware> void initBeanComboBox(
            BeanComboBox<E> comboBox,
            List<E> data,
            E selectedData) {

        Class<E> beanType = (Class<E>) comboBox.getHandler().getTargetClass();

        DecoratorService decoratorService =
                context.getService(DecoratorService.class);
        Decorator<E> decorator = decoratorService.getDecoratorByType(beanType);

        if (log.isInfoEnabled()) {
            log.info("entity list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        comboBox.init((JXPathDecorator<E>) decorator, data);

        comboBox.setSelectedItem(selectedData);

        if (log.isDebugEnabled()) {
            log.debug("combo [" + beanType.getName() + "] : " +
                      comboBox.getData().size());
        }
    }

    /**
     * Prépare un component de choix d'entités pour un type d'entité donné et
     * pour un service de persistance donné.
     *
     * @param list         le component graphique à initialiser
     * @param data
     * @param selectedData
     */
    protected <E extends IdAware> void initBeanList(
            BeanDoubleList<E> list,
            List<E> data,
            List<E> selectedData) {

        Class<E> beanType = list.getBeanType();

        DecoratorService decoratorService =
                context.getService(DecoratorService.class);
        Decorator<E> decorator = decoratorService.getDecoratorByType(beanType);

        if (log.isInfoEnabled()) {
            log.info("entity list [" + beanType.getName() + "] : " +
                     (data == null ? 0 : data.size()));
        }

        // add data list to combo box
        list.init((JXPathDecorator<E>) decorator, data, selectedData);

//        OneClicListSelectionModel.installModel(listHeader.getList());

//        updateIndices(listHeader.getList(), selectedData);

        if (log.isDebugEnabled()) {
            log.debug("Jlist [" + beanType.getName() + "] : " +
                      list.getUniverseList().getModel().getSize());
        }
    }

    protected void initNumberEditor(NumberEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init number editor " + editor.getName());
        }
        editor.init();

        // Force binding if value is already in model
        Number model = editor.getModel();
        if (model != null) {
            editor.setModel(null);
            editor.setModel(model);
        }
    }

    protected void initTimeEditor(SimpleTimeEditor editor) {
        if (log.isDebugEnabled()) {
            log.debug("init time editor " + editor.getName() +
                      " for property " + editor.getModel().getProperty());
        }
        editor.init();
    }

    protected void initDatePicker(JXDatePicker picker) {

        if (log.isDebugEnabled()) {
            log.debug("disable JXDatePicker editor" + picker.getName());
        }
        picker.getEditor().setEditable(false);
    }

    protected <B extends IdAware> void updateIndices(JList jlist,
                                                     List<B> list) {

        List<String> ids = TuttiEntities.toIds(list);

        List<Integer> indices = new ArrayList<Integer>();

        if (list != null && !list.isEmpty()) {

            ListModel model = jlist.getModel();

            for (int i = 0, max = model.getSize(); i < max; i++) {
                B s = (B) model.getElementAt(i);
                if (ids.contains(s.getId())) {
                    indices.add(i);
                }
            }
        }
        int[] result = new int[indices.size()];
        int i = 0;
        for (Integer indice : indices) {
            result[i++] = indice;
        }
        indices.clear();
        if (log.isDebugEnabled()) {
            log.debug("Selected indices : " + Arrays.toString(result));
        }
        jlist.setSelectedIndices(result);
    }

    protected <O> Decorator<O> getDecorator(Class<O> type, String name) {
        DecoratorService decoratorService =
                context.getService(DecoratorService.class);

        Preconditions.checkNotNull(type);

        Decorator<O> decorator = decoratorService.getDecoratorByType(type, name);
        Preconditions.checkNotNull(decorator);
        return decorator;
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type) {

        return newListCellRender(type, null);
    }

    protected <O> ListCellRenderer newListCellRender(Class<O> type, String name) {

        Decorator<O> decorator = getDecorator(type, name);
        return newListCellRender(decorator);
    }

    protected <O> ListCellRenderer newListCellRender(Decorator<O> decorator) {

        Preconditions.checkNotNull(decorator);

        ListCellRenderer result = new DecoratorListCellRenderer(decorator);
        return result;
    }

    protected void listenValidatorValid(SimpleBeanValidator validator,
                                        final AbstractTuttiBeanUIModel model) {
        validator.addPropertyChangeListener(SimpleBeanValidator.VALID_PROPERTY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                model.setValid((Boolean) evt.getNewValue());
            }
        });
    }


    protected void listModelIsModify(AbstractTuttiBeanUIModel model) {
        model.addPropertyChangeListener(new PropertyChangeListener() {

            final Set<String> excludeProperties = Sets.newHashSet(
                    AbstractTuttiBeanUIModel.PROPERTY_MODIFY,
                    AbstractTuttiBeanUIModel.PROPERTY_VALID);

            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                if (!excludeProperties.contains(evt.getPropertyName())) {
                    ((AbstractTuttiBeanUIModel) evt.getSource()).setModify(true);
                }
            }
        });
    }
}
