package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIContext.java 15 2012-11-24 13:10:24Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.1/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/TuttiUIContext.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.CampaignBean;
import fr.ifremer.tutti.persistence.entities.SurveyBean;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.TuttiService;
import fr.ifremer.tutti.service.TuttiServiceContext;
import fr.ifremer.tutti.ui.swing.config.TuttiConfig;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.beans.AbstractBean;
import org.nuiton.widget.SwingSession;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Closeable;
import java.util.Set;

/**
 * UI application context.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class TuttiUIContext extends AbstractBean implements Closeable {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIContext.class);

    public static final String PROPERTY_SURVEY_ID = "surveyId";

    public static final String PROPERTY_CAMPAIGN_ID = "campaignId";

    public static final String PROPERTY_SCREEN = "screen";

    public static final String PROPERTY_CAMPAIGN_CONTEXT_FILLED = "campaignContextFilled";

    /**
     * Application context (only one for all the application).
     *
     * @since 0.1
     */
    private static TuttiUIContext applicationContext;

    /**
     * Application global configuration.
     *
     * @since 0.1
     */
    protected final TuttiConfig config;

    /**
     * Service context used by any service.
     *
     * @since 0.1
     */
    protected final TuttiServiceContext serviceContext;

    /**
     * Swing session used to save ui states.
     *
     * @since 0.1
     */
    protected final SwingSession swingSession;

    /**
     * Id of last selected survey (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String surveyId;

    /**
     * Id of last selected campaign (can be null if none ever selected).
     *
     * @since 0.1
     */
    protected String campaignId;

    /**
     * Current screen displayed in ui.
     *
     * @since 0.1
     */
    protected TuttiScreen screen;

    public static TuttiUIContext newContext(TuttiConfig config) {
        Preconditions.checkNotNull(config);
        Preconditions.checkState(applicationContext == null,
                                 "Application context was already opened!");
        applicationContext = new TuttiUIContext(config);
        return applicationContext;
    }

    protected TuttiUIContext(TuttiConfig config) {
        this.config = config;
        this.serviceContext = new TuttiServiceContext(config.getServiceConfig());
        this.swingSession = new SwingSession(getConfig().getUIConfigFile(), false);
    }

    public <S extends TuttiService> S getService(Class<S> serviceType) {
        return serviceContext.getService(serviceType);
    }

    public TuttiConfig getConfig() {
        return config;
    }

    public SwingSession getSwingSession() {
        return swingSession;
    }

    public String getSurveyId() {
        return surveyId;
    }

    public String getCampaignId() {
        return campaignId;
    }

    public boolean isCampaignContextFilled() {
        return StringUtils.isNotBlank(surveyId) &&
               StringUtils.isNotBlank(campaignId);
    }

    public TuttiScreen getScreen() {
        return screen;
    }

    public void setSurveyId(String surveyId) {
        boolean oldValue = isCampaignContextFilled();

        this.surveyId = surveyId;

        // always propagate the change
        firePropertyChange(PROPERTY_SURVEY_ID, null, surveyId);
        firePropertyChange(PROPERTY_CAMPAIGN_CONTEXT_FILLED,
                           oldValue, isCampaignContextFilled());
    }

    public void setCampaignId(String campaignId) {
        boolean oldValue = isCampaignContextFilled();

        this.campaignId = campaignId;

        // always propagate the change
        firePropertyChange(PROPERTY_CAMPAIGN_ID, null, campaignId);
        firePropertyChange(PROPERTY_CAMPAIGN_CONTEXT_FILLED,
                           oldValue, isCampaignContextFilled());
    }

    public void setScreen(TuttiScreen screen) {
        Object oldValue = getScreen();
        this.screen = screen;
        firePropertyChange(PROPERTY_SCREEN, oldValue, screen);
    }

    public void open() {

        serviceContext.open();

        if (surveyId == null) {

            // load it from config
            setSurveyId(getConfig().getSurveyId());
        }

        if (campaignId == null) {

            // load it from config
            setCampaignId(getConfig().getCampaignId());
        }

        //check if surveyId is sane
        PersistenceService persistenceService =
                getService(PersistenceService.class);

        if (surveyId != null) {

            SurveyBean survey = persistenceService.getSurvey(surveyId);
            if (survey == null) {

                // not found in this db
                setSurveyId(null);
                setCampaignId(null);

                if (log.isWarnEnabled()) {
                    log.warn("Remove invalid surveyId: " + surveyId);
                }
            } else {

                if (log.isInfoEnabled()) {
                    log.info("SurveyId valid: " + surveyId);
                }

                setSurveyId(surveyId);

                // test campaignId
                if (campaignId != null) {

                    CampaignBean campaign =
                            persistenceService.getCampaign(campaignId);

                    if (campaign != null &&
                        !campaign.getSurvey().getId().equals(surveyId)) {

                        // not matchin survey, reset campaign id
                        campaign = null;
                    }

                    if (campaign == null) {

                        // not found in this db
                        setCampaignId(null);

                        if (log.isWarnEnabled()) {
                            log.warn("Remove invalid campaignId: " + campaignId);
                        }
                    } else {

                        if (log.isInfoEnabled()) {
                            log.info("CampaignId valid: " + campaignId);
                        }
                    }
                }
            }
        }

        // save back to config
        saveContextToConfig();

        // list when surveyId or campaingId change to save the configuration
        addPropertyChangeListener(new PropertyChangeListener() {

            Set<String> acceptedProperties = Sets.newHashSet(
                    PROPERTY_SURVEY_ID, PROPERTY_CAMPAIGN_ID);

            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                if (acceptedProperties.contains(evt.getPropertyName())) {
                    saveContextToConfig();
                }
            }
        });
    }

    @Override
    public void close() {

        // Clear data references
        surveyId = null;
        campaignId = null;

        IOUtils.closeQuietly(serviceContext);

        // remove listeners
        PropertyChangeListener[] listeners = getPropertyChangeListeners();
        for (PropertyChangeListener listener : listeners) {
            if (log.isDebugEnabled()) {
                log.debug("Remove listener: " + listener);
            }
            removePropertyChangeListener(listener);
        }
    }

    protected void saveContextToConfig() {
        if (log.isInfoEnabled()) {
            log.info("Save config (surveyId: " + surveyId + ", campaignId: " +
                     campaignId + ")");
        }
        config.setSurveyId(surveyId);
        config.setCampaignId(campaignId);
        config.save();
    }
}
