/*
 * #%L
 * IsisFish
 * 
 * $Id: CompileHelper.java 3671 2012-04-03 13:31:36Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2012 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.util;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;
import javax.tools.StandardLocation;
import javax.tools.ToolProvider;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import com.sun.tools.javac.api.JavacTool;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.datastore.CodeSourceStorage.Location;
import fr.ifremer.isisfish.datastore.JavaSourceStorage;

/**
 * Compile helper used to compile Java code.
 * 
 * JDK must be installed to use compilation.
 * (JRE won't work).
 * 
 * Created: 12 janv. 2006 15:29:53
 *
 * @author poussin
 * @version $Revision: 3671 $
 *
 * Last update: $Date: 2012-04-03 15:31:36 +0200 (Tue, 03 Apr 2012) $
 * by : $Author: echatellier $
 */
public class CompileHelper {

    /** Logger for this class. */
    private static final Log log = LogFactory.getLog(CompileHelper.class);

    /** CP detector proxy. */
    //protected static CodepageDetectorProxy detector;
    
    /**
     * Recherche tous les fichiers qui un source plus recent que la version compilé.
     * 
     * @param srcDir
     * @param destDir
     * @return File list
     */
    public static List<File> searchSrcToCompile(File srcDir, File destDir) {
        List<File> result = new ArrayList<File>();
        for (File src : srcDir.listFiles()) {
            File dest = new File(FileUtil.basename(src, ".java"), ".class");
            if (src.getName().endsWith(".java") && FileUtil.isNewer(src, dest)) {
                result.add(src);
            }
        }
        return result;
    }

    /**
     * Load la class demandé.
     * 
     * @param fqn le nom complet de la classe a charger
     * @return la classe souhaité ou null si la class n'est pas trouvée
     */
    public static Class<?> loadClass(String fqn) {
        Class<?> result = null;
        try {
            ClassLoader cl = IsisFish.config.getScriptClassLoader();
            result = cl.loadClass(fqn);
        } catch (ClassNotFoundException eee) {
            log.info(_("isisfish.error.load.class", fqn), eee);
        }
        return result;
    }

    public static Object newInstance(String fqn) {
        Object result = null;
        try {
            Class<?> clazz = loadClass(fqn);

            if (clazz != null) {
                result = clazz.newInstance();
            }
        } catch (Exception eee) {
            log.warn(_("isisfish.error.instanciate", fqn), eee);
        }
        return result;
    }

    /**
     * Compile le fichier source en .class si le source est plus recent que
     * le .class
     * @param source le JavaSourceStorage a compiler
     * @param destDir le repertoire destination de la compilation
     * @param force si vrai alors meme si le fichier destination est plus
     * recent la compilation aura lieu
     * @param out le flux sur lequel le resultat de la compilation doit
     * apparaitre. Peut-etre null, dans ce cas les sorties standards sont
     * utilisées.
     * @return 0 si la compilation a reussi une autre valeur sinon
     */
    public static int compile(JavaSourceStorage source, File destDir,
            boolean force, PrintWriter out) {
        File src = source.getFile();
        File dst = new File(destDir, source.getFQN().replace('.',
                File.separatorChar) + ".class");
        if (force || FileUtil.isNewer(src, dst)) {
            return compile(source.getRoot(), src, destDir, out);
        }
        return 0;
    }

    /**
     * Methode permettant de compiler un fichier Java.
     * 
     * @param rootSrc le répertoire ou se trouve les sources 
     * @param src Le fichier source a compiler, il doit etre dans un sous
     * répertoire de rootSrc en fonction du package
     * @param dest le repertoire destination de la compilation
     * @param out l'objet sur lequel on ecrit la sortie (erreur) de la 
     * compilation
     * @return un nombre different de 0 s'il y a une erreur
     * <li> -1000 si l'exception vient de la recherche du compilateur par 
     * introspection
     * <li> -10000 si une autre exception
     * <li> sinon les valeurs retourné par le compilateur java
     */
    public static int compile(File rootSrc, File src, File dest, PrintWriter out) {
        int result = compile(rootSrc, Collections.singletonList(src), dest, out);
        return result;
    }

    /**
     * Methode permettant de compiler un ensemble de fichiers Java.
     * 
     * @param rootSrc le répertoire ou se trouve les sources 
     * @param src Le fichier source a compiler, il doit etre dans un sous
     * répertoire de rootSrc en fonction du package
     * @param dest le repertoire destination de la compilation
     * @param out l'objet sur lequel on ecrit la sortie (erreur) de la 
     * compilation
     * @return un nombre different de 0 s'il y a une erreur
     * <li> -1000 si l'exception vient de la recherche du compilateur par 
     * introspection
     * <li> -10000 si une autre exception
     * <li> sinon les valeurs retourné par le compilateur java
     */
    public static int compile(File rootSrc, Collection<File> src, File dest,
            PrintWriter out) {
        int result = -10000;
        try {
            List<File> classpath = new ArrayList<File>();

            classpath.add(rootSrc.getAbsoluteFile());

            // works better than
            // fileManager.setLocation(StandardLocation.SOURCE_PATH, Location.ALL);
            // for some test
            for (File dir : Location.ALL.getDirectories()) {
                classpath.add(dir);
            }

            result = compile(classpath, src, dest, out);
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Compilation failed", eee);
            }
        }
        return result;
    }

    /**
     * Compile un fichier java.
     * 
     * @param src les fichiers java source
     * @param dest le repertoire destination
     */
    protected static int compile(List<File> classpath, Collection<File> src,
            File dest, PrintWriter out) {
        dest.mkdirs();

        int result = -1000;
        try {

            // look for best available Java compiler (if none, use provided)
            JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
            if (compiler == null) {
                compiler = JavacTool.create();
            }

            // Use system compiler
            // JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
            StandardJavaFileManager fileManager = compiler
                    .getStandardFileManager(null, null, null);
            Iterable<? extends JavaFileObject> compilationUnits = fileManager
                    .getJavaFileObjectsFromFiles(src);

            // Options de compilations
            String classpathAsString = getClassPathAsString(classpath);
            List<String> args = new ArrayList<String>();
            args.add("-g");
            // Show a description of each use or override of a deprecated member or class.
            args.add("-deprecation");
            args.add("-classpath");
            args.add(classpathAsString);
            args.add("-d");
            args.add(dest.getAbsolutePath());

            // Compilation
            boolean b = compiler.getTask(out, fileManager, null, args, null,
                    compilationUnits).call();
            // on retourne 0 si tout s'est bien déroulé et -1 sinon
            result = b ? 0 : -1;

            fileManager.close();
        } catch (Exception eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't get compiler", eee);
            }
        }
        return result;
    }

    /**
     * Return full classpath (for compilation or javadoc) as string.
     * Separated by {@link File#pathSeparator}.
     * 
     * Add :
     * <ul>
     *  <li>System.getProperty("java.class.path")
     *  <li>All first jar dependency (META-INF/MANIFEST.MF)
     * </ul>
     * @param classpath initial classpath
     * @return classpath as string
     * @throws Exception
     */
    public static String getClassPathAsString(List<File> classpath)
            throws Exception {
        String result = StringUtils.join(classpath.iterator(),
                File.pathSeparator);

        // chatellier : since 20090512 java.class.path in not added to
        // classpath.
        // result in duplicated entry or non normalised entry
        // en compilation fail some times
        //+ File.pathSeparator + System.getProperty("java.class.path");
        
        // Ajout des jars
        for (Enumeration<?> e = CompileHelper.class.getClassLoader()
                .getResources("META-INF/MANIFEST.MF"); e.hasMoreElements();) {
            URL url = (URL) e.nextElement();
            if (log.isDebugEnabled()) {
                log.debug("Found manifest : " + url);
            }
            if ((url != null) && url.getFile().startsWith("file:/")) {
                String jarName = url.getPath().substring(5,
                        url.getPath().indexOf("!"));
                if (!result.contains(jarName)) {
                    result += File.pathSeparator + jarName;
                }
            }
        }

        // chatellier : mais sous eclipse par exemple, on a besoin du classpath
        // on test que le class path n'est pas un jar et qu'il n'apparait
        // pas deja :
        String systemClassPath = System.getProperty("java.class.path");
        String[] systemClassPathes = systemClassPath.split(File.pathSeparator);
        for (String path : systemClassPathes) {
            String absolutePath = new File(path).getCanonicalPath();
            if (!result.contains(absolutePath)) {
                result += File.pathSeparator + absolutePath;
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("CLASSPATH : " + result);
        }

        return result;
    }

    /**
     * Extract documentation from interface (Equation).
     * 
     * @param category
     * @param name
     * @param javaInterface
     * @return doc
     */
    public static String extractDoc(String category, String name,
            Class<?> javaInterface) {
        String content = "<p>Equation : "
                + createHREF(javaInterface.getName(), category) + " - " + name
                + "</p>";
        content += "<h3>" + _("isisfish.editor.parametersnameandtypes") + " :</h3>";

        Method[] methods = javaInterface.getDeclaredMethods();
        Method interfaceMethod = methods[0];

        Args args = interfaceMethod.getAnnotation(Args.class);
        String[] names = args.value();
        String[] stringTypes = null;
        ArgTypes argTypes = interfaceMethod.getAnnotation(ArgTypes.class);
        if (argTypes != null) {
            stringTypes = argTypes.value();
        } else {
            stringTypes = new String[names.length];
            Class<?>[] types = interfaceMethod.getParameterTypes();
            System.out.println(Arrays.toString(types));
            for (int i = 0; i < types.length; i++) {
                stringTypes[i] = types[i].getName();
            }
        }

        for (int i = 0; i < names.length; i++) {
            content += "<li>" + names[i] + " : " + createHREF(stringTypes[i]);
        }

        return content;
    }

    /**
     * Create a html link to isis javadoc.
     * 
     * @param type class type
     * @param texts link display text
     * @return
     */
    protected static String createHREF(String type, String... texts) {
        
        // si pas de text, on affiche le type
        String text = type;
        if (texts.length > 0) {
            text = texts[0];
        }

        String result = null;
        if (type.startsWith("fr.ifremer.isisfish")) {
            String ref = IsisFish.config.getJavadocURL() + type.replaceAll("\\.", "/") + ".html";
            result = "<a href='" + ref + "'>" + text + "</a>";
        } else {
            // n'affiche pas de lien pour les types
            // genre "double" qui ne sont pas dans la javadoc d'isisfish
            result = text;
        }
        return result;
    }

    /*protected static CodepageDetectorProxy getCodepageDetector() {
        
        if (detector == null) {
            detector = CodepageDetectorProxy.getInstance(); // A singleton.

            // Add the implementations of info.monitorenter.cpdetector.io.ICodepageDetector: 
            // This one is quick if we deal with unicode codepages: 
            detector.add(new ByteOrderMarkDetector()); 
            // The first instance delegated to tries to detect the meta charset attribut in html pages.
            detector.add(new ParsingDetector(true)); // be verbose about parsing.
            // This one does the tricks of exclusion and frequency detection, if first implementation is 
            // unsuccessful:
            detector.add(JChardetFacade.getInstance()); // Another singleton.
            detector.add(ASCIIDetector.getInstance()); // Fallback, see javadoc.
        }
        return detector;
    }*/
    
    /*
     * Convert all files to UTF-8.
     * 
     * @param files fiels to convert
     * @return converted file list
     *
    public static List<File> convertToUnicode(List<File> files) {

        CodepageDetectorProxy myDetector = getCodepageDetector();
        
        for (File file : files) {
            try {
                Charset charset = myDetector.detectCodepage(file.toURI().toURL());
                
                if (log.isDebugEnabled()) {
                    log.debug("Charset for " + file.getAbsolutePath() + " is " + charset);
                }
                
                if (charset != null && !charset.name().equalsIgnoreCase("utf-8")) {
                    
                    if (log.isDebugEnabled()) {
                        log.debug("Convert " + file.getAbsolutePath() + " to unicode");
                    }
                    
                    File tmpFile = File.createTempFile(file.getName(), ".copy");
                    tmpFile.deleteOnExit();
                    
                    // direct copy
                    InputStream is = new FileInputStream(file);
                    OutputStream os = new FileOutputStream(tmpFile);
                    try {
                        IOUtils.copy(is, os);
                    }
                    finally {
                        is.close();
                        os.close();
                    }
                    
                    // copy using cp transaltion
                    is = new FileInputStream(tmpFile);
                    os = new FileOutputStream(file);
                    Reader ir = new InputStreamReader(is, charset);
                    Writer ow = new OutputStreamWriter(new FileOutputStream(file), "utf-8");
                    try {
                        IOUtils.copy(ir, ow);
                    }
                    finally {
                        ir.close();
                        ow.close();
                        is.close();
                        os.close();
                    }

                }
                else {
                    if (log.isDebugEnabled()) {
                        log.debug("File " + file.getAbsolutePath() + " already in unicode : skip");
                    }
                }
            } catch (MalformedURLException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't convert file in unicode", e);
                }
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't convert file in unicode", e);
                }
            }
            
        }

        return files;
    }*/
}
