/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationParameterImpl.java 3627 2012-03-02 13:08:28Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator;

import static org.nuiton.i18n.I18n._;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixHelper;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.SortedProperties;
import org.nuiton.util.StringUtil;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.SensitivityAnalysisStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SimulationPlanStorage;
import fr.ifremer.isisfish.datastore.StorageHelper;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.PopulationDAO;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.entities.StrategyDAO;
import fr.ifremer.isisfish.export.ExportHelper;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.rule.RuleHelper;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;

/**
 * Real {@link SimulationParameter} implementation.
 * 
 * @author chatellier
 * @version $Revision: 3627 $
 * 
 * Last update : $Date: 2012-03-02 14:08:28 +0100 (Fri, 02 Mar 2012) $
 * By : $Author: echatellier $
 */
public class SimulationParameterImpl implements SimulationParameter {

    /** Logger for this class. */
    private static Log log = LogFactory.getLog(SimulationParameterImpl.class);

    /** Remember last read properties. */
    protected Properties propertiesParameters;

    /**
     * Isis-fish version that permit to do the simulation.
     * Must be set just before simulation by simulator.
     */
    protected String isisFishVersion;

    /** Description de la simulation. */
    protected String description;

    /** Nom de la region sur lequel on simule. */
    protected String regionName;

    /** Nombre d'année sur lequel on souhaite simuler. */
    protected Integer numberOfYear;

    /** Le nom de l'objet script a utiliser pour faire la simulation. */
    protected String simulatorName;

    /** Utilisation ou non des optimisations. */
    protected Boolean useOptimization;

    /** Utilisation ou non des statistiques. */
    protected Boolean useStatistic;

    /** La liste des strategies a utiliser pour la simulation. */
    protected List<Strategy> strategies;

    /** La liste des populations a utiliser pour la simulation. */
    protected List<Population> populations;

    /** Les effectifs initiaux des différentes population. */
    protected Map<Population, MatrixND> numbers;

    /** La liste des regles de gestions a utiliser pour la simulation. */
    protected List<Rule> rules;

    /** La liste des regles potentiellement ajoutée par les plans de simulation. */
    protected List<String> extraRules;

    /** La liste des plans a utiliser pour la simulation. */
    protected List<SimulationPlan> simulationPlans;

    /** Ma liste des exports automatique a faire en fin de simulation. */
    protected List<String> exportNames;

    /** Utilisation du script de pre simulation. */
    protected Boolean usePreScript;

    /** Le script de pre simulation a utiliser. */
    protected String preScript;

    /** Utilisation du plan de simulation. */
    protected Boolean useSimulationPlan;

    /** 
     * Le numero de sequence de la liste des plans de simulation. Le premier
     * élement du plan doit etre 0. Si la simulation ne fait pas partie d'un
     * plan la valeur est -1. */
    protected Integer simulationPlanNumber;

    /** Export utilisés pour les analyses de sensibilités. */
    protected List<SensitivityExport> sensitivityExports;

    /**
     * Nombre de simulation constituant l'analyse de sensibilité.
     * 
     * FIXME ce parametre ne devrait pas ce trouver à cet endroit.
     * Ce n'est pas un parametres, mais une infos de simulation.
     * Cela permet de savoir combien de simulation il y a au total
     * pour savoir quand elle sont terminées.
     */
    protected Integer numberOfSensitivitySimulation;

    /** Script utilisé pour les analyses de sensibilités. */
    protected SensitivityAnalysis sensitivityAnalysis;

    /** La liste des resultats qui nous interesse. */
    protected Collection<String> resultEnabled;

    /** Autre parametre defini par l'utilisateur. */
    protected Map<String, String> tagValue;

    /**
     * Si vrai seul les exports automatiques ne sont souhaiter, la simulation
     * pourrait etre supprimée a la fin.
     */
    protected Boolean onlyExport;

    /** Le niveau de log du simulateur à utiliser pendant la simulation. Par defaut info. */
    protected String simulLogLevel;
    /** Le niveau de log des scripts à utiliser pendant la simulation. Par defaut info. */
    protected String scriptLogLevel;
    /** Le niveau de log des librairies à utiliser pendant la simulation. Par defaut error. */
    protected String libLogLevel;

    /** La region corespondant au parametre {@link #regionName}. */
    protected transient RegionStorage region;

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#check()
     */
    @Override
    public List<String> check() {
        List<String> result = new ArrayList<String>();
        // TODO faire la verif et pour chaque erreur mettre un message
        // dans la liste result (par exemple si pas d'effectif
        // pour une pop
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getIsisFishVersion()
     */
    @Override
    public String getIsisFishVersion() {
        // if not already set
        if (isisFishVersion == null) {
            if (propertiesParameters != null) {
                isisFishVersion = propertiesParameters.getProperty("isisFishVersion", "");
            } else {
                // default value
                isisFishVersion = IsisConfig.getVersion();
            }
        }
        return isisFishVersion;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setIsisFishVersion(java.lang.String)
     */
    @Override
    public void setIsisFishVersion(String isisFishVersion) {
        this.isisFishVersion = isisFishVersion;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getDescription()
     */
    @Override
    public String getDescription() {

        if (description == null) {
            if (propertiesParameters != null) {
                description = propertiesParameters.getProperty("description", "");
            } else {
                // default value
                description = "";
            }
        }
        return this.description;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setDescription(java.lang.String)
     */
    @Override
    public void setDescription(String description) {
        this.description = description;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRegion()
     */
    @Override
    public RegionStorage getRegion() {

        if (region == null || !region.getName().equals(getRegionName())) {
            // bien faire attention ici que la bonne region est ouverte
            // - isis-database-3 dans le cas d'une region normale
            // - simulation dans le cas d'une simulation
            region = RegionStorage.getRegion(getRegionName());
        } else {
            if (log.isDebugEnabled()) {
                log.debug(_("Region %s already inited", getRegionName()));
            }
        }
        return this.region;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOf(fr.ifremer.isisfish.entities.Population)
     */
    @Override
    public MatrixND getNumberOf(Population pop) {
        if (numbers == null) {
            numbers = new LinkedHashMap<Population, MatrixND>();
        }
        MatrixND result = numbers.get(pop);
        if (result == null) {
            List groups = pop.getPopulationGroup();
            List zones = pop.getPopulationZone();

            if (groups.size() == 0) {
                groups.add("No group");
            }
            if (zones.size() == 0) {
                zones.add("No zone");
            }
            List[] semantics = new List[] { groups, zones };
            result = MatrixFactory.getInstance().create(semantics);

            numbers.put(pop, result);
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getExtraRules()
     */
    @Override
    public List<String> getExtraRules() {

        if (extraRules == null) {
            // no properties in config ?
            extraRules = new ArrayList<String>();
        }
        return extraRules;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addExtraRules(java.lang.String)
     */
    @Override
    public void addExtraRules(String... extraRules) {
        getExtraRules().addAll(Arrays.asList(extraRules));
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getPopulations()
     */
    @Override
    public List<Population> getPopulations() {
        if (populations == null) {
            populations = new ArrayList<Population>();

            if (propertiesParameters != null) {

                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    PopulationDAO populationDAO = IsisFishDAOHelper
                            .getPopulationDAO(tx);
                    String[] populationList = propertiesParameters.getProperty(
                            "populations", "").split(",");
                    for (String name : populationList) {
                        if (name != null && !"".equals(name)) {
                            try {
                                Population population = populationDAO.findByName(name);
                                populations.add(population);

                                String numberAsString = propertiesParameters.getProperty("population." + name + ".number");
                                List number = MatrixHelper.convertStringToList(numberAsString);
                                MatrixND mat = getNumberOf(population);
                                mat.fromList(number);
                            } catch (TopiaException eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find population: " + name, eee);
                                }
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get PopulationDAO", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip population reloading");
                }
            }
        }
        return populations;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setPopulations(java.util.List)
     */
    @Override
    public void setPopulations(List<Population> populations) {
        this.populations = populations;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getStrategies()
     */
    @Override
    public List<Strategy> getStrategies() {

        if (strategies == null) {

            strategies = new ArrayList<Strategy>();

            if (propertiesParameters != null) {
                strategies = new ArrayList<Strategy>();

                try {

                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    StrategyDAO strategyDAO = IsisFishDAOHelper.getStrategyDAO(tx);
                    String[] strategyList = propertiesParameters.getProperty("strategies", "").split(",");
                    for (String name : strategyList) {
                        if (name != null && !"".equals(name)) {
                            try {
                                Strategy str = strategyDAO.findByName(name);
                                strategies.add(str);
                            } catch (TopiaException eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find strategy: " + name, eee);
                                }
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get StrategyDAO", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip strategies reloading");
                }
            }
        }
        return strategies;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setStrategies(java.util.List)
     */
    @Override
    public void setStrategies(List<Strategy> strategies) {
        this.strategies = strategies;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addSimulationPlan(fr.ifremer.isisfish.simulator.SimulationPlan)
     */
    @Override
    public void addSimulationPlan(SimulationPlan plan) {
        getSimulationPlans().add(plan);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#removeSimulationPlan(fr.ifremer.isisfish.simulator.SimulationPlan)
     */
    @Override
    public boolean removeSimulationPlan(SimulationPlan plan) {
        return getSimulationPlans().remove(plan);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulationPlans()
     */
    @Override
    public List<SimulationPlan> getSimulationPlans() {
        if (simulationPlans == null) {

            simulationPlans = new ArrayList<SimulationPlan>();

            if (propertiesParameters != null) {
                // simulation plan
                String[] planList = propertiesParameters.getProperty("plans",
                        "").split(",");
                int planIndex = 0;
                for (String name : planList) {
                    if (name != null && !"".equals(name)) {
                        try {
                            SimulationPlanStorage planStorage = SimulationPlanStorage.getSimulationPlan(name);
                            if (planStorage != null) { // since 4.0.0.3 can return null
                                SimulationPlan plan = planStorage.getNewSimulationPlanInstance();
                                StorageHelper.populateStorageParams(planIndex++,
                                        getRegion().getStorage(), plan, propertiesParameters, "plan");
                                simulationPlans.add(plan);
                            }
                        } catch (IsisFishException eee) {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find plan: " + name, eee);
                            }
                        }
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip plans reloading");
                }
            }
        }
        return this.simulationPlans;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulationPlans(java.util.List)
     */
    @Override
    public void setSimulationPlans(List<SimulationPlan> plans) {
        this.simulationPlans = plans;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isIndependentPlan()
     */
    @Override
    public boolean isIndependentPlan() {
        boolean result = true;
        for (SimulationPlan plan : getSimulationPlans()) {
            if (!(plan instanceof SimulationPlanIndependent)) {
                result = false;
                break;
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#clearPlans()
     */
    @Override
    public void clearPlans() {
        if (simulationPlans != null) {
            simulationPlans.clear();
        }
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addRule(fr.ifremer.isisfish.rule.Rule)
     */
    @Override
    public void addRule(Rule rule) {
        getRules().add(rule);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#removeRule(fr.ifremer.isisfish.rule.Rule)
     */
    @Override
    public boolean removeRule(Rule rule) {
        return getRules().remove(rule);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRules()
     */
    @Override
    public List<Rule> getRules() {
        if (rules == null) {
            rules = new ArrayList<Rule>();

            if (propertiesParameters != null) {
                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    // rules
                    String[] ruleList = propertiesParameters.getProperty("rules", "").split(",");
                    int ruleIndex = 0;
                    for (String name : ruleList) {
                        if (name != null && !"".equals(name)) {
                            try {
                                Rule rule = RuleStorage.getRule(name).getNewRuleInstance();
                                RuleHelper.populateRule(ruleIndex++, tx, rule, propertiesParameters);
                                rules.add(rule);
                            } catch (IsisFishException eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find rule: " + name, eee);
                                }
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get TopiaContext", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip rule reloading");
                }
            }
        }
        return rules;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setRules(java.util.List)
     */
    @Override
    public void setRules(List<Rule> rules) {
        this.rules = rules;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#clearRules()
     */
    @Override
    public void clearRules() {
        if (rules != null) {
            rules.clear();
        }
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseOptimization()
     */
    @Override
    public boolean getUseOptimization() {

        if (useOptimization == null) {
            if (propertiesParameters != null) {
                useOptimization = Boolean.valueOf(propertiesParameters.getProperty("useOptimization", "true"));
            } else {
                useOptimization = Boolean.TRUE;
            }
        }
        return useOptimization.booleanValue();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseOptimization(boolean)
     */
    @Override
    public void setUseOptimization(boolean useOptimization) {
        this.useOptimization = useOptimization;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseStatistic()
     */
    @Override
    public boolean getUseStatistic() {

        if (useStatistic == null) {
            if (propertiesParameters != null) {
                useStatistic = Boolean.valueOf(propertiesParameters.getProperty("useStatistic", "false"));
            } else {
                useStatistic = Boolean.FALSE;
            }
        }
        return this.useStatistic;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseStatistic(boolean)
     */
    @Override
    public void setUseStatistic(boolean useStatistic) {
        this.useStatistic = useStatistic;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getExportNames()
     */
    @Override
    public List<String> getExportNames() {
        if (exportNames == null) {
            exportNames = new ArrayList<String>();

            if (propertiesParameters != null) {
                // exports
                String[] exportList = propertiesParameters.getProperty("exports", "").split(",");
                for (String name : exportList) {
                    if (name != null && !"".equals(name)) {
                        exportNames.add(name);
                    }
                }
            }
        }
        return this.exportNames;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setExportNames(java.util.List)
     */
    @Override
    public void setExportNames(List<String> exportNames) {
        this.exportNames = exportNames;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOfSensitivitySimulation()
     */
    @Override
    public int getNumberOfSensitivitySimulation() {

        if (numberOfSensitivitySimulation == null) {

            if (propertiesParameters != null) {
                numberOfSensitivitySimulation = Integer.valueOf(propertiesParameters.getProperty(
                                "numberOfSensitivitySimulation", "-1"));
            } else {
                numberOfSensitivitySimulation = Integer.valueOf(-1);
            }
        }

        return numberOfSensitivitySimulation.intValue();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setNumberOfSensitivitySimulation(int)
     */
    @Override
    public void setNumberOfSensitivitySimulation(
            int numberOfSensitivitySimulation) {
        this.numberOfSensitivitySimulation = numberOfSensitivitySimulation;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSensitivityAnalysis()
     */
    @Override
    public SensitivityAnalysis getSensitivityAnalysis() {

        if (sensitivityAnalysis == null) {
            if (propertiesParameters != null) {
                String analysisName = propertiesParameters
                        .getProperty("sensitivityanalysis");
                if (!StringUtils.isEmpty(analysisName)) {
                    try {
                        SensitivityAnalysisStorage sensitivityStorage = SensitivityAnalysisStorage.getSensitivityAnalysis(analysisName);
                        sensitivityAnalysis = sensitivityStorage.getNewSensitivityAnalysisInstance();
                        // 0 = only single sensitivity
                        StorageHelper.populateStorageParams(0, getRegion().getStorage(), sensitivityAnalysis,
                                propertiesParameters, "sensitivity");
                    } catch (IsisFishException eee) {
                        sensitivityAnalysis = null;
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find sensitivity: " + sensitivityAnalysis, eee);
                        }
                    }
                }
            }
        }

        return sensitivityAnalysis;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSensitivityAnalysis(fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis)
     */
    @Override
    public void setSensitivityAnalysis(
            SensitivityAnalysis sensitivityAnalysis) {
        this.sensitivityAnalysis = sensitivityAnalysis;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSensitivityExport()
     */
    @Override
    public List<SensitivityExport> getSensitivityExport() {
        if (sensitivityExports == null) {
            sensitivityExports = new ArrayList<SensitivityExport>();

            if (propertiesParameters != null) {
                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDbResult();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    // sensitivity export
                    String[] sensitivityExportList = propertiesParameters
                            .getProperty("sensitivityexports", "").split(",");
                    int sensitivityExportIndex = 0;
                    for (String name : sensitivityExportList) {
                        try {
                            if (!StringUtils.isEmpty(name)) {
                                SensitivityExport sensitivityExport = SensitivityExportStorage
                                        .getSensitivityExport(name)
                                        .getNewSensitivityExportInstance();
                                ExportHelper.populateSensitivityExport(
                                                sensitivityExportIndex++, tx,
                                                sensitivityExport,
                                                propertiesParameters);
                                sensitivityExports.add(sensitivityExport);
                            }
                        } catch (IsisFishException eee) {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find SensitivityExport: " + name, eee);
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get TopiaContext", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip sensitivity exports reloading");
                }
            }
        }
        return sensitivityExports;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSensitivityExport(java.util.List)
     */
    @Override
    public void setSensitivityExport(List<SensitivityExport> sensitivityExport) {
        this.sensitivityExports = sensitivityExport;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseAnalysePlan()
     */
    @Override
    public boolean getUseSimulationPlan() {

        if (useSimulationPlan == null) {

            if (propertiesParameters != null) {
                useSimulationPlan = Boolean.valueOf(propertiesParameters.getProperty("useSimulationPlan", "false"));
            } else {
                useSimulationPlan = Boolean.FALSE;
            }
        }
        return this.useSimulationPlan;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseAnalysePlan(boolean)
     */
    @Override
    public void setUseSimulationPlan(boolean useSimulationPlan) {
        this.useSimulationPlan = useSimulationPlan;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulationPlanNumber()
     */
    @Override
    public int getSimulationPlanNumber() {

        if (simulationPlanNumber == null) {

            if (propertiesParameters != null) {
                simulationPlanNumber = Integer.valueOf(propertiesParameters.getProperty("simulationPlanNumber", "-1"));
            } else {
                simulationPlanNumber = Integer.valueOf(-1);
            }
        }

        return this.simulationPlanNumber.intValue();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulationPlanNumber(int)
     */
    @Override
    public void setSimulationPlanNumber(int simulationPlanNumber) {
        this.simulationPlanNumber = simulationPlanNumber;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOfYear()
     */
    @Override
    public int getNumberOfYear() {

        if (numberOfYear == null) {

            if (propertiesParameters != null) {
                numberOfYear = Integer.valueOf(propertiesParameters.getProperty("numberOfYear", "1"));
            } else {
                numberOfYear = Integer.valueOf(1);
            }
        }
        return numberOfYear.intValue();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setNumberOfYear(int)
     */
    @Override
    public void setNumberOfYear(int numberOfYear) {
        this.numberOfYear = numberOfYear;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUsePreScript()
     */
    @Override
    public boolean getUsePreScript() {

        if (usePreScript == null) {

            if (propertiesParameters != null) {
                usePreScript = Boolean.valueOf(propertiesParameters.getProperty("usePreScript", "false"));
            } else {
                usePreScript = Boolean.FALSE;
            }
        }

        return usePreScript.booleanValue();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUsePreScript(boolean)
     */
    @Override
    public void setUsePreScript(boolean usePreScript) {
        this.usePreScript = usePreScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getPreScript()
     */
    @Override
    public String getPreScript() {

        if (preScript == null) {

            if (propertiesParameters != null) {
                preScript = propertiesParameters.getProperty("preScript", "");
            } else {
                preScript = "";
            }
        }

        return preScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setPreScript(java.lang.String)
     */
    @Override
    public void setPreScript(String preScript) {
        this.preScript = preScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRegionName()
     */
    @Override
    public String getRegionName() {

        if (regionName == null) {
            if (propertiesParameters != null) {
                regionName = propertiesParameters.getProperty("regionName", "");
            } else {
                // defaut value
                regionName = "";
            }
        }

        return regionName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setRegionName(java.lang.String)
     */
    @Override
    public void setRegionName(String regionName) {
        this.regionName = regionName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulatorName()
     */
    @Override
    public String getSimulatorName() {

        if (simulatorName == null) {
            if (propertiesParameters != null) {
                simulatorName = propertiesParameters.getProperty(
                        "simulatorName", "DefaultSimulator.java");
            } else {
                simulatorName = "DefaultSimulator.java";
            }
        }
        return simulatorName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulatorName(java.lang.String)
     */
    @Override
    public void setSimulatorName(String simulatorName) {
        this.simulatorName = simulatorName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getResultEnabled()
     */
    @Override
    public Collection<String> getResultEnabled() {

        if (resultEnabled == null) {

            resultEnabled = new LinkedList<String>();

            if (propertiesParameters != null) {
                String[] resultList = propertiesParameters.getProperty("resultNames", "").split(",");
                for (String name : resultList) {
                    if (name != null && !"".equals(name)) {
                        resultEnabled.add(name);
                    }
                }
            }
        }
        return resultEnabled;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setResultEnabled(java.util.Collection)
     */
    @Override
    public void setResultEnabled(Collection<String> resultEnabled) {
        this.resultEnabled = resultEnabled;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getTagValue()
     */
    @Override
    public Map<String, String> getTagValue() {

        if (tagValue == null) {
            tagValue = new LinkedHashMap<String, String>();

            if (propertiesParameters != null) {
                for (Object k : propertiesParameters.keySet()) {
                    if (k.toString().startsWith("tagValue.")) {
                        String key = k.toString().substring("tagValue.".length());
                        String value = propertiesParameters.getProperty(k.toString());
                        getTagValue().put(key, value);
                    }
                }
            }
        }

        return tagValue;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setTagValue(java.util.Map)
     */
    @Override
    public void setTagValue(Map<String, String> tagValue) {
        this.tagValue = tagValue;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulLogLevel()
     */
    @Override
    public String getSimulLogLevel() {

        if (simulLogLevel == null) {

            if (propertiesParameters != null) {
                simulLogLevel = propertiesParameters.getProperty("simulLogLevel", "info");
            } else {
                simulLogLevel = "info";
            }
        }

        return simulLogLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulLogLevel(java.lang.String)
     */
    @Override
    public void setSimulLogLevel(String logLevel) {
        if (log.isDebugEnabled()) {
            log.debug(_("isisfish.params.changeLogLev", simulLogLevel,
                            logLevel));
        }
        this.simulLogLevel = logLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getScriptLogLevel()
     */
    @Override
    public String getScriptLogLevel() {

        if (scriptLogLevel == null) {
            if (propertiesParameters != null) {
                scriptLogLevel = propertiesParameters.getProperty(
                        "scriptLogLevel", "info");
            } else {
                scriptLogLevel = "info";
            }
        }

        return scriptLogLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setScriptLogLevel(java.lang.String)
     */
    @Override
    public void setScriptLogLevel(String logLevel) {
        if (log.isDebugEnabled()) {
            log.debug(_("isisfish.params.changeLogLev", scriptLogLevel,
                    logLevel));
        }
        this.scriptLogLevel = logLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getLibLogLevel()
     */
    @Override
    public String getLibLogLevel() {
        if (libLogLevel == null) {
            if (propertiesParameters != null) {
                libLogLevel = propertiesParameters.getProperty("libLogLevel", "error");
            } else {
                libLogLevel = "error";
            }
        }

        return libLogLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setLibLogLevel(java.lang.String)
     */
    @Override
    public void setLibLogLevel(String logLevel) {
        if (log.isDebugEnabled()) {
            log.debug(_("isisfish.params.changeLogLev", libLogLevel, logLevel));
        }
        this.libLogLevel = logLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulErrorLevel()
     */
    @Override
    public boolean isSimulErrorLevel() {
        return "error".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulWarnLevel()
     */
    @Override
    public boolean isSimulWarnLevel() {
        return "warn".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulInfoLevel()
     */
    @Override
    public boolean isSimulInfoLevel() {
        return "info".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulDebugLevel()
     */
    @Override
    public boolean isSimulDebugLevel() {
        return "debug".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptErrorLevel()
     */
    @Override
    public boolean isScriptErrorLevel() {
        return "error".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptWarnLevel()
     */
    @Override
    public boolean isScriptWarnLevel() {
        return "warn".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptInfoLevel()
     */
    @Override
    public boolean isScriptInfoLevel() {
        return "info".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptDebugLevel()
     */
    @Override
    public boolean isScriptDebugLevel() {
        return "debug".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibErrorLevel()
     */
    @Override
    public boolean isLibErrorLevel() {
        return "error".equals(getLibLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibWarnLevel()
     */
    @Override
    public boolean isLibWarnLevel() {
        return "warn".equals(getLibLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibInfoLevel()
     */
    @Override
    public boolean isLibInfoLevel() {
        return "info".equals(getLibLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibDebugLevel()
     */
    @Override
    public boolean isLibDebugLevel() {
        return "debug".equals(getLibLogLevel());
    }

    /**
     * Permet d'ajouter des parametres directement à partir de leur
     * representation chaine.
     * 
     * A ne pas utiliser normalement, sert uniquement dans les prescripts des
     * simulation des AS.
     * 
     * @param key key
     * @param value value
     * @since 3.4.0.0
     */
    @Override
    public void setProperty(String key, String value) {
        propertiesParameters.setProperty(key, value);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#copy()
     */
    @Override
    public SimulationParameter copy() {
        SimulationParameterImpl result = new SimulationParameterImpl();

        if (propertiesParameters != null) {
            result.propertiesParameters = new SortedProperties();
            // I think that iteration work better than parameters in contructor
            result.propertiesParameters.putAll(propertiesParameters);
        }

        // still needed to be copied, if there is no propertiesParameters
        result.isisFishVersion = getIsisFishVersion();
        result.description = getDescription();
        result.regionName = getRegionName();
        result.numberOfYear = getNumberOfYear();
        result.simulatorName = getSimulatorName();
        result.useOptimization = getUseOptimization();
        result.useStatistic = getUseStatistic();
        result.usePreScript = getUsePreScript();
        result.preScript = getPreScript();
        result.useSimulationPlan = getUseSimulationPlan();
        result.simulationPlanNumber = getSimulationPlanNumber();
        if (exportNames != null) {
            result.exportNames = new LinkedList<String>(exportNames);
        }
        if (strategies != null) {
            result.strategies = new LinkedList<Strategy>(strategies);
        }
        if (populations != null) {
            result.populations = new LinkedList<Population>(populations);
        }
        if (rules != null) {
            result.rules = new LinkedList<Rule>(rules);
        }
        if (simulationPlans != null) {
            result.simulationPlans = new LinkedList<SimulationPlan>(simulationPlans);
        }
        if (resultEnabled != null) {
            result.resultEnabled = new LinkedList<String>(resultEnabled);
        }
        if (numbers != null) {
            result.numbers = new HashMap<Population, MatrixND>(numbers);
        }
        if (tagValue != null) {
            result.tagValue = new HashMap<String, String>(tagValue);
        }
        result.simulLogLevel = getSimulLogLevel();
        result.scriptLogLevel = getScriptLogLevel();
        result.libLogLevel = getLibLogLevel();
        result.numberOfSensitivitySimulation = getNumberOfSensitivitySimulation();
        if (sensitivityExports != null) {
            result.sensitivityExports = new LinkedList<SensitivityExport>(sensitivityExports);
        }
        result.sensitivityAnalysis = sensitivityAnalysis;

        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#deepCopy()
     */
    @Override
    public SimulationParameter deepCopy() {
        Properties props = toProperties();
        SimulationParameter newInstance = new SimulationParameterImpl();
        newInstance.fromProperties(props);
        return newInstance;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#toString()
     */
    @Override
    public String toString() {
        StringBuffer result = new StringBuffer();
        result.append(
                _("isisfish.params.toString.simulation.done",
                        getIsisFishVersion())).append('\n');
        result.append("--------------------\n");
        result.append(getDescription()).append('\n');
        result.append("--------------------\n");

        if (getUseSimulationPlan()) {
            result.append(_("isisfish.params.toString.plan.number",
                    getSimulationPlanNumber()));
        }

        RegionStorage region = getRegion();
        result.append(_("isisfish.params.toString.fishery", region.getName()));
        result.append(_("isisfish.params.toString.number.years",
                getNumberOfYear()));

        // startegies
        List<Strategy> strs = getStrategies();
        result.append(_("isisfish.params.toString.strategies"));
        String sep = "";
        for (Strategy str : strs) {
            result.append(sep).append(str.getName());
            sep = ",";
        }
        result.append("\n\n");

        // populations
        result.append(_("isisfish.params.toString.populations"));
        sep = "";
        for (Population pop : getPopulations()) {
            result.append(sep).append(pop.getName());
            sep = ",";
        }
        result.append("\n\n");

        // rules
        for (Rule rule : getRules()) {
            String name = RuleStorage.getName(rule);
            String param = "";
            try {
                param = RuleStorage.getParamAsString(rule);
            } catch (IsisFishException eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't convert rule parameter to string for "
                            + name, eee);
                }
            }
            result.append(_("isisfish.params.toString.rule", name));
            result.append('\n');
            result.append(param);
            result.append("\n");

        }
        result.append("\n\n");

        // simulation plans
        for (SimulationPlan plan : getSimulationPlans()) {
            String name = RuleStorage.getName(plan);
            String param = "";
            try {
                param = SimulationPlanStorage.getParamAsString(plan);
            } catch (IsisFishException eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't convert plan parameter to string for "
                            + name, eee);
                }
            }
            result.append(_("isisfish.params.toString.plan", name));
            result.append('\n');
            result.append(param);
            result.append('\n');

        }
        result.append('\n');
        result.append(_("isisfish.params.toString.script.presimulation"))
                .append(":\n");
        result.append(getPreScript());
        result.append('\n');
        result.append(_("isisfish.params.toString.simul.logger.level",
                simulLogLevel));
        result.append('\n');
        result.append(_("isisfish.params.toString.script.logger.level",
                scriptLogLevel));
        result.append('\n');
        result.append(_("isisfish.params.toString.lib.logger.level",
                libLogLevel));
        result.append('\n');
        return result.toString();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#toProperties()
     */
    @Override
    public Properties toProperties() {
        Properties result = new SortedProperties();

        result.setProperty("isisFishVersion", getIsisFishVersion());
        result.setProperty("description", getDescription());
        result.setProperty("regionName", getRegionName());
        result.setProperty("numberOfYear", String.valueOf(getNumberOfYear()));
        result.setProperty("simulatorName", getSimulatorName());
        result.setProperty("useOptimization", String.valueOf(getUseOptimization()));
        result.setProperty("useStatistic", String.valueOf(getUseStatistic()));

        // strategies
        if (strategies != null) {
            String strategyList = "";
            for (Strategy str : getStrategies()) {
                strategyList += str.getName() + ",";
            }
            result.setProperty("strategies", StringUtil.substring(strategyList, 0, -1));
        } else {
            if (propertiesParameters != null
                    && propertiesParameters.containsKey("strategies")) {
                result.setProperty("strategies", propertiesParameters.getProperty("strategies"));
            }
        }

        // populations
        if (populations != null) {
            String populationList = "";
            for (Population pop : getPopulations()) {
                populationList += pop.getName() + ",";
                MatrixND number = getNumberOf(pop);
                String numberAsString = String.valueOf(number.toList());
                result.setProperty("population." + pop.getName() + ".number", numberAsString);
            }
            result.setProperty("populations", StringUtil.substring(
                    populationList, 0, -1));
        } else {
            if (propertiesParameters != null
                    && propertiesParameters.containsKey("populations")) {
                result.setProperty("populations", propertiesParameters.getProperty("populations"));
                for (String key : propertiesParameters.stringPropertyNames()) {
                    if (key.startsWith("population.")) {
                        result.setProperty(key, propertiesParameters.getProperty(key));
                    }
                }
            }
        }

        // rules
        if (rules != null) {
            String ruleList = "";
            int ruleIndex = 0;
            for (Rule rule : getRules()) {
                ruleList += RuleStorage.getName(rule) + ",";
                Properties ruleProp = RuleHelper.getRuleAsProperties(ruleIndex++, getRegion().getStorage(), rule);
                result.putAll(ruleProp);
            }
            result.setProperty("rules", ruleList);
        } else {
            if (propertiesParameters != null
                    && propertiesParameters.containsKey("rules")) {
                result.setProperty("rules", propertiesParameters
                        .getProperty("rules"));
                for (String key : propertiesParameters.stringPropertyNames()) {
                    if (key.startsWith("rule.")) {
                        result.setProperty(key, propertiesParameters
                                .getProperty(key));
                    }
                }
            }
        }

        // anaylyse plans
        if (simulationPlans != null) {
            String planList = "";
            int planIndex = 0;
            for (SimulationPlan plan : getSimulationPlans()) {
                planList += SimulationPlanStorage.getName(plan) + ",";
                Properties planProp = StorageHelper.getParamsAsProperties(planIndex++,
                        getRegion().getStorage(), plan, "plan");
                result.putAll(planProp);
            }
            result.setProperty("plans", planList);
        } else {
            if (propertiesParameters != null
                    && propertiesParameters.containsKey("plans")) {
                result.setProperty("plans", propertiesParameters
                        .getProperty("plans"));
                for (String key : propertiesParameters.stringPropertyNames()) {
                    if (key.startsWith("plan.")) {
                        result.setProperty(key, propertiesParameters.getProperty(key));
                    }
                }
            }
        }

        // export names
        String exportList = "";
        for (String export : getExportNames()) {
            exportList += export + ",";

        }
        result.setProperty("exports", exportList);

        // number of sensitivity simulation
        result.setProperty("numberOfSensitivitySimulation", String
                .valueOf(getNumberOfSensitivitySimulation()));

        // analysis name
        if (sensitivityAnalysis != null) {
            String analysisName = SensitivityAnalysisStorage
                    .getName(getSensitivityAnalysis());
            result.setProperty("sensitivityanalysis", analysisName);

            // analysis parameter
            Properties analysisParams = StorageHelper.getParamsAsProperties(0,
                    getRegion().getStorage(), getSensitivityAnalysis(), "sensitivity");
            result.putAll(analysisParams);
        } else {
            if (propertiesParameters != null
                    && propertiesParameters.containsKey("sensitivityanalysis")) {
                result.setProperty("sensitivityanalysis",
                        propertiesParameters.getProperty("sensitivityanalysis"));
                for (String key : propertiesParameters.stringPropertyNames()) {
                    if (key.startsWith("sensitivity.")) {
                        result.setProperty(key, propertiesParameters.getProperty(key));
                    }
                }
            }
        }

        // sensitivity exports
        if (sensitivityExports != null) {
            String sensitivityExportList = "";
            int sensitivityExportIndex = 0;
            for (SensitivityExport sensitivityExport : getSensitivityExport()) {
                sensitivityExportList += SensitivityExportStorage
                        .getName(sensitivityExport) + ",";
                Properties exportProp = ExportHelper.getSensitivityExportAsProperties(
                        sensitivityExportIndex++, getRegion().getStorage(), sensitivityExport);
                result.putAll(exportProp);
            }
            result.setProperty("sensitivityexports", sensitivityExportList);
        } else {
            if (propertiesParameters != null
                    && propertiesParameters.containsKey("sensitivityexports")) {
                result.setProperty("sensitivityexports", propertiesParameters
                        .getProperty("sensitivityexports"));
                for (String key : propertiesParameters.stringPropertyNames()) {
                    if (key.startsWith("sensitivityexport.")) {
                        result.setProperty(key, propertiesParameters.getProperty(key));
                    }
                }
            }
        }

        result.setProperty("usePreScript", String.valueOf(getUsePreScript()));
        result.setProperty("preScript", getPreScript());
        result.setProperty("useSimulationPlan", String.valueOf(getUseSimulationPlan()));
        result.setProperty("simulationPlanNumber", String.valueOf(getSimulationPlanNumber()));

        String resultList = "";
        for (String r : getResultEnabled()) {
            resultList += r + ",";
        }
        result.setProperty("resultNames", resultList);

        for (Map.Entry<String, String> e : getTagValue().entrySet()) {
            result.setProperty("tagValue." + e.getKey(), e.getValue());
        }

        result.setProperty("simulLogLevel", getSimulLogLevel());
        result.setProperty("scriptLogLevel", getScriptLogLevel());
        result.setProperty("libLogLevel", getLibLogLevel());
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#fromProperties(java.util.Properties)
     */
    @Override
    public void fromProperties(Properties props) {

        // save properties (use full to read again parameter)
        // for exports, or rules....
        this.propertiesParameters = props;

    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#reloadContextParameters()
     */
    @Override
    public void reloadContextParameters() throws TopiaException {

        // On verifie tout d'abord que l'on ai pas dans une simulation
        // si on y es, on utilise le context static non null du thread local
        // Resoud les lazy exceptions des parametres des regles
        boolean mustClose = false;
        TopiaContext tx = SimulationContext.get().getDB();

        if (tx == null) {
            // not in simulation, create transaction
            tx = getRegion().getStorage().beginTransaction();
            mustClose = true;
        }

        // reload rules
        if (log.isDebugEnabled()) {
            log.debug("Reloading rules");
        }
        for (Rule rule : getRules()) {
            try {
                for (Field field : rule.getClass().getFields()) {
                    // le champ ne doit pas être privé
                    if (Modifier.isPublic(field.getModifiers())) {
                        // si le type est topia entity (ou sous classe)
                        if (TopiaEntity.class.isAssignableFrom(field.getType())) {
                            TopiaEntity entity = (TopiaEntity) field.get(rule);
                            // il est possible que les parametres soient null
                            // dans ce cas, il ne faut pas essayer de les recharger
                            if (entity != null) {
                                // reloading
                                TopiaEntity newEntity = tx.findByTopiaId(entity
                                        .getTopiaId());
                                field.set(rule, newEntity);
                            }
                        }
                    }
                }
            } catch (IllegalArgumentException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't refresh rule field", e);
                }
            } catch (IllegalAccessException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't access rule field", e);
                }
            }
        }

        // si la transaction a été ouverte (pas dans une simulation)
        // on la referme
        if (mustClose) {
            tx.commitTransaction();
            tx.closeContext();
        }
    }
}
