/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationParameter.java 3445 2011-07-15 13:12:45Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2010 - 2011 Ifremer, Code Lutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaException;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;

/**
 * Contains all parameters for one simulation.
 *
 * {@link SimulationParameter} must now be loaded from a {@link Properties}
 * object.
 * {@link #toProperties()} method, now return a new {@link Properties} object
 * filled with :
 * <ul>
 *  <li>non modified original properties
 *  <li>new properties if some objects has been instantiated by getters().
 * </ul>
 * 
 * Created: 10 janv. 2006 17:03:37
 *
 * @author poussin
 * @author chatellier
 * 
 * @version $Revision: 3445 $
 *
 * Last update: $Date: 2011-07-15 15:12:45 +0200 (Fri, 15 Jul 2011) $
 * By : $Author: chatellier $
 */
public interface SimulationParameter {

    /**
     * Permet de verifier que tous les parametres sont bon. Si ce n'est pas le
     * cas la liste retournée contient la liste des messages d'erreur.
     *
     * @return la liste d'erreur
     */
    public List<String> check();

    /**
     * Get isis fish version for the simulation.
     * 
     * Default value to : {@link IsisConfig#getVersion()}.
     * 
     * @return the isisFishVersion.
     */
    public String getIsisFishVersion();

    /**
     * Set isis fish Version.
     * 
     * @param isisFishVersion The isisFishVersion to set.
     */
    public void setIsisFishVersion(String isisFishVersion);

    /**
     * Get description property.
     *
     * Default value to : "".
     * 
     * @return Description property.
     */
    public abstract String getDescription();

    /**
     * Set simulation description.
     * 
     * @param description the description to set.
     */
    public abstract void setDescription(String description);

    /**
     * Load region corresponding to {@link #getRegionName()}.
     * 
     * @return the region
     */
    public abstract RegionStorage getRegion();

    public abstract MatrixND getNumberOf(Population pop);

    /**
     * Get extra rules list.
     * 
     * @return Returns the extraRules.
     */
    public abstract List<String> getExtraRules();

    /**
     * Add extra rules.
     * 
     * Appelé par les plan de simulation, pour ajouter des regles supplémentaires
     * dans être integrés au zip.
     * 
     * @param extraRules extra rules to add
     */
    public abstract void addExtraRules(String... extraRules);

    /**
     * Get instantiated population list.
     * 
     * @return Returns the populations.
     */
    public abstract List<Population> getPopulations();

    /**
     * @param populations The populations to set.
     */
    public abstract void setPopulations(List<Population> populations);

    /**
     * Get instantiated strategies list.
     * 
     * Defaut to : empty array list
     * 
     * @return Returns the strategies.
     */
    public abstract List<Strategy> getStrategies();

    /**
     * Set strategies list.
     * 
     * @param strategies strategies list to set
     */
    public abstract void setStrategies(List<Strategy> strategies);

    /**
     * Add new simulation plan.
     * 
     * @param plan simulation plan to add
     */
    public abstract void addSimulationPlan(SimulationPlan plan);

    /**
     * Remove a plan.
     * 
     * @param plan
     * @return {@code true} if this list contained the specified element
     */
    public abstract boolean removeSimulationPlan(SimulationPlan plan);

    /**
     * Get instantiated simulation plan list.
     * 
     * Default to empty list.
     * 
     * @return the plans.
     */
    public abstract List<SimulationPlan> getSimulationPlans();

    /**
     * Set plans list.
     * 
     * @param plans plans to set
     */
    public abstract void setSimulationPlans(List<SimulationPlan> plans);

    /**
     * Return {@code true} if simulation is composed of independent plan only.
     * 
     * @return {@code true} if all {@link SimulationPlan} are {@link SimulationPlanIndependent}
     */
    public abstract boolean isIndependentPlan();

    /**
     * Clear plan list.
     */
    public abstract void clearPlans();

    /**
     * Add new rule to rules list.
     * @param rule rule to add
     */
    public abstract void addRule(Rule rule);

    /**
     * Remove a rule.
     * 
     * @param rule rule to remove
     * @return {@code true} if this list contained the specified element
     */
    public abstract boolean removeRule(Rule rule);

    /**
     * Get parameters rules list.
     *
     * @return the rules
     */
    public abstract List<Rule> getRules();

    /**
     * Set simulation rules.
     * 
     * @param rules rules to set
     */
    public abstract void setRules(List<Rule> rules);

    /**
     * Clear rule list.
     */
    public abstract void clearRules();

    /**
     * Return if optimization sould be used.
     * 
     * Default to : true.
     * 
     * @return use optimization
     */
    public abstract boolean getUseOptimization();

    /**
     * Change use optimization parameter.
     * 
     * @param useOptimization use optimization to set
     */
    public abstract void setUseOptimization(boolean useOptimization);

    /**
     * Return if statistic sould be used.
     * 
     * Default to false;
     * 
     * @return use statistic
     */
    public abstract boolean getUseStatistic();

    /**
     * Change use statistic property.
     * 
     * @param useStatistic use statistic to set
     */
    public abstract void setUseStatistic(boolean useStatistic);

    /**
     * Get export names list.
     * 
     * @return export names list
     */
    public abstract List<String> getExportNames();

    /**
     * Set export names list.
     * 
     * @param exportNames export names list to set
     */
    public abstract void setExportNames(List<String> exportNames);

    /**
     * Get number of sensitivity simulation.
     * 
     * Default value to : -1
     * 
     * @return the numberOfSensitivitySimulation
     */
    public abstract int getNumberOfSensitivitySimulation();

    /**
     * Set number of sensitivity simulation.
     * 
     * @param numberOfSensitivitySimulation number of sensitivity simulation to set
     */
    public abstract void setNumberOfSensitivitySimulation(
            int numberOfSensitivitySimulation);

    /**
     * Get instantiated sensitivity calculator.
     * 
     * Default to : null.
     * 
     * @return the sensitivityAnalysis
     */
    public abstract SensitivityAnalysis getSensitivityAnalysis();

    /**
     * @param sensitivityAnalysis the sensitivityAnalysis to set
     */
    public abstract void setSensitivityAnalysis(
            SensitivityAnalysis sensitivityAnalysis);

    /**
     * Return loaded sensitivity export.
     * 
     * If exports are null or empty, try to load it from last ready parameters.
     * 
     * @return the sensitivityExportNames
     */
    public abstract List<SensitivityExport> getSensitivityExport();

    /**
     * @param sensitivityExport the sensitivityExportNames to set
     */
    public abstract void setSensitivityExport(
            List<SensitivityExport> sensitivityExport);

    /**
     * Get use simulation plans property.
     * 
     * @return use simulation plan.
     */
    public abstract boolean getUseSimulationPlan();

    /**
     * Set use simulation plans property.
     * 
     * @param useSimulationPlan use simulation plan to set
     */
    public abstract void setUseSimulationPlan(boolean useSimulationPlan);

    /**
     * Get simulation number in simulation plan.
     * 
     * @return simulation number in simulation plan
     */
    public abstract int getSimulationPlanNumber();

    /**
     * Set simulation number in simulation plan.
     * 
     * @param simulationPlanNumber simulation plan number to set
     */
    public abstract void setSimulationPlanNumber(int simulationPlanNumber);

    /**
     * Get number of year to run to simulate.
     * 
     * @return number of year
     */
    public abstract int getNumberOfYear();

    /**
     * Set number of year to run to simulate.
     * 
     * @param numberOfYear number of year to set
     */
    public abstract void setNumberOfYear(int numberOfYear);

    /**
     * Get use prescript.
     * 
     * Default to false.
     * 
     * @return Returns the usePreScript.
     */
    public abstract boolean getUsePreScript();

    /**
     * Set use prescript property
     * 
     * @param usePreScript use preScript to set
     */
    public abstract void setUsePreScript(boolean usePreScript);

    /**
     * Get prescript content.
     * 
     * @return preScript content
     */
    public abstract String getPreScript();

    /**
     * Set pre script content.
     * 
     * @param preScript prescript content
     */
    public abstract void setPreScript(String preScript);

    /**
     * Get region name.
     * 
     * @return region name
     */
    public abstract String getRegionName();

    /**
     * Set region name.
     * 
     * @param regionName region name to set
     */
    public abstract void setRegionName(String regionName);

    /**
     * Set simulator name.
     * 
     * @return simulator name.
     */
    public abstract String getSimulatorName();

    /**
     * Set simulator name.
     * 
     * @param simulatorName simulator name to set
     */
    public abstract void setSimulatorName(String simulatorName);

    /**
     * Get enabled result names list.
     * 
     * @return enabled result names list
     */
    public abstract Collection<String> getResultEnabled();

    /**
     * Set enabled result names list.
     * 
     * @param resultEnabled enabled result names list
     */
    public abstract void setResultEnabled(Collection<String> resultEnabled);

    /**
     * Get tag values.
     * 
     * Default to empty map
     * 
     * @return Returns the tagValue.
     */
    public abstract Map<String, String> getTagValue();

    /**
     * Set tag values.
     * 
     * @param tagValue tagValues to set.
     */
    public abstract void setTagValue(Map<String, String> tagValue);

    /**
     * Get simulator log level.
     * 
     * Default to "info".
     * 
     * @return simualtor log level
     */
    public abstract String getSimulLogLevel();

    /**
     * Set simulator log level.
     * 
     * @param logLevel simulator log level
     */
    public abstract void setSimulLogLevel(String logLevel);

    /**
     * Get script log level.
     * 
     * Default to "info".
     * 
     * @return script log level
     */
    public abstract String getScriptLogLevel();

    /**
     * Set script log level.
     * 
     * @param logLevel script log level
     */
    public abstract void setScriptLogLevel(String logLevel);

    /**
     * Get librairies log level.
     * 
     * Default to "error".
     * 
     * @return librairies log level
     */
    public abstract String getLibLogLevel();

    /**
     * Set lib log level.
     * 
     * @param logLevel
     */
    public abstract void setLibLogLevel(String logLevel);

    public abstract boolean isSimulErrorLevel();

    public abstract boolean isSimulWarnLevel();

    public abstract boolean isSimulInfoLevel();

    public abstract boolean isSimulDebugLevel();

    public abstract boolean isScriptErrorLevel();

    public abstract boolean isScriptWarnLevel();

    public abstract boolean isScriptInfoLevel();

    public abstract boolean isScriptDebugLevel();

    public abstract boolean isLibErrorLevel();

    public abstract boolean isLibWarnLevel();

    public abstract boolean isLibInfoLevel();

    public abstract boolean isLibDebugLevel();

    /**
     * Permet d'ajouter des parametres directement à partir de leur
     * representation chaine.
     * 
     * A ne pas utiliser normalement, sert uniquement dans les prescripts des
     * simulation des AS.
     * 
     * @param key key
     * @param value value
     * @since 3.4.0.0
     */
    public void setProperty(String key, String value);

    /**
     * A copy instance of SimulationParameter.
     * 
     * Warning, this is not a deep copy, already instancied objects are
     * not duplicated.
     * 
     * @return a copy of this instance
     */
    public abstract SimulationParameter copy();

    /**
     * Make a deep copy of current parameters.
     * 
     * Bump all current parameters to properties and make a new one with
     * those propeties.
     * 
     * @return new parameters instance
     */
    public abstract SimulationParameter deepCopy();

    /**
     * The toString() method call getters.
     * 
     * So make instances of rules/export/plans...
     */
    public abstract String toString();

    /**
     * Permet de convertir l'objet SimulationParameter en un objet Properties
     * Cela permet de le stocker facilement sur le disque.
     * 
     * Recopie certaines proprietes si elle n'ont pas été instancié :
     * <ul>
     * <li>strategies</li>
     * <li>rules</li>
     * <li>simulationplans</li>
     * <li>sensitivityexports</li>
     * <li>sensitivityanalysis</li>
     * </ul>
     *
     * @return L'objet Properties representant les parametres
     * @see #fromProperties(Properties)
     */
    public abstract Properties toProperties();

    /**
     * Load properties from file.
     *
     * @param props property to read
     */
    public abstract void fromProperties(Properties props);

    /**
     * Reload parameters du to context change.
     * 
     * ie : in simulators when rollbacking transaction
     * 
     * Actually : reload rules parameters
     * 
     * @throws TopiaException
     */
    public abstract void reloadContextParameters() throws TopiaException;

}