/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2011 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixIterator;
import org.nuiton.math.matrix.MatrixND;

import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.PopulationGroup;
import fr.ifremer.isisfish.entities.PopulationSeasonInfo;
import fr.ifremer.isisfish.entities.Species;
import fr.ifremer.isisfish.entities.Zone;
import fr.ifremer.isisfish.types.TimeStep;
import fr.ifremer.isisfish.types.Month;
import java.util.Collections;

/**
 * Classe permettant le suivi des populations de la simulation.
 * <p>
 * Cette classe est normalement multi-thread safe
 * <p>
 * FIXME: certain calcul sont les memes que ceux implanter dans les scripts
 * a cause du groupe des juveniles qui n'est pas un vrai groupe de population.
 * Il serait bon que ce groupe deviennent un vrai groupe et que ce code specifique
 * puisse etre supprimer.
 *
 * @author poussin
 * @version $Revision: 428 $
 *
 * Last update: $Date: 2007-10-15 14:56:13 +0200 (lun, 15 oct 2007) $
 * by : $Author: bpoussin $
 */
public class PopulationMonitor {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory.getLog(PopulationMonitor.class);

    /** La liste des pops monitorer par cette instance */
    protected List<Population> pops = null;
    /** current number of fish by Pop */
    protected Map<Population, MatrixND> Ns = new HashMap<Population, MatrixND>();
    /** reproduction key: <date, pop> value:<MatrixND> */
    protected Map<Population, Map<TimeStep, MatrixND>> reproductions = new HashMap<Population, Map<TimeStep, MatrixND>>();
    /** discard key: <date, pop> value:<MatrixND> */
    protected Map<Population, Map<TimeStep, MatrixND>> discards = new HashMap<Population, Map<TimeStep, MatrixND>>();
    /** catch per population, only last catch is remaining */
    protected Map<Population, MatrixND> catchs = new HashMap<Population, MatrixND>();
    /** catch per population, only last catch is remaining */
    protected Map<Population, MatrixND> holdCatchs = new HashMap<Population, MatrixND>();

    protected double totalHoldCatch = 0;

    /**
     * Initialise le monitor pour l'ensemble de pop passe en parametre.
     * Normalement cet init est fait dans le thread principale de la simulation
     * et il n'y a pas de besoin de le synchroniser.
     *
     * Cet init permet d'avoir ensuite le maxime de variable en lecture, et donc
     * de permettre le multithreading de facon legere (pas de synchronise) et
     * sur (pas de concurent modification exception)
     *
     * @param pops
     */
    public void init(List<Population> pops) {
        this.pops = Collections
                .unmodifiableList(new ArrayList<Population>(pops));
        for (Population pop : this.pops) {
            reproductions.put(pop, new HashMap<TimeStep, MatrixND>());
            discards.put(pop, new HashMap<TimeStep, MatrixND>());
        }
    }

    /**
     * Return all population actualy in PopulationMonitor
     * @return new list of Population
     */
    public List<Population> getPopulations() {
        return pops;
    }

    /**
     * Return current biomass for species.
     * 
     * @param species species
     * @return species biomass
     */
    public double getBiomass(Species species) {
        double result = 0;

        for (Population pop : species.getPopulation()) {
            result += getBiomass(pop);
        }

        return result;
    }

    /**
     * Return current biomass for population.
     * 
     * @param pop population
     * @return population biomass
     */
    public double getBiomass(Population pop) {
        double result = 0;

        MatrixND n = getN(pop);
        if (n != null) {
            n = n.sumOverDim(1);
            for (MatrixIterator i = n.iterator(); i.next();) {
                Object[] coord = i.getSemanticsCoordinates();
                PopulationGroup group = (PopulationGroup) coord[0];
                result += i.getValue() * group.getMeanWeight();
            }
        }

        return result;
    }

    /**
     * Return current numbers for population.
     * 
     * @param pop population
     * @return population numbers
     */
    public MatrixND getN(Population pop) {
        MatrixND result = Ns.get(pop);

        if (result != null) {
            // change semantics with list from argument to ensure that
            // semantics don't used too old context
            result.setSemantic(0, pop.getPopulationGroup());
            result.setSemantic(1, pop.getPopulationZone());
        }

        return result;
    }

    public void setN(Population pop, MatrixND mat) {
        Ns.put(pop, mat);
    }

    /**
     * Return reproduction.
     * 
     * @param step step
     * @param pop population
     * @return the reproduction
     */
    public MatrixND getReproduction(TimeStep step, Population pop) {
        return this.reproductions.get(pop).get(step);
    }

    /**
     * Set new reproduction.
     * 
     * @param step step
     * @param pop population
     * @param repro reproduction to set
     */
    public void setReproduction(TimeStep step, Population pop, MatrixND repro) {
        reproductions.get(pop).put(step, repro);
    }

    /**
     * Applique de la mortalite naturelle aux poissons qui sont pas encore
     * dans les classes de populations. (Reproduction)
     * @param pop
     */
    public void applyReproductionMortality(Population pop) {
        for (MatrixND reproduction : reproductions.get(pop).values()) {
            if (log.isTraceEnabled()) {
                log.trace("Matrix repro before mortality: " + reproduction);
            }

            for (MatrixIterator mi = reproduction.iterator(); mi.next();) {
                Object[] sems = mi.getSemanticsCoordinates();
                Zone z = (Zone) sems[0];
                double coeff = pop.getNaturalDeathBirth(z);
                if (log.isTraceEnabled()) {
                    log.trace("NaturalDeath zone " + z + "=" + coeff);
                }
                mi.setValue(mi.getValue()
                        * Math.exp(-coeff / Month.NUMBER_OF_MONTH));
            }
            if (log.isTraceEnabled()) {
                log.trace("Matrix repro after mortality: " + reproduction);
            }
        }
    }

    /**
     * 
     * TODO les reproductions qui sont trop veille pour encore servir
     * doivent être supprimées. c-a-d date < currentDate - etalement - gap between repro recru
     * Une autre facon de faire est de supprimer les repro == 0 car normalement
     * toute la repro doit etre utilisé au bout d'un certain temps
     * 
     * @param step
     * @param pop
     * @return population recruitment
     */
    public MatrixND getRecruitment(TimeStep step, Population pop) {
        MatrixND matEtalement = pop.getRecruitmentDistribution();
        int etalement = matEtalement.getDim(0);

        MatrixND result = MatrixFactory.getInstance()
                .create(
                        new List[] { pop.getPopulationGroup(),
                                pop.getPopulationZone() });

        // pour chaque
        for (int e = 0; e < etalement; e++) {
            // recuperation de la reproduction stucture en zone repro
            TimeStep t = new TimeStep(step.getStep() - e
                    - pop.getMonthGapBetweenReproRecrutement());
            MatrixND repro = (MatrixND) reproductions.get(pop).get(t);

            if (repro != null) { // si une repro existe pour le mois

                // on fait la correspondance entre les zones repro et
                // recrutement

                PopulationGroup classe;
                int indiceClasse = 0;
                // si on a change d'annee le recrutement ne se fait pas en
                // classe 0 mais en classe 1, si on a change 2 fois d'annee
                // le recrutement se fait en age 2, etc.
                classe = pop.getPopulationGroup().get(indiceClasse);

                List<Zone> zoneRepros = pop.getReproductionZone();

                // on multiplie la repro par le coeff de recrutement
                double coeff = matEtalement.getValue(e);

                MatrixND matRepro = repro.copy();
                matRepro = matRepro.mults(coeff);

                matRepro.setSemantic(0, zoneRepros);

                MatrixND mapping = pop.getMappingZoneReproZoneRecru();
                for (Zone zoneRepro : zoneRepros) {
                    MatrixND submapping = mapping.getSubMatrix(0,
                            new Object[] { zoneRepro });
                    for (MatrixIterator i = submapping.iterator(); i.hasNext();) {
                        i.next();
                        Object[] sem = i.getSemanticsCoordinates();
                        Zone zoneRecru = (Zone) sem[1];
                        double c = i.getValue();
                        result.setValue(classe, zoneRecru, c
                                * matRepro.getValue(zoneRepro)
                                + result.getValue(classe, zoneRecru));
                    }
                }
            }
        }

        if (pop.getSpecies().getAgeGroupType()) {
            // conversion et retour de la matrice en vecteur
            MatrixND N = pop.N2DToN1D(result);
            // on applique les migrations et le changement d'age sur le resultat
            // on suppose que la reproduction est toujours dans une seul saison
            TimeStep stepRepro = new TimeStep(step.getStep()
                    - pop.getMonthGapBetweenReproRecrutement());

            // recherche les saisons des differents mois entre les deux dates
            List<PopulationSeasonInfo> infos = pop.getPopulationSeasonInfo();
            List<PopulationSeasonInfo> usedSeasons = new ArrayList<PopulationSeasonInfo>();

            while (step.after(stepRepro)) {
                stepRepro = stepRepro.next();
                Month month = stepRepro.getMonth();
                for (PopulationSeasonInfo info : infos) {
                    if (month.equals(info.getFirstMonth())) {
                        usedSeasons.add(info);
                        stepRepro = new TimeStep(stepRepro.getStep()
                                + info.getMonths().size() - 1); // -1 because, for have next()
                        break;
                    }
                }
            }

            for (PopulationSeasonInfo info : usedSeasons) {
                Month month = info.getFirstMonth();
                MatrixND CA = info.getGroupChangeMatrix(month);
                MatrixND MI = info.getMigrationMatrix(month, result);
                MatrixND tmp0 = N.mult(CA);
                MatrixND tmp2 = tmp0.mult(MI);
                N = tmp2;
            }
            result = pop.split2D(N);
        }
        return result;
    }

    /**
     * @param pop population
     * @param catchPerStrategyMet
     */
    public void holdCatch(Population pop, MatrixND catchPerStrategyMet) {
        catchs.put(pop, catchPerStrategyMet);

        MatrixND holdCatch = holdCatchs.get(pop);
        if (holdCatch == null) {
            holdCatch = MatrixFactory.getInstance().create(catchPerStrategyMet);
            holdCatchs.put(pop, holdCatch);
        } else {
            holdCatch.add(catchPerStrategyMet);
        }

        // compute total
        for (MatrixIterator i = catchPerStrategyMet.iterator(); i.next();) {
            this.totalHoldCatch += i.getValue();
        }

    }

    /**
     * Get population catch.
     * 
     * @param pop population
     * @return population catch
     */
    public MatrixND getCatch(Population pop) {
        MatrixND result = catchs.get(pop);
        return result;
    }

    /**
     * Get population hold catch.
     * 
     * @param pop population
     * @return population hold catch
     */
    public MatrixND getHoldCatch(Population pop) {
        MatrixND result = holdCatchs.get(pop);
        return result;
    }

    /**
     * Population total hold catch.
     * 
     * @param pop population
     * @return population total hold catch
     */
    public double getTotalHoldCatch(Population pop) {
        double result = totalHoldCatch;
        return result;
    }

    /**
     * RAZ capture cumulée de toutes les pops.
     */
    public void clearCatch() {
        catchs.clear();
        holdCatchs.clear();
        totalHoldCatch = 0;
    }

    /**
     * Get discard.
     * 
     * @param step step to get discard
     * @param pop population to get discard
     * @return le discard
     */
    public MatrixND getDiscard(TimeStep step, Population pop) {
        MatrixND result = discards.get(pop).get(step);
        return result;
    }

    /**
     * Set discard.
     * 
     * Force discard at date for population.
     * 
     * @param step step to set discard
     * @param pop population to set discard
     * @param discard le discard
     */
    public void setDiscard(TimeStep step, Population pop, MatrixND discard) {
        Map<TimeStep, MatrixND> oneDiscard = discards.get(pop);
        synchronized (oneDiscard) {
            // meme si discards.get(pop) != null
            // replace toujours le précédent
            MatrixND tmp = discard.copy();
            discards.get(pop).put(step, tmp);
        }
    }

    /**
     * FIXME: discards n'est jamais vider, on le conserve pour toutes les annees
     * il serait bon de faire un peu le menage de temps en temps.
     *
     * @param step
     * @param pop
     * @param discard
     */
    public void addDiscard(TimeStep step, Population pop, MatrixND discard) {
        Map<TimeStep, MatrixND> oneDiscard = discards.get(pop);
        synchronized (oneDiscard) {
            MatrixND tmp = oneDiscard.get(step);
            if (tmp == null) {
                tmp = discard.copy();
                discards.get(pop).put(step, tmp);
            } else {
                tmp.add(discard);
            }
        }
    }
}
