/*
 * #%L
 * IsisFish
 * 
 * $Id: IsisMapBean.java 3671 2012-04-03 13:31:36Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.map;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

import com.bbn.openmap.BufferedMapBean;
import com.bbn.openmap.Layer;
import com.bbn.openmap.LayerHandler;
import com.bbn.openmap.MouseDelegator;
import com.bbn.openmap.event.MapMouseListener;
import com.bbn.openmap.event.MapMouseMode;
import com.bbn.openmap.event.NavMouseMode;
import com.bbn.openmap.layer.GraticuleLayer;
import com.bbn.openmap.layer.dted.DTEDLayer;
import com.bbn.openmap.layer.e00.E00Layer;
import com.bbn.openmap.layer.mif.MIFLayer;
import com.bbn.openmap.layer.rpf.RpfLayer;
import com.bbn.openmap.layer.shape.ShapeLayer;
import com.bbn.openmap.layer.vpf.VPFLayer;
import com.bbn.openmap.omGraphics.DrawingAttributes;
import com.bbn.openmap.proj.coords.LatLonPoint;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.entities.Cell;
import fr.ifremer.isisfish.entities.CellDAO;
import fr.ifremer.isisfish.entities.FisheryRegion;

/**
 * IsisMapBean.java
 *
 * Created: 16 mai 2005
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 3671 $
 *
 * Mise a jour: $Date: 2012-04-03 15:31:36 +0200 (Tue, 03 Apr 2012) $
 * par : $Author: echatellier $
 */
public class IsisMapBean extends BufferedMapBean {

    /** serialVersionUID. */
    private static final long serialVersionUID = -4162103384132928473L;

    /** to use log facility, just put in your code: log.info("..."); */
    private static Log log = LogFactory.getLog(IsisMapBean.class);

    protected FisheryRegion fisheryRegion = null;
    protected float pasMailleLatitude;
    protected float pasMailleLongitude;
    protected float minLatitude;
    protected float maxLatitude;
    protected float minLongitude;
    protected float maxLongitude;

    protected List<MapMouseListener> orderedListener = new ArrayList<MapMouseListener>();
    protected LayerHandler layerHandler = new LayerHandler();
    protected MapMouseMode currentMouseMode;
    protected MouseDelegator md;

    protected CellSelectionLayer activeSelectionLayer = null;
    protected int selectionMode = CellSelectionLayer.MULT_SELECTION;

    protected static final float SCALE = 9500000f;

    public IsisMapBean() {
        init();
        setActiveMouseMode(new NavMouseMode());
    }

    protected void init() {
        setScale(SCALE);
        // Let the LayerHandler know who is interested in Layers.
        //MapBean is just one component
        layerHandler.addLayerListener(this);

        md = new MouseDelegator(this);
        // md.addMouseMode(selectMouseMode);
        // Tell the delegator to use the default modes: Navigation
        // and Selection
        md.setDefaultMouseModes();
    }

    public void setActiveMouseMode(MapMouseMode mode) {
        md.setActiveMouseMode(mode);
        currentMouseMode = mode;
        // il faut remettre les listeners
        for (MapMouseListener l : orderedListener) {
            currentMouseMode.addMapMouseListener(l);
        }
    }

    public void addMapMouseListener(MapMouseListener mml) {
        if (mml instanceof CellSelectionLayer || mml instanceof ResultatLayer) {
            // faut que certain listener soit toujours en premier
            orderedListener.add(0, mml);
            currentMouseMode.removeAllMapMouseListeners();
            for (MapMouseListener l : orderedListener) {
                currentMouseMode.addMapMouseListener(l);
            }
        } else {
            orderedListener.add(mml);
            currentMouseMode.addMapMouseListener(mml);
        }
    }

    public void removeMapMouseListener(MapMouseListener mml) {
        currentMouseMode.removeMapMouseListener(mml);
    }

    /**
     * Get the value of region.
     * @return value of region.
     */
    public FisheryRegion getFisheryRegion() {
        return fisheryRegion;
    }

    public void setFisheryRegion(FisheryRegion fisheryRegion) {
        if (log.isDebugEnabled()) {
            log.debug("current fishery is now: " + fisheryRegion + " old was: "
                    + this.fisheryRegion);
        }

        try {
            FisheryRegion oldFisheryRegion = this.fisheryRegion;
            this.fisheryRegion = fisheryRegion;
            if (fisheryRegion == null) {
                layerHandler.removeAll();
            } else {
                if (!fisheryRegion.equals(oldFisheryRegion)) {
                    pasMailleLatitude = fisheryRegion.getCellLengthLatitude();
                    pasMailleLongitude = fisheryRegion.getCellLengthLongitude();
                    minLatitude = fisheryRegion.getMinLatitude();
                    minLongitude = fisheryRegion.getMinLongitude();
                    maxLatitude = fisheryRegion.getMaxLatitude();
                    maxLongitude = fisheryRegion.getMaxLongitude();
                    initMap();
                }
            }
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error("Impossible d'initialiser la Region", eee);
            }
        }
    }

    protected TopiaContext getTopiaContext() {
        TopiaContext result = getFisheryRegion().getTopiaContext();
        if (result == null) {
            throw new IsisFishRuntimeException(
                    "Can't get topiaContext from FisheryRegion");
        }
        return result;
    }

    protected CellDAO getCellDAO() throws TopiaException {
        return IsisFishDAOHelper.getCellDAO(getTopiaContext());
    }

    /**
     * Retourne la liste des mailles selectionnées.
     * 
     * @return selected map list
     */
    public List<Cell> getSelectedCells() {
        try {
            List<Cell> result = new ArrayList<Cell>();
            List<LatLonPoint> pts = activeSelectionLayer.getSelected();
            for (LatLonPoint llp : pts) {
                List<Cell> cells = null;
                try {
                    cells = getCellDAO().findAllByProperties("latitude",
                            llp.getLatitude(), "longitude", llp.getLongitude());
                } catch (TopiaException eee) {
                    log.warn("Can't find cell for this point: " + llp, eee);
                }
                if (cells != null && cells.size() > 0) {
                    result.add(cells.get(0));
                }
            }
            return result;
        } catch (RuntimeException eee) {
            if (log.isErrorEnabled()) {
                log.error("Can't find selected cells", eee);
            }
            throw eee;
        }
    }

    public void setSelectedCells(List<Cell> cells) {
        if (cells != null) {
            setSelectedCells(cells.toArray(new Cell[cells.size()]));
        } else {
            setSelectedCells();
        }
    }

    public void setSelectedCells(Cell... cells) {
        if (log.isDebugEnabled()) {
            log.debug("Select cells: " + Arrays.toString(cells));
        }
        activeSelectionLayer.unSelectAll();
        addSelectedCells(cells);
    }

    public void addSelectedCells(Cell... cells) {
        if (log.isDebugEnabled()) {
            log.debug("add select cells: " + Arrays.toString(cells));
        }
        for (Cell cell : cells) {
            if (cell != null) {
                activeSelectionLayer.select((cell.getLatitude()
                        + cell.getLatitude() + pasMailleLatitude) / 2f, (cell
                        .getLongitude()
                        + cell.getLongitude() + pasMailleLongitude) / 2f);
            }
        }
    }

    public void removeSelectedCells(Cell... cells) {
        if (log.isDebugEnabled()) {
            log.debug("remove select cells: " + Arrays.toString(cells));
        }
        for (Cell cell : cells) {
            activeSelectionLayer.unSelect((cell.getLatitude()
                    + cell.getLatitude() + pasMailleLatitude) / 2f, (cell
                    .getLongitude()
                    + cell.getLongitude() + pasMailleLongitude) / 2f);
        }
    }

    /**
     * Get the value of layerHandler.
     * @return value of layerHandler.
     */
    public LayerHandler getLayerHandler() {
        return layerHandler;
    }

    public void addResultatLayer(String id, ResultatLayer layer) {
        layer.setName(id);
        addMapMouseListener(layer);
        layerHandler.addLayer(layer);
    }

    public void removeAllResultatLayer() {
        Layer[] layers = layerHandler.getLayers();
        for (int i = 0; i < layers.length; i++) {
            if (layers[i] instanceof ResultatLayer) {
                removeMapMouseListener((MapMouseListener) layers[i]);
                layerHandler.removeLayer(layers[i]);
            }
        }
    }

    /**
     * Add a new layer to the map depending on mapFile extension.
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     */
    protected void addLayer(String layerId, String mapFile, String lineColor,
            String fillColor) {

        Layer layer = null;
        // get layer depending on type
        if (mapFile.endsWith(".shp")) {
            layer = getShapeLayer(layerId, mapFile, lineColor, fillColor);
        }
        else if (mapFile.endsWith(".e00")) {
            layer = getE00Layer(layerId, mapFile, lineColor, fillColor);
        }
        else if (mapFile.endsWith(".mif")) {
            layer = getMIFLayer(layerId, mapFile, lineColor, fillColor);
        }
        else if (mapFile.endsWith(".rpf") || mapFile.endsWith(".cadrg") || mapFile.endsWith(".cib")) {
            layer = getRPFLayer(layerId, mapFile, lineColor, fillColor);
        }
        else if (mapFile.endsWith(".vmap") || mapFile.endsWith(".dcw") || mapFile.endsWith(".vpf")) {
            layer = getVPFLayer(layerId, mapFile, lineColor, fillColor);
        }
        /*else if (mapFile.endsWith(".dt0")) {
            layer = getDTEDLayer(layerId, mapFile, lineColor, fillColor);
        }*/
        else {
            if (log.isErrorEnabled()) {
                log.error("Can't find layer for " + mapFile + " (unknown type)");
            }
        }

        // display layer
        if (layer != null) {
            if (log.isDebugEnabled()) {
                log.debug("Add layer " + layer);
            }
            layerHandler.addLayer(layer);
        }
    }

    /**
     * Manage shp layer display.
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     * 
     * @return shape layer
     * @see ShapeLayer
     */
    protected Layer getShapeLayer(String layerId, String mapFile, String lineColor,
            String fillColor) {

        if (log.isDebugEnabled()) {
            log.debug("Get ShapeLayer with filename : " + mapFile);
        }

        ShapeLayer shapeLayer = new ShapeLayer();
        Properties p = new Properties();
        p.setProperty(layerId + "." + ShapeLayer.shapeFileProperty, mapFile);
        p.setProperty(layerId + "." + DrawingAttributes.linePaintProperty, lineColor);
        p.setProperty(layerId + "." + DrawingAttributes.fillPaintProperty, fillColor);
        shapeLayer.setProperties(layerId, p);
        shapeLayer.setName(layerId);
        
        return shapeLayer;
    }

    /**
     * Manage E00 layer display.
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     * 
     * @return e00 layer
     * @see E00Layer
     */
    protected Layer getE00Layer(String layerId, String mapFile, String lineColor,
            String fillColor) {
        if (log.isDebugEnabled()) {
            log.debug("Get E00Layer with filename : " + mapFile);
        }
        E00Layer e00Layer = new E00Layer();
        Properties p = new Properties();
        p.setProperty(layerId + ".FileName", mapFile);
        p.setProperty(layerId + "." + DrawingAttributes.linePaintProperty, lineColor);
        p.setProperty(layerId + "." + DrawingAttributes.fillPaintProperty, fillColor);
        e00Layer.setProperties(layerId, p);
        e00Layer.setName(layerId);
        
        return e00Layer;
    }

    /**
     * Manage DTED layer display.
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     * 
     * @return dted layer
     * @see DTEDLayer
     */
    protected Layer getDTEDLayer(String layerId, String mapFile, String lineColor,
            String fillColor) {
        if (log.isDebugEnabled()) {
            log.debug("Get DTEDLayer with filename : " + mapFile);
        }

        // get current map parent file
        File mapFileFile = new File(mapFile);
        String parentFolder = mapFileFile.getParent();

        DTEDLayer dtedLayer = new DTEDLayer();
        Properties p = new Properties();
        p.setProperty(layerId + ".dted.paths", parentFolder);
        p.setProperty(layerId + "." + DrawingAttributes.linePaintProperty, lineColor);
        p.setProperty(layerId + "." + DrawingAttributes.fillPaintProperty, fillColor);
        
        dtedLayer.setProperties(layerId, p);
        dtedLayer.setName(layerId);
        
        return dtedLayer;
    }
    
    /**
     * Manage MIF layer display.
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     * 
     * @return mif layer
     * @see MIFLayer
     */
    protected Layer getMIFLayer(String layerId, String mapFile, String lineColor,
            String fillColor) {
        
        if (log.isDebugEnabled()) {
            log.debug("Get MIFLayer with filename : " + mapFile);
        }

        MIFLayer mifLayer = new MIFLayer();
        Properties p = new Properties();
        p.setProperty(layerId + "." + MIFLayer.MIF_FileProperty, mapFile);
        p.setProperty(layerId + "." + MIFLayer.pointVisibleProperty, "true");
        p.setProperty(layerId + "." + DrawingAttributes.linePaintProperty, lineColor);
        p.setProperty(layerId + "." + DrawingAttributes.fillPaintProperty, fillColor);
        
        mifLayer.setProperties(layerId, p);
        mifLayer.setName(layerId);
        
        return mifLayer;
        
    }
    
    /**
     * Manage RPF layer display.
     * 
     * Seams to handle cadrg and cib files too.
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     * 
     * @return rpf layer
     * @see RpfLayer
     */
    protected Layer getRPFLayer(String layerId, String mapFile, String lineColor,
            String fillColor) {
        
        if (log.isDebugEnabled()) {
            log.debug("Get RpfLayer with filename : " + mapFile);
        }

        // get current map parent file
        File mapFileFile = new File(mapFile);
        String parentFolder = mapFileFile.getParent();
        
        RpfLayer rpfLayer = new RpfLayer();
        Properties p = new Properties();
        // This property should reflect the paths to the RPF directories
        p.setProperty(layerId + ".rpf.paths", parentFolder);
        p.setProperty(layerId + "." + MIFLayer.pointVisibleProperty, "true");
        p.setProperty(layerId + "." + DrawingAttributes.linePaintProperty, lineColor);
        p.setProperty(layerId + "." + DrawingAttributes.fillPaintProperty, fillColor);
        
        rpfLayer.setProperties(layerId, p);
        rpfLayer.setName(layerId);
        
        return rpfLayer;
        
    }
    
    /**
     * Manage VPF layer display.
     * 
     * Can manage .vmap, .dcw, .vpf files ?
     * 
     * @param layerId layer id
     * @param mapFile absolute map fail path
     * @param lineColor line color
     * @param fillColor fill color
     * 
     * @return vpf layer
     * @see VPFLayer
     */
    protected Layer getVPFLayer(String layerId, String mapFile, String lineColor,
            String fillColor) {
        
        // on presume que, dans le cas d'IsisFish, les cartes VMap ne
        // sont pas politiques.

        if (log.isDebugEnabled()) {
            log.debug("Get vpfLayer with filename : " + mapFile);
        }

        VPFLayer vpfLayer = new VPFLayer();
        Properties p = new Properties();
        p.setProperty(layerId + "." + VPFLayer.pathProperty, mapFile);
        p.setProperty(layerId + "." + VPFLayer.defaultLayerProperty, "vmapCoastline");
        p.setProperty(layerId + "." + DrawingAttributes.linePaintProperty, lineColor);
        p.setProperty(layerId + "." + DrawingAttributes.fillPaintProperty, fillColor);
        
        vpfLayer.setProperties(layerId, p);
        vpfLayer.setName(layerId);
        
        return vpfLayer;
        
    }

    protected void addGraticuleLayer() {
        GraticuleLayer layer = new GraticuleLayer();
        Properties p = new Properties();
        // Show lat / lon spacing labels
        p.setProperty("." + GraticuleLayer.ShowRulerProperty, "true");
        p.setProperty("." + GraticuleLayer.ShowOneAndFiveProperty, "true");
        // Controls when the five degree lines and one degree lines kick in
        // - when there is less than the threshold of ten degree lat or lon
        // lines, five degree lines are drawn.  The same relationship is there
        // for one to five degree lines.
        p.setProperty("." + GraticuleLayer.ThresholdProperty, "5");
        // the color of 10 degree spacing lines (Hex ARGB)
        p.setProperty("." + GraticuleLayer.TenDegreeColorProperty, "FF000000");
        // the color of 5 degree spacing lines (Hex ARGB)
        p.setProperty("." + GraticuleLayer.FiveDegreeColorProperty, "C7009900");
        // the color of 1 degree spacing lines (ARGB)
        p.setProperty("." + GraticuleLayer.OneDegreeColorProperty, "FF003300");
        // the color of the equator (ARGB)
        p.setProperty("." + GraticuleLayer.EquatorColorProperty, "FFFF0000");
        // the color of the international dateline (ARGB)
        p.setProperty("." + GraticuleLayer.DateLineColorProperty, "FF000099");
        // the color of the special lines (ARGB) (Tropic of Cancer, Capricorn)
        p.setProperty("." + GraticuleLayer.SpecialLineColorProperty, "FF000000");
        // the color of the labels (ARGB)
        p.setProperty("." + GraticuleLayer.TextColorProperty, "FF000000");

        layer.setProperties("", p);
        layerHandler.addLayer(layer);
    }

    protected void addZoneDelimiterLayer() throws TopiaException {
        ZoneDelimiterLayer layer = new ZoneDelimiterLayer(fisheryRegion);
        layerHandler.addLayer(layer);
    }

    protected void addSpecificLayer() {
        try {
            addSelectionLayer();
            addZoneDelimiterLayer();
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't add specific layer", eee);
            }
        }
    }

    /**
     * init the mServer property value.
     */
    protected void initMap() {
        try {
            float centerLat = (maxLatitude + minLatitude) / 2f;
            float centerLong = (maxLongitude + minLongitude) / 2f;
            layerHandler.removeAll();
            //setProjection( new Mercator(new LatLonPoint(centerLat, centerLong), SCALE, 480, 540));

            //FIXME requiered for DTED datas
            //setProjection(new LLXY(new LatLonPoint(centerLat, centerLong), SCALE, 480, 540));
            // OR this one
            //setProjection(new CADRG(new LatLonPoint(centerLat, centerLong), SCALE, 480, 540));

            setCenter(new LatLonPoint.Float(centerLat, centerLong));
            //setScale(SCALE);
            addSpecificLayer();
            addGraticuleLayer();

            // ajout des shapes
            boolean shapeLoaded = false;
            for (String filename : getFisheryRegion().getMapFilePath()) {
                if (!StringUtils.isEmpty(filename)) {
                    if (log.isDebugEnabled()) {
                        log.debug(_("isisfish.message.load.map", filename));
                    }
                    addLayer(filename, filename, "ff000000", "ffbdde83");
                    shapeLoaded = true;
                }
            }

            if (!shapeLoaded) {
                if (log.isDebugEnabled()) {
                    log.debug("Can't load custom map, load default one");
                }
                // a pas reussi a charger les fichiers demandés, on charge la
                // carte du monde
                String filename = IsisFish.config.getDefaultMapFilename();
                addLayer(filename, filename, "ff000000", "ffbdde83");
            }

        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("isisfish.error.init.map"), eee);
            }
        }
    }

    /**
     * Get the value of selectionMode.
     * @return value of selectionMode.
     */
    public int getSelectionMode() {
        return selectionMode;
    }

    /**
     * Set the value of selectionMode.
     * @param v value to assign to selectionMode.
     */
    public void setSelectionMode(int v) {
        this.selectionMode = v;
        
        if (activeSelectionLayer != null) {
            activeSelectionLayer.setSelectionMode(v);
        }
    }

    public void addSelectionLayer() {
        try {
            activeSelectionLayer = new CellSelectionLayer(fisheryRegion, getSelectionMode());
            addMapMouseListener(activeSelectionLayer);
            layerHandler.addLayer(activeSelectionLayer);
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn("Can't add selection layer", eee);
            }
        }
    }

} // IsisMapBean
