/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.rds.waiters;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.rds.RdsClient;
import software.amazon.awssdk.services.rds.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.rds.model.DescribeDbClusterSnapshotsRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbClusterSnapshotsResponse;
import software.amazon.awssdk.services.rds.model.DescribeDbClustersRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbClustersResponse;
import software.amazon.awssdk.services.rds.model.DescribeDbInstancesRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbInstancesResponse;
import software.amazon.awssdk.services.rds.model.DescribeDbSnapshotsRequest;
import software.amazon.awssdk.services.rds.model.DescribeDbSnapshotsResponse;
import software.amazon.awssdk.services.rds.model.DescribeTenantDatabasesRequest;
import software.amazon.awssdk.services.rds.model.DescribeTenantDatabasesResponse;
import software.amazon.awssdk.services.rds.model.RdsRequest;
import software.amazon.awssdk.services.rds.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultRdsWaiter implements RdsWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final RdsClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeDbInstancesResponse> dBInstanceAvailableWaiter;

    private final Waiter<DescribeDbInstancesResponse> dBInstanceDeletedWaiter;

    private final Waiter<DescribeDbSnapshotsResponse> dBSnapshotAvailableWaiter;

    private final Waiter<DescribeDbSnapshotsResponse> dBSnapshotDeletedWaiter;

    private final Waiter<DescribeDbClusterSnapshotsResponse> dBClusterSnapshotAvailableWaiter;

    private final Waiter<DescribeDbClusterSnapshotsResponse> dBClusterSnapshotDeletedWaiter;

    private final Waiter<DescribeDbClustersResponse> dBClusterAvailableWaiter;

    private final Waiter<DescribeDbClustersResponse> dBClusterDeletedWaiter;

    private final Waiter<DescribeTenantDatabasesResponse> tenantDatabaseAvailableWaiter;

    private final Waiter<DescribeTenantDatabasesResponse> tenantDatabaseDeletedWaiter;

    private DefaultRdsWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = RdsClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.dBInstanceAvailableWaiter = Waiter.builder(DescribeDbInstancesResponse.class)
                .acceptors(dBInstanceAvailableWaiterAcceptors())
                .overrideConfiguration(dBInstanceAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.dBInstanceDeletedWaiter = Waiter.builder(DescribeDbInstancesResponse.class)
                .acceptors(dBInstanceDeletedWaiterAcceptors())
                .overrideConfiguration(dBInstanceDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.dBSnapshotAvailableWaiter = Waiter.builder(DescribeDbSnapshotsResponse.class)
                .acceptors(dBSnapshotAvailableWaiterAcceptors())
                .overrideConfiguration(dBSnapshotAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.dBSnapshotDeletedWaiter = Waiter.builder(DescribeDbSnapshotsResponse.class)
                .acceptors(dBSnapshotDeletedWaiterAcceptors())
                .overrideConfiguration(dBSnapshotDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.dBClusterSnapshotAvailableWaiter = Waiter.builder(DescribeDbClusterSnapshotsResponse.class)
                .acceptors(dBClusterSnapshotAvailableWaiterAcceptors())
                .overrideConfiguration(dBClusterSnapshotAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.dBClusterSnapshotDeletedWaiter = Waiter.builder(DescribeDbClusterSnapshotsResponse.class)
                .acceptors(dBClusterSnapshotDeletedWaiterAcceptors())
                .overrideConfiguration(dBClusterSnapshotDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.dBClusterAvailableWaiter = Waiter.builder(DescribeDbClustersResponse.class)
                .acceptors(dBClusterAvailableWaiterAcceptors())
                .overrideConfiguration(dBClusterAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.dBClusterDeletedWaiter = Waiter.builder(DescribeDbClustersResponse.class)
                .acceptors(dBClusterDeletedWaiterAcceptors())
                .overrideConfiguration(dBClusterDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.tenantDatabaseAvailableWaiter = Waiter.builder(DescribeTenantDatabasesResponse.class)
                .acceptors(tenantDatabaseAvailableWaiterAcceptors())
                .overrideConfiguration(tenantDatabaseAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.tenantDatabaseDeletedWaiter = Waiter.builder(DescribeTenantDatabasesResponse.class)
                .acceptors(tenantDatabaseDeletedWaiterAcceptors())
                .overrideConfiguration(tenantDatabaseDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeDbClustersResponse> waitUntilDBClusterAvailable(
            DescribeDbClustersRequest describeDbClustersRequest) {
        return dBClusterAvailableWaiter.run(() -> client.describeDBClusters(applyWaitersUserAgent(describeDbClustersRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbClustersResponse> waitUntilDBClusterAvailable(
            DescribeDbClustersRequest describeDbClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBClusterAvailableWaiter.run(() -> client.describeDBClusters(applyWaitersUserAgent(describeDbClustersRequest)),
                dBClusterAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbClustersResponse> waitUntilDBClusterDeleted(
            DescribeDbClustersRequest describeDbClustersRequest) {
        return dBClusterDeletedWaiter.run(() -> client.describeDBClusters(applyWaitersUserAgent(describeDbClustersRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbClustersResponse> waitUntilDBClusterDeleted(
            DescribeDbClustersRequest describeDbClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBClusterDeletedWaiter.run(() -> client.describeDBClusters(applyWaitersUserAgent(describeDbClustersRequest)),
                dBClusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbClusterSnapshotsResponse> waitUntilDBClusterSnapshotAvailable(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest) {
        return dBClusterSnapshotAvailableWaiter.run(() -> client
                .describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbClusterSnapshotsResponse> waitUntilDBClusterSnapshotAvailable(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBClusterSnapshotAvailableWaiter.run(
                () -> client.describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)),
                dBClusterSnapshotAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbClusterSnapshotsResponse> waitUntilDBClusterSnapshotDeleted(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest) {
        return dBClusterSnapshotDeletedWaiter.run(() -> client
                .describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbClusterSnapshotsResponse> waitUntilDBClusterSnapshotDeleted(
            DescribeDbClusterSnapshotsRequest describeDbClusterSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBClusterSnapshotDeletedWaiter.run(
                () -> client.describeDBClusterSnapshots(applyWaitersUserAgent(describeDbClusterSnapshotsRequest)),
                dBClusterSnapshotDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceAvailable(
            DescribeDbInstancesRequest describeDbInstancesRequest) {
        return dBInstanceAvailableWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceAvailable(
            DescribeDbInstancesRequest describeDbInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBInstanceAvailableWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)),
                dBInstanceAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceDeleted(
            DescribeDbInstancesRequest describeDbInstancesRequest) {
        return dBInstanceDeletedWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbInstancesResponse> waitUntilDBInstanceDeleted(
            DescribeDbInstancesRequest describeDbInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBInstanceDeletedWaiter.run(() -> client.describeDBInstances(applyWaitersUserAgent(describeDbInstancesRequest)),
                dBInstanceDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbSnapshotsResponse> waitUntilDBSnapshotAvailable(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest) {
        return dBSnapshotAvailableWaiter.run(() -> client.describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbSnapshotsResponse> waitUntilDBSnapshotAvailable(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBSnapshotAvailableWaiter.run(() -> client.describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)),
                dBSnapshotAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeDbSnapshotsResponse> waitUntilDBSnapshotDeleted(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest) {
        return dBSnapshotDeletedWaiter.run(() -> client.describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)));
    }

    @Override
    public WaiterResponse<DescribeDbSnapshotsResponse> waitUntilDBSnapshotDeleted(
            DescribeDbSnapshotsRequest describeDbSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return dBSnapshotDeletedWaiter.run(() -> client.describeDBSnapshots(applyWaitersUserAgent(describeDbSnapshotsRequest)),
                dBSnapshotDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTenantDatabasesResponse> waitUntilTenantDatabaseAvailable(
            DescribeTenantDatabasesRequest describeTenantDatabasesRequest) {
        return tenantDatabaseAvailableWaiter.run(() -> client
                .describeTenantDatabases(applyWaitersUserAgent(describeTenantDatabasesRequest)));
    }

    @Override
    public WaiterResponse<DescribeTenantDatabasesResponse> waitUntilTenantDatabaseAvailable(
            DescribeTenantDatabasesRequest describeTenantDatabasesRequest, WaiterOverrideConfiguration overrideConfig) {
        return tenantDatabaseAvailableWaiter.run(
                () -> client.describeTenantDatabases(applyWaitersUserAgent(describeTenantDatabasesRequest)),
                tenantDatabaseAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTenantDatabasesResponse> waitUntilTenantDatabaseDeleted(
            DescribeTenantDatabasesRequest describeTenantDatabasesRequest) {
        return tenantDatabaseDeletedWaiter.run(() -> client
                .describeTenantDatabases(applyWaitersUserAgent(describeTenantDatabasesRequest)));
    }

    @Override
    public WaiterResponse<DescribeTenantDatabasesResponse> waitUntilTenantDatabaseDeleted(
            DescribeTenantDatabasesRequest describeTenantDatabasesRequest, WaiterOverrideConfiguration overrideConfig) {
        return tenantDatabaseDeletedWaiter.run(
                () -> client.describeTenantDatabases(applyWaitersUserAgent(describeTenantDatabasesRequest)),
                tenantDatabaseDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeDbInstancesResponse>> dBInstanceAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=incompatible-restore) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=incompatible-parameters) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbInstancesResponse>> dBInstanceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("DBInstances").length().compare("==", input.constant(new BigDecimal("0"))).value(),
                    true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBInstanceNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBInstances").flatten().field("DBInstanceStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBInstances[].DBInstanceStatus=resetting-master-credentials) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> dBSnapshotAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=incompatible-restore) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=incompatible-parameters) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> dBSnapshotDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("DBSnapshots").length().compare("==", input.constant(new BigDecimal("0"))).value(),
                    true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBSnapshotNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBSnapshots[].Status=resetting-master-credentials) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> dBClusterSnapshotAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=incompatible-restore) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=incompatible-parameters) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> dBClusterSnapshotDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbClusterSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("DBClusterSnapshots").length().compare("==", input.constant(new BigDecimal("0")))
                    .value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "DBClusterSnapshotNotFoundFault")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusterSnapshots").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusterSnapshots[].Status=resetting-master-credentials) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbClustersResponse>> dBClusterAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbClustersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=incompatible-restore) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=incompatible-parameters) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeDbClustersResponse>> dBClusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeDbClustersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("DBClusters").length().compare("==", input.constant(new BigDecimal("0"))).value(),
                    true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBClusterNotFoundFault")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "rebooting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=rebooting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("DBClusters").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "resetting-master-credentials"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (DBClusters[].Status=resetting-master-credentials) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTenantDatabasesResponse>> tenantDatabaseAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTenantDatabasesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("TenantDatabases").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("TenantDatabases").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (TenantDatabases[].Status=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("TenantDatabases").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-parameters"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (TenantDatabases[].Status=incompatible-parameters) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("TenantDatabases").flatten().field("Status").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-restore"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (TenantDatabases[].Status=incompatible-restore) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTenantDatabasesResponse>> tenantDatabaseDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTenantDatabasesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("TenantDatabases").length().compare("==", input.constant(new BigDecimal("0")))
                    .value(), true);
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "DBInstanceNotFoundFault")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration dBInstanceAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBInstanceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBSnapshotAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBSnapshotDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBClusterSnapshotAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBClusterSnapshotDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBClusterAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration dBClusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration tenantDatabaseAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration tenantDatabaseDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static RdsWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends RdsRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements RdsWaiter.Builder {
        private RdsClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public RdsWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public RdsWaiter.Builder client(RdsClient client) {
            this.client = client;
            return this;
        }

        public RdsWaiter build() {
            return new DefaultRdsWaiter(this);
        }
    }
}
