/* *##%
 * Copyright (c) 2009 ruchaud. All rights reserved.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty.jms;

import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_CLEAR;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_DELETE;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.ACTION_STORE;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.JNDI_PROVIDER_URL;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.PARAM_DISABLE_AUTO_VERSION_INCREMENT;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.PARAM_IDS;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.PARAM_WIKITTIES;
import static org.sharengo.wikitty.jms.WikittyJMSUtil.TOPIC_WIKITTY_STORAGE;

import java.util.ArrayList;
import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.NamingException;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.command.ActiveMQTopic;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyStorage;
import org.sharengo.wikitty.WikittyTransaction;

/**
 * WikittyStorageSubscriberJMS.
 *
 * @author $Author: echatellier $
 * @version $Revision: 1 $ $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * @since 11 févr. 2010 11:07:30
 */
public class WikittyStorageSubscriberJMS implements MessageListener {
    private static final Log log = LogFactory.getLog(WikittyStorageSubscriberJMS.class);

    protected final TopicConnectionFactory connectionFactory;
    protected final TopicConnection topicConnection;
    protected final Topic topic;

    protected final TopicSession topicSession;
    protected final TopicSubscriber topicSubscriber;
    
    protected final WikittyStorage delegateStorage;
    
    /**
     * Creates a durable subscriber on a topic named <code>WikittyStorageTopic</code>.
     * All received messages are dispatched to the specified <code>delegateStorage</code>.
     * 
     * @param delegateStorage an instance of <code>WikittyStorage</code>
     * @param properties properties passed to <code>InitialContext</code>,
     *         it must contains at least a value for the <code>Context.PROVIDER_URL</code>
     * @throws NamingException
     * @throws JMSException
     */
    public WikittyStorageSubscriberJMS(WikittyStorage delegateStorage, Properties properties)
            throws NamingException, JMSException {
        
        if (delegateStorage == null) {
            throw new IllegalArgumentException("delegateStorage is null");
        }
        this.delegateStorage = delegateStorage;
        
        topic = new ActiveMQTopic(TOPIC_WIKITTY_STORAGE);

        String provider = properties.getProperty(JNDI_PROVIDER_URL);
        connectionFactory = new ActiveMQConnectionFactory(provider);
        
        topicConnection = connectionFactory.createTopicConnection();
        topicConnection.setClientID("WikittyStorageSubscriberJMS");
        topicSession = topicConnection.createTopicSession(true, Session.SESSION_TRANSACTED);
        
        topicSubscriber = topicSession.createDurableSubscriber(topic, "WikittyStorageSubscriberJMS");
        topicSubscriber.setMessageListener(this);
    }
    
    /*
     * Before your application can consume messages,
     * you must call the connection start method
     */
    public void start() throws JMSException {
        topicConnection.start();
    }
    
    /*
     * If you want to stop message delivery temporarily
     * without closing the connection, you call the stop method.
     */
    public void stop() throws JMSException {
        topicConnection.stop();
    }
    
    /*
     * Closing a connection also closes its sessions and
     * their message producers and message consumers.
     */
    public void close() throws JMSException {
        topicConnection.close();
    }
    
    @Override
    protected void finalize() throws Throwable {
        topicConnection.close();
    }
    
    /*
     * Your onMessage method should handle all exceptions. It must not throw
     * checked exceptions, and throwing a RuntimeException is considered a programming error.
     */
    /**
     * Forwards the message to the <code>delegateStorage</code> according
     * to its type (TextMessage or ObjectMessage) and its client properties.
     * More precisely, the <code>action</code> client property determines the delegateStorage's method
     * which will be invoked.
     * 
     * @param message the incoming message
     */
    @Override
    public void onMessage(Message message) {
        final WikittyTransaction transaction = new WikittyTransaction();
        boolean sessionCommit = false;
        
        try {
            transaction.begin();
            
            if (log.isDebugEnabled()) {
                WikittyJMSUtil.logMessageHeaders(log, message);
            }
            
            if (message instanceof ObjectMessage) {
                WikittyActionMessage actionMessage =
                        WikittyActionMessage.createReceiveMessage((ObjectMessage) message);
                onActionMessage(transaction, actionMessage);

            } else {
                throw new WikittyException("Unsupported message: " +
                        message.getClass().getName());
            }
            
            transaction.commit();
            sessionCommit = true;
            
        } catch (Exception e) {
            log.error("Exception while message dispatch", e);
            
            try {
                transaction.rollback();
                
            } catch (Exception e1) {
                log.error("Exception while transaction rollback", e1);
            }
            
            try {
                topicSession.rollback();
                
            } catch (JMSException e1) {
                log.error("JMSException while session rollback", e1);
            }
        }
    
        if (sessionCommit) {
            try {
                topicSession.commit();
            } catch (JMSException e) {
                log.error("JMSException while session commit", e);
            }
        }
    }
    
    protected void onActionMessage(WikittyTransaction transaction, WikittyActionMessage message) {
        try {
            String action = message.getAction();
            
            if (ACTION_STORE.equals(action)) {
                actionStore(transaction, message);
                
            } else if (ACTION_CLEAR.equals(action)) {
                actionClear(transaction, message);
                    
            } else if (ACTION_DELETE.equals(action)) {
                actionDelete(transaction, message);
                
            } else {
                throw new UnsupportedOperationException("Unsupported action: " + action);
            }
            
        } catch (Exception e) {
            throw new WikittyException(e);
        }
    }
    
    protected void actionStore(WikittyTransaction transaction, WikittyActionMessage message) throws JMSException {
        ArrayList<Wikitty> wikitties =
                (ArrayList<Wikitty>) message.getParameter(PARAM_WIKITTIES);

        Boolean disableAutoVersionIncrement =
                (Boolean) message.getParameter(PARAM_DISABLE_AUTO_VERSION_INCREMENT);

        delegateStorage.store(transaction, wikitties, disableAutoVersionIncrement);
    }
    
    protected void actionClear(WikittyTransaction transaction, WikittyActionMessage message) {
        delegateStorage.clear(transaction);
    }

    protected void actionDelete(WikittyTransaction transaction, WikittyActionMessage message)
            throws JMSException {
        ArrayList<String> idList =
                (ArrayList<String>) message.getParameter(PARAM_IDS);
        
        delegateStorage.delete(transaction, idList);
    }
}
