/*
 * #%L
 * Wikitty :: api
 * 
 * $Id: WikittyExtensionMigrationRename.java 1141 2011-08-13 15:27:14Z bpoussin $
 * $HeadURL: http://svn.nuiton.org/svn/wikitty/tags/wikitty-3.2/wikitty-api/src/main/java/org/nuiton/wikitty/services/WikittyExtensionMigrationRename.java $
 * %%
 * Copyright (C) 2009 - 2010 CodeLutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.wikitty.services;

import org.nuiton.wikitty.entities.Wikitty;
import org.nuiton.wikitty.entities.WikittyExtension;
import org.nuiton.wikitty.entities.FieldType;
import org.nuiton.wikitty.entities.WikittyImpl;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.wikitty.WikittyService;

/**
 * Basic extension migration use to manage rename field name. Order to detect a
 * rename, not creation a new field, you should use tag value "renameFrom" with
 * name of use in before extension.
 *
 * Since 1.5, this class also take care about extension order.
 * 
 * @author ruchaud
 */
public class WikittyExtensionMigrationRename implements WikittyExtensionMigration {

    static private Log log = LogFactory.getLog(WikittyExtensionMigrationRename.class);

    public static final String TAG_RENAME = "renameFrom";

    /**
     * Recursively add extension with requires ones (orderer).
     * WikittyService used in this method must not have WikittyServiceSecurity
     * in stack because we pass null as secuirtyToken to method. (TODO poussin
     * 2010 verify this assertion, when we use Migration, we don't need
     * Security)
     * 
     * @param service service
     * @param wikitty wikitty to add extension to
     * @param newExt extension to add
     */
    protected void addExtension(
            WikittyService service, Wikitty wikitty, WikittyExtension newExt) {

        // manage requires before current
        List<String> requires = newExt.getRequires();
        if (requires != null && !requires.isEmpty()) {
            for (String require : requires) {
                WikittyExtension newRequireExt =
                        service.restoreExtensionLastVersion(null, require);
                addExtension(service, wikitty, newRequireExt);
            }
        }

        // add current extension
        wikitty.addExtension(newExt);
    }

    @Override
    public Wikitty migrate(WikittyService service, Wikitty wikitty,
            WikittyExtension oldExt, WikittyExtension newExt) {

        String wikittyId = wikitty.getId();
        String wikittyVersion = wikitty.getVersion();

        Wikitty result = new WikittyImpl(wikittyId);
        result.setVersion(wikittyVersion);

        // Add all extension and replaceWith old by the new
        for (WikittyExtension extension : wikitty.getExtensions()) {
            // add one by one to manage require
            
            // during loop add new when old one is found
            if (extension.equals(oldExt)) {
                addExtension(service, result, newExt);
            }
            else {
                addExtension(service, result, extension);
            }
        }

        // Copy other values
        Set<String> fqFieldNames = wikitty.fieldNames();
        String extName = newExt.getName();
        for (String fieldName : fqFieldNames) {
            if(!fieldName.startsWith(extName + ".")) {
                Object value = wikitty.getFqField(fieldName);
                log.debug("Value " + fieldName + "=" + value);
                result.setFqField(fieldName, value);
            }
        }

        // Migrate field name
        Collection<String> oldFieldNames = oldExt.getFieldNames();
        Collection<String> newFieldNames = newExt.getFieldNames();
        for (String fieldName : newFieldNames) {
            FieldType fieldType = newExt.getFieldType(fieldName);

            String renameFrom = fieldType.getTagValue(TAG_RENAME);
            log.debug("Scan rename migration on " + extName + "." + fieldName + ":" + TAG_RENAME + "=" + renameFrom);

            if(renameFrom != null && !renameFrom.isEmpty()) {
                Object value = wikitty.getFieldAsObject(extName, renameFrom);
                log.debug("Rename " + extName + "." + fieldName + "=" + value);
                result.setField(extName, fieldName, value);
            } else {
                if(oldFieldNames.contains(fieldName)) {
                    Object value = wikitty.getFieldAsObject(extName, fieldName);
                    log.debug("Copy " + extName + "." + fieldName + "=" + value);
                    result.setField(extName, fieldName, value);
                }
            }
        }

        return result;
    }

}
