/*
 * #%L
 * ToPIA :: Service Migration
 * 
 * $Id: TopiaMigrationCallbackByClassNG.java 2792 2013-08-05 10:41:12Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-3.0-alpha-3.1/topia-service-migration/src/main/java/org/nuiton/topia/migration/TopiaMigrationCallbackByClassNG.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin, Tony chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.topia.migration;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.Version;
import org.nuiton.util.VersionUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.Set;
import java.util.TreeMap;

/**
 * Migration callback which use a different class for each version to migrate.
 * <p/>
 * You must fill in the constructor the mapping for each version of
 * {@link #getAvailableVersions()} a matching migrator for version which
 * extends {@link MigrationCallBackForVersion}.
 * <p/>
 * Use the callback when you have a lot of version to migrate and the
 * {@link TopiaMigrationCallbackByMethod} begins to be messy.
 *
 * @author tchemit <chemit@codelutin.com>
 * @version $Id: TopiaMigrationCallbackByClassNG.java 2792 2013-08-05 10:41:12Z tchemit $
 * @since 2.9.11
 */
public abstract class TopiaMigrationCallbackByClassNG extends AbstractTopiaMigrationCallback {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(TopiaMigrationCallbackByClassNG.class);

    protected MigrationCallBackForVersionResolver callBackResolver;

    protected TopiaMigrationCallbackByClassNG(MigrationCallBackForVersionResolver callBackResolver) {

        this.callBackResolver = callBackResolver;
    }

    @Override
    public Version[] getAvailableVersions() {
        Set<Version> allVersions = callBackResolver.getAllVersions();
        return allVersions.toArray(new Version[allVersions.size()]);
    }

    @Override
    protected void migrateForVersion(Version version,
                                     TopiaContext tx,
                                     boolean showSql,
                                     boolean showProgression) throws Exception {

        MigrationCallBackForVersion migrator = callBackResolver.getCallBack(version);

        migrator.setCallBack(this);

        String[] queries = migrator.prepareMigration(tx, showSql, showProgression);

        executeSQL(tx, showSql, showProgression, queries);

    }

    /**
     * Call back for a given version.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.5
     */
    public abstract static class MigrationCallBackForVersion {

        protected TopiaMigrationCallbackByClassNG callBack;

        public abstract Version getVersion();

        public void setCallBack(TopiaMigrationCallbackByClassNG callBack) {
            this.callBack = callBack;
        }

        protected String[] prepareMigration(TopiaContext tx,
                                            boolean showSql,
                                            boolean showProgression) throws TopiaException {

            List<String> queries = new ArrayList<String>();

            prepareMigrationScript(tx, queries, showSql, showProgression);

            return queries.toArray(new String[queries.size()]);
        }

        protected abstract void prepareMigrationScript(TopiaContext tx,
                                                       List<String> queries,
                                                       boolean showSql,
                                                       boolean showProgression) throws TopiaException;

        public void executeSQL(TopiaContext tx,
                               String... sqls) throws TopiaException {
            callBack.executeSQL(tx, sqls);
        }

        public void executeSQL(TopiaContext tx,
                               boolean showSql,
                               boolean showProgression,
                               String... sqls) throws TopiaException {
            callBack.executeSQL(tx, showSql, showProgression, sqls);
        }

    }

    /**
     * Resolver to obtain the correct migration class for a given version.
     *
     * @since 2.6.11
     */
    public interface MigrationCallBackForVersionResolver {

        /**
         * Returns all detected versions.
         *
         * @return all detected versions.
         */
        Set<Version> getAllVersions();

        /**
         * for a given version, returns his migration callback.
         *
         * @param version the version to migrate
         * @return the migration call for the given version, or {@code null}
         * if no such migration callback exists for the version
         */
        MigrationCallBackForVersion getCallBack(Version version);
    }

    /**
     * A simple call back resolver via a service loader.
     *
     * @author tchemit <chemit@codelutin.com>
     * @since 2.9.11
     */
    public static class MigrationCallBackForVersionResolverByServiceLoader implements MigrationCallBackForVersionResolver {

        protected final Map<Version, MigrationCallBackForVersion> versionMigrationMapping;

        public MigrationCallBackForVersionResolverByServiceLoader() {
            this.versionMigrationMapping = new TreeMap<Version, MigrationCallBackForVersion>(
                    new VersionUtil.VersionComparator());
            ServiceLoader<MigrationCallBackForVersion> load = ServiceLoader.load(MigrationCallBackForVersion.class);
            for (MigrationCallBackForVersion callBackForVersion : load) {
                Version version = callBackForVersion.getVersion();
                if (log.isInfoEnabled()) {
                    log.info("Detects migration version " + version + " [" +
                             callBackForVersion + "]");
                }
                versionMigrationMapping.put(version, callBackForVersion);
            }
        }

        @Override
        public MigrationCallBackForVersion getCallBack(Version version) {
            return versionMigrationMapping.get(version);
        }

        @Override
        public Set<Version> getAllVersions() {
            return versionMigrationMapping.keySet();
        }
    }
}