/*
 * #%L
 * ToPIA :: Service Migration
 * 
 * $Id: DatabaseManager.java 1894 2010-04-15 15:44:51Z tchemit $
 * $HeadURL: http://svn.nuiton.org/svn/topia/tags/topia-2.3.3/topia-service-migration/src/main/java/org/nuiton/topia/migration/DatabaseManager.java $
 * %%
 * Copyright (C) 2004 - 2010 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

package org.nuiton.topia.migration;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Version;
import org.nuiton.topia.migration.mappings.TMSVersion;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.cfg.Configuration;
import org.hibernate.cfg.Environment;
import org.hibernate.connection.ConnectionProvider;
import org.hibernate.connection.ConnectionProviderFactory;
import org.hibernate.dialect.Dialect;
import org.hibernate.exception.JDBCConnectionException;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.mapping.ForeignKey;
import org.hibernate.mapping.Table;
import org.hibernate.tool.hbm2ddl.DatabaseMetadata;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.hbm2ddl.TableMetadata;

/**
 * DatabaseManager.java
 *
 * Cette classe sert à acceder a la base, pour la version notement
 * creer les schemas, renommer et supprimer les tables.
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon Sébastien
 * @author Trève Vincent
 * @version $Revision: 1894 $
 *
 * Last update : $Date: 2010-04-15 17:44:51 +0200 (jeu., 15 avril 2010) $
 */
public class DatabaseManager {

    /**
     * Session factory
     */
    private SessionFactory sessionFactory;
    /**
     * Configuration pour se connecter a la base
     * et manipuler la table version seulement
     */
    private Configuration dbConfiguration;
    /**
     * Logger (common-logging)
     */
    private static Log logger = LogFactory.getLog(DatabaseManager.class);
    /**
     * Suffix du nom des tables : _tms_v0 par exemple
     */
    private static String VERSIONNED_TABLES_SUFFIX = "_tmsv";

    /**
     * Constructeur
     * @param pInfosConnexion Properties hibernate
     */
    public DatabaseManager(Properties pInfosConnexion) {
        // initie les proprietes
        dbConfiguration = new Configuration();
        dbConfiguration.setProperties(pInfosConnexion);

        logger.debug("Configuration url : " + pInfosConnexion.getProperty(Environment.URL));
        logger.debug("Configuration driver : " + pInfosConnexion.getProperty(Environment.DRIVER));
        logger.debug("Configuration dialect : " + pInfosConnexion.getProperty(Environment.DIALECT));

        // remplit la suite avec la liste de nos mappings...
        logger.debug("Adding mappings for " + TMSVersion.class.getSimpleName());

        // Hibernate will look for mapping files named
        // /org/nuiton/topia/migration/mappings/TMSVersion.hbm.xml in the classpath.
        // This approach eliminates any hardcoded filenames.
        dbConfiguration.addClass(TMSVersion.class);

        // retourne une session factory
        sessionFactory = dbConfiguration.buildSessionFactory();
    }

    /**
     * Retourne la configuration Hibernate de la base.
     * @return la configuration de la base
     */
    public Configuration getDbConfiguration() {
        return dbConfiguration;
    }

    /**
     * @return une {@link Connection}, charge a l'appelant de la fermer par la
     * suite
     */
    public Connection getConnection() {
        Connection result = sessionFactory.openStatelessSession().connection();
        return result;
    }

    /**
     * Retourne la version de la base
     * @return la version present en base, ou <tt>null</tt> si la version ne peut pas etre determinee
     * @throws MigrationServiceException si un pb
     */
    public Version getDataBaseVersion() throws MigrationServiceException {
        Version version = null;

        logger.debug("Begin transaction to get version in database");

        // get session
        Session session = sessionFactory.openSession();

        try {

            // debut d'une transaction
            Transaction tx = session.beginTransaction();

            // execute query
            TMSVersion result = (TMSVersion) session.createCriteria(TMSVersion.class).uniqueResult();
            if (result != null) {
                version = new Version(result.getVersion());
                logger.debug("Query executed, version found : " + version.getVersion());
            } else {
                logger.debug("Query executed, no version found");
            }

            // commit
            tx.commit();
        } catch (JDBCConnectionException e) {
            throw new MigrationServiceException("Connection to database refused, check your configuration !");
        } catch (SQLGrammarException e) {
            // si la table n'existe pas, on obtient une exception : base non versionnee
            logger.debug("Exception on request : table not found");

            // on retourn null
            version = null;
        } finally {
            session.close();
        }

        return version;
    }

    /**
     * Renomme les tables en supprimant le suffixe
     *
     * @param vdbVersion Version a ajouter
     * @param oldConfiguration Configuration contenant le schema
     */
    public void renameTables(Configuration oldConfiguration, Version vdbVersion) {

        logger.debug("Renaming tables in configuration and database for version " + vdbVersion.getVersion());

        // get session
        Session session = sessionFactory.openSession();

        // debut d'une transaction
        Transaction tx = session.beginTransaction();

        // DONE : voir si les relation *<->* sont listees par l'iterateur
        // -> ok, normalement c bon, c gere
        Iterator<?> i = oldConfiguration.getTableMappings();
        while (i.hasNext()) {
            Table table = (Table) i.next();

            String tableName = table.getName();
            String suffix = getTableSuffixForVersion(vdbVersion);
            String newTableName = tableName;
            if (tableName.endsWith(suffix)) {
                newTableName = tableName.substring(0, tableName.length() - suffix.length());
            }

            // ALTER TABLE name RENAME TO newName
            //  marche normalement pour mysql, h2, postgres

            // TODO table existe pas
            // TODO autre table existe deja
            // TODO get sql string from dialect ...

            //Dialect dialect = Dialect.getDialect(oldConfiguration.getProperties());

            logger.debug("Renaming table " + tableName + " to " + newTableName);
            String sQuery = "ALTER TABLE " + tableName + " RENAME TO " + newTableName;
            //logger.debug("Query : " + sQuery);

            SQLQuery sqlq = session.createSQLQuery(sQuery);
            sqlq.executeUpdate();

            // rennomage dans la config
            table.setName(newTableName);
        }

        // commit
        tx.commit();

        // close
        session.close();
    }

    /**
     * Renome les table dans la configuration hibernate
     *
     * @param oldConfiguration
     * @param vdbVersion
     * @return la nouvelle configuration
     */
    public Configuration setRenamedTableSchema(Configuration oldConfiguration, Version vdbVersion) {

        logger.debug("Renaming tables in configuration for version " + vdbVersion.getVersion());

        // voir si les relation *<->* sont listees par l'iterateur
        // -> ok, normalement c bon, c gere

        // bug: sans cette ligne, les tables many-to-many ne sont pas list�es
        oldConfiguration.buildMappings();

        Iterator<?> i = oldConfiguration.getTableMappings();
        while (i.hasNext()) {
            Table table = (Table) i.next();

            if (table.isPhysicalTable()) { // hibernate utilse ca

                // rename constraints FK
                Iterator<?> fKeys = table.getForeignKeyIterator();
                while (fKeys.hasNext()) {
                    ForeignKey fKey = (ForeignKey) fKeys.next();

                    if (logger.isTraceEnabled()) {
                        logger.trace("Changing constraints name : " + fKey.getName() + " to " + fKey.getName() + vdbVersion.getValidName());
                    }
                    fKey.setName(fKey.getName() + vdbVersion.getValidName());
                }

                String tableName = table.getName();
                String newTableName = tableName + getTableSuffixForVersion(vdbVersion);

                // rennomage dans la config
                logger.debug("Renaming table " + tableName + " to " + newTableName);
                table.setName(newTableName);
            }
        }

        return oldConfiguration;
    }

    /**
     * Creer le nouveau schema pour l'application
     * @param newConfiguration la configuration contenant le nouveau schema
     */
    public void setApplicationSchemaInDatabase(Configuration newConfiguration) {

        // log
        logger.debug("Creating new schema");

        createSchema(newConfiguration);

        // log
        logger.debug("Schema created");
    }

    /**
     * Creer un schema
     * @param configuration la configuration contenant les schemas
     */
    protected void createSchema(Configuration configuration) {
        // creer le schema en base
        SchemaExport schemaExport = new SchemaExport(configuration);
        schemaExport.execute(false/*script*/, true/*export*/, false/*justDrop*/, true/*justCreate*/);
    }

    /**
     * Supprimer un schema
     * @param configuration la configuration contenant les schemas
     */
    protected void dropSchema(Configuration configuration) {
        // supprimer le schema en base
        SchemaExport schemaExport = new SchemaExport(configuration);
        schemaExport.drop(false, true);
    }

    /**
     * Creer le schema pour la table "tms_version"
     */
    public void createVersionTable() {

        logger.debug("Adding table to put version");

        // creer le schema en base
        // dans la configuration dbConfiguration, il n'y a que la table version
        SchemaExport schemaExport = new SchemaExport(dbConfiguration);
        schemaExport.create(false, true);

        logger.debug("Table for " + TMSVersion.class.getSimpleName() + " created");
    }

    /**
     * Introduit la version du nouveau schema dans la base
     * @param newVersion la version
     */
    public void putVersionInDatabase(Version newVersion) {

        // get session
        Session session = sessionFactory.openSession();

        // debut d'une transaction
        Transaction tx = session.beginTransaction();

        logger.debug("Deleting existing versions");

        // supprime les versions existants
        // au pire, on pourrait remplacer la premiere
        // execute query
        List<?> lVersion = session.createCriteria(TMSVersion.class).list();
        for (Object o : lVersion) {
            TMSVersion v = (TMSVersion) o;
            logger.debug("Deleting version " + v.getVersion());

            session.delete(v);
        }

        // positionne la version
        logger.debug("Setting database version to " + newVersion.getVersion());
        TMSVersion version = new TMSVersion(newVersion.getVersion());
        session.save(version);

        // commit
        tx.commit();

        // session close
        session.close();
    }

    /**
     * Supprime les tables des l'ancien mapping
     *
     * @param oldConfiguration configuration contenant le schema
     */
    public void removeTablesFromOldMapping(Configuration oldConfiguration) {

        if (logger.isDebugEnabled()) {
            logger.debug("Removing schema");
        }
        dropSchema(oldConfiguration);
    }

    /**
     * Return table suffix name
     * 
     * @param version version
     * @return suffix name
     */
    protected String getTableSuffixForVersion(Version version) {
        String suffix = VERSIONNED_TABLES_SUFFIX + version.getValidName();
        return suffix;
    }

    /**
     * Test si les tables correspondant a une configuration existent.
     * 
     * Test si au moins une table de la configuration existe.
     * 
     * @param configuration la configuration
     * @return <tt>true</tt> si le schema existe
     */
    public boolean isSchemaExist(Configuration configuration) {

        boolean exist = false;

        Dialect dialect = Dialect.getDialect(configuration.getProperties());
        ConnectionProvider connectionProvider = ConnectionProviderFactory.newConnectionProvider(configuration.getProperties());

        try {

            Iterator<?> tables = configuration.getTableMappings();

            if (tables.hasNext()) {
                Table testTable = (Table) tables.next();

                Connection connection = connectionProvider.getConnection();

                DatabaseMetadata meta = new DatabaseMetadata(connection, dialect);

                TableMetadata tmd = meta.getTableMetadata(testTable.getName(), testTable.getSchema(), testTable.getCatalog(), testTable.isQuoted());

                if (tmd != null) { //table not found
                    exist = true;
                }
            }

        } catch (SQLException e) {
            logger.error("Cant connect to database", e);
        }

        return exist;
    }

    /**
     * Se deconnecte
     */
    public void disconnect() {
        //nettoye
        sessionFactory.close();
    }
}
