/* *##% 
 * ToPIA :: Service Migration
 * Copyright (C) 2004 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*/
package org.nuiton.topia.migration.kernel;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.HashSet;
import java.util.List;

import java.util.Set;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.Resource;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.hibernate.cfg.Configuration;
import org.hibernate.util.DTDEntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * ConfigurationHelper.java
 * 
 * Charge des configuration en mode "dynamic-map".
 * 
 * @author Chatellier Eric
 * @author Chevallereau Benjamin
 * @author Eon Sébastien
 * @author Trève Vincent
 * @version $Revision: 1459 $
 *
 * Last update : $Date: 2009-05-16 09:56:47 +0200 (Sat, 16 May 2009) $
 */
public class ConfigurationHelper {

    /**
     * Logger (common-logging)
     */
    private static Log logger = LogFactory.getLog(ConfigurationHelper.class);
    /**
     * Single instance
     */
    protected static ConfigurationHelper instance = new ConfigurationHelper();

    /**
     * Constructeur
     */
    protected ConfigurationHelper() {
    }

    /**
     * Get single instance
     *
     * @return the shared instance
     */
    public static ConfigurationHelper getInstance() {
        return instance;
    }

    /**
     * Retourne une configuration correspondant au chargement
     * de tous les mappings du dossier specifie.
     *
     * Ici, on pourrait utiliser configuration.addDirectory
     * mais on modifie les mappings version "entity-map"
     *
     * @param pathDirectory Le dossier ou se trouvent les mappings (contient
     * la version souhaitee)
     * @return La configuration associee
     */
    public Configuration getConfigurationInDirectory(String pathDirectory) {

        // schema des noms de fichier a lire
        final String regexFilename = ".*\\.hbm\\.xml";

        // nouvelle configuration en retour
        Configuration configuration = null;

        // FIXME faire deux cas ici. Soit on est sur le filesystem comme deja
        // code. Soit on est dans un jar (a faire). Dans tous les cas, il faut
        // ensuite travailler avec des URLs et non pas des Strings.
        //
        // si mappingDir n'est pas un repertoire, rechercher avec
        // ResourceUtil.getUrls(String pattern): List<URL> ou equivalent.

        // mappingDir est toujours dans le classpath et Resource.getURLs doit
        // le trouver, donc que ce soit un dossier ou un jar ici importe peu

        // Don't use File.separator, don't work on windows
        String pattern = ".*" + pathDirectory + "/" + regexFilename;

        // FIXME attention ne fonctionne pas sur du multi-module maven!
        // il faut bien verifier que on a pas de doublon
        //

        // find all mapping
        ClassLoader loader = ConfigurationHelper.class.getClassLoader();
        List<URL> urls = Resource.getURLs(pattern, loader instanceof URLClassLoader ? (URLClassLoader) loader : null);

        logger.debug("Loading mappings in " + pathDirectory);

        if (urls != null && urls.size() > 0) {
            // s'il y a des fichier a charger
            configuration = new Configuration();
            Set<String> done = new HashSet<String>();

            for (URL url : urls) {
                String filename = url.getFile();
                if (logger.isTraceEnabled()) {
                    logger.trace(filename);
                }
                int index = filename.indexOf(pathDirectory);
                filename = filename.substring(index + 1);
                if (logger.isTraceEnabled()) {
                    logger.trace(filename);
                }
                if (done.contains(filename)) {
                    logger.debug("already loaded resource : " + url);
                    continue;
                }
                String xmlmapping = ConfigurationHelper.getEntityMappingFromFile(url);

                // lit le fichier avec lutin xml
                org.w3c.dom.Document domDoc = getDocumentResolvedByHibernate(xmlmapping);

                logger.debug("Local mapping adding : " + url);
                configuration.addDocument(domDoc);

                logger.debug("Local mapping added  : " + url);
                done.add(filename);
            }
        } else {
            logger.error("No mappings files found in directory " + pathDirectory);
        }

        return configuration;
    }

    /**
     * Charge un fichier et retourne un mapping valide pour etre manipule via les map
     *
     * @param fichier le nom du fichier
     * @return le mapping en forme xml
     */
    protected static String getEntityMappingFromFile(URL fichier) {
        String sXml = null;

        // charge le document en representation dom4j
        Document domDoc = getDom4jDocument(fichier);

        // transforme en entity
        domDoc = transformAsValidHibernateMapConfigFile(domDoc);

        // transforme le document en chaine
        sXml = domDoc.asXML();

        return sXml;
    }

    /**
     * Transforme en ajoutant des info de mappings.
     *
     * Ajoute un entity-name avec la valeur de name.
     * Supprime name , sinon hibernate l'utilise encore.
     *
     * @param dom4jdoc le document
     * @return un document mapping entite
     */
    protected static Document transformAsValidHibernateMapConfigFile(Document dom4jdoc) {
        Element rootElem = dom4jdoc.getRootElement();

        List<Element> childElements = rootElem.elements("class");

        for (Element eChild : childElements) {

            // suivant si l'attribut est deja la ou pas
            String entityname = eChild.attributeValue("entity-name");
            String odlClassName = eChild.attributeValue("name");

            if (entityname == null) {
                // on l'ajoute alors
                eChild.addAttribute("entity-name", odlClassName);
            }

            // et supprime l'attribut name (sinon, hibernate continue de l'utiliser)
            Attribute attName = eChild.attribute("name");
            if (attName != null) {
                eChild.remove(attName);
            }
        }

        return dom4jdoc;
    }

    /**
     * Charge un document dom4j
     *
     * @param fichier le nom du fichier
     * @return un document dom4j
     */
    protected static Document getDom4jDocument(URL fichier) {
        SAXReader reader = new SAXReader();
        reader.setValidation(false);
        //DTDEntityResolver = celui d'hibernate
        reader.setEntityResolver(new DTDEntityResolver());
        org.dom4j.Document document = null;
        try {
            document = reader.read(fichier);
        } catch (DocumentException e) {
            e.printStackTrace();
        }
        return document;
    }

    // a partir d'ici, on reprend du code de lutin xml
    // mais on doit le modifier pour specifier un DTDResolver , celui d'hibernate
    /**
     * Construit un nouveau document a partir du flux
     * @param xmlString el flux d'entree
     * @return le document construit
     */
    protected static org.w3c.dom.Document getDocumentResolvedByHibernate(String xmlString) {
        DocumentBuilder domBuild = ConfigurationHelper.getDocumentBuilder();
        // resolver hibernate
        domBuild.setEntityResolver(new DTDEntityResolver());

        org.w3c.dom.Document domDoc = null;
        try {
            domDoc = domBuild.parse(new InputSource(new BufferedReader(new StringReader(xmlString))));
        } catch (IOException eee) {
            logger.error("Error while reading xml mapping");
        } catch (SAXException eee) {
            logger.error("Error while parsing xml mapping");
        }

        return domDoc;
    }

    /**
     * Recopie, elle est protected dans lutin xml
     * @return un DocumentBuilder
     */
    protected static DocumentBuilder getDocumentBuilder() {
        DocumentBuilderFactory builderFactory =
                DocumentBuilderFactory.newInstance();

        // on configure le builder
        builderFactory.setCoalescing(true);
        builderFactory.setExpandEntityReferences(true);
        builderFactory.setIgnoringComments(true);
        builderFactory.setNamespaceAware(true);
        builderFactory.setValidating(false);

        DocumentBuilder builder = null;
        try {
            builder = builderFactory.newDocumentBuilder();
        } catch (ParserConfigurationException eee) {
            logger.error("Error while parsing xml mapping");
        }

        return builder;
    }
}
