/**
 * ##% Copyright (C) 2008 Code Lutin, Tony Chemit
 * This program is free software; you
 * can redistribute it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version. This program is
 * distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details. You
 * should have received a copy of the GNU General Public License along with this
 * program; if not, write to the Free Software Foundation, Inc., 59 Temple Place
 * - Suite 330, Boston, MA 02111-1307, USA. 
 * ##%
 */
package org.nuiton.jaxx.util;

import static org.nuiton.i18n.I18n._;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.ImageIcon;
import java.lang.reflect.Constructor;

/**
 * Definition of an ui, with his model, handler and ui class definitions.
 * <p/>
 * The class contains also a shared instace of concrete ui.
 *
 * @param <M> type of model
 * @param <U> type of ui
 * @param <H> type of handler
 * @author chemit
 */
public class DialogUIDef<M extends DialogUIModel, U extends DialogUI<H>, H extends DialogUIHandler<M, U>> implements java.io.Serializable {

    static protected final Log log = LogFactory.getLog(DialogUIDef.class);

    public static <M extends DialogUIModel, U extends DialogUI<H>, H extends DialogUIHandler<M, U>> DialogUIDef<M, U, H> newDef(Class<H> handlerClass, Class<U> uiClass, Class<M> modelClass, String showActionLibelle, String showActionTip, String uiTitle) {
        DialogUIDef<M, U, H> result;
        result = new DialogUIDef<M, U, H>(handlerClass, uiClass, modelClass, showActionLibelle, showActionTip, uiTitle);
        return result;
    }

    /**
     * model class
     */
    private final Class<M> modelClass;

    /**
     * handler class
     */
    private final Class<H> handlerClass;

    /**
     * abstract ui class
     */
    private final Class<U> uiClass;

    /**
     * concrete lookup ui class
     */
    private Class<? extends U> uiImplClass;

    /**
     * shared instance of ui
     */
    protected U uiInstance;

    /**
     * unique name of ui def
     */
    protected final String name;

    protected final String uiTitle;
    protected final String showActionLibelle;
    protected final String showActionTip;

    protected ImageIcon showUIActionIcon;


    private static final long serialVersionUID = 1L;

    private DialogUIDef(Class<H> handlerClass, Class<U> uiClass, Class<M> modelClass,
                        String showActionLibelle, String showActionTip, String uiTitle) {
        this.handlerClass = handlerClass;
        this.uiClass = uiClass;
        this.modelClass = modelClass;
        this.showActionLibelle = showActionLibelle;
        this.name = uiClass.getSimpleName().toLowerCase();
        this.showActionTip = showActionTip;
        this.uiTitle = uiTitle;
    }

    public Class<U> getUiClass() {
        return uiClass;
    }

    public Class<H> getHandlerClass() {
        return handlerClass;
    }

    public Class<M> getModelClass() {
        return modelClass;
    }

    public Class<? extends U> getUiImplClass() {
        return uiImplClass;
    }

    public String getUiTitle() {
        return _(uiTitle);
    }

    public String getShowActionLibelle() {
        return _(showActionLibelle);
    }

    public String getShowActionTip() {
        return _(showActionTip);
    }

    public ImageIcon getShowUIActionIcon() {
        if (showUIActionIcon == null) {
            showUIActionIcon = UIHelper.createActionIcon("show-" + name);
        }
        return showUIActionIcon;
    }

    @SuppressWarnings({"unchecked"})
    public void setUiImplClass(Class<?> uiImplClass) {
        this.uiImplClass = (Class<? extends U>) uiImplClass;
    }

    @Override
    public boolean equals(Object o) {
        return this == o || o instanceof DialogUIDef<?,?,?> && uiClass.equals(((DialogUIDef<?,?,?>) o).uiClass);
    }

    @Override
    public int hashCode() {
        return uiClass.hashCode();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString()).append('<');
        sb.append(printClass("handler", handlerClass, true));
        sb.append(printClass("model", modelClass, true));
        sb.append(printClass("ui", uiClass, true));
        sb.append(printClass("uiImpl", uiImplClass, false));
        return sb.toString();
    }

    protected U getUiInstance() {
        // no lazy instanciation, to control ui instanciation...
        /*if (uiInstance == null) {
            if (uiImplClass == null) {
                throw new IllegalStateException("no concrete ui impl found in " + this);
            }
            synchronized (this) {
                try {
                    uiInstance = uiImplClass.newInstance();
                } catch (Exception e) {
                    throw new IllegalStateException("could not instanciate ui " + this,e);
                }
            }
        }*/
        return uiInstance;
    }

    protected void setUiInstance(U uiInstance) {
        this.uiInstance = uiInstance;
    }

    protected U newUI() {
        if (uiImplClass == null) {
            throw new IllegalStateException("no concrete ui impl found in " + this);
        }
        try {
            U result = uiImplClass.newInstance();
            log.info(result);
            return result;
        } catch (Exception e) {
            throw new IllegalStateException("could not instanciate ui " + this, e);
        }
    }

    protected M newModel() {
        if (modelClass == null) {
            throw new IllegalStateException("no model impl found in " + this);
        }
        try {
            M model = modelClass.newInstance();
            log.info(model);
            return model;
        } catch (Exception e) {
            throw new IllegalStateException("could not instanciate ui " + this, e);
        }
    }

    protected H newHandler(U ui, M model, Object... params) {
        if (handlerClass == null) {
            throw new IllegalStateException("no handler impl found in " + this);
        }
        try {
            Class<?>[] prototype = getHandlerPrototype(params);
            Object[] parameters = getHandlerParameters(ui, model, params);
            H result = handlerClass.getConstructor(prototype).newInstance(parameters);
            log.info(result);
            return result;
        } catch (Exception e) {
            throw new IllegalStateException("could not instanciate ui " + this, e);
        }
    }

    protected Object[] getHandlerParameters(U ui, M model, Object[] params) {
        Object[] result = new Object[2 + params.length];
        result[0] = ui;
        result[1] = model;
        System.arraycopy(params, 0, result, 2, params.length);
        return result;
    }

    protected Class<?>[] getHandlerPrototype(Object[] params) {
        int length = params.length;
        for (Constructor<?> constructor : handlerClass.getConstructors()) {
            Class<?>[] prototype = constructor.getParameterTypes();
            if (prototype.length == 2 + length && prototype[0] == uiClass && prototype[1] == modelClass) {
                return prototype;
            }
        }
        throw new IllegalStateException("could not find a matching constructor in " + handlerClass);
    }

    protected String printClass(String s, Class<?> aClass, boolean notLast) {
        return s + ':' + (aClass == null ? null : aClass.getSimpleName()) + (notLast ? ", " : ">");
    }
}
