/*
 * *##% 
 * JAXX Compiler
 * Copyright (C) 2008 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * ##%*
 */
package jaxx.compiler.tags;

import jaxx.compiler.CompilerException;
import jaxx.compiler.UnsupportedAttributeException;
import jaxx.compiler.JAXXCompiler;
import jaxx.compiler.css.parser.CSSParser;
import jaxx.compiler.css.parser.CSSParserConstants;
import jaxx.compiler.css.parser.CSSParserTreeConstants;
import jaxx.compiler.css.parser.SimpleNode;
import jaxx.runtime.css.Rule;
import jaxx.runtime.css.Selector;
import jaxx.runtime.css.Stylesheet;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Handles the <code>&lt;style&gt;</code> tag.
 *
 * @author Ethan Nicholas
 */
public class StyleHandler implements TagHandler {
    public static final String SOURCE_ATTRIBUTE = "source";

    @Override
    public void compileFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        boolean source = false;
        NamedNodeMap attributes = tag.getAttributes();
        for (int i = 0; i < attributes.getLength(); i++) {
            Attr attribute = (Attr) attributes.item(i);
            String name = attribute.getName();
            String attrValue = attribute.getValue();
            if (name.equals(SOURCE_ATTRIBUTE)) {
                source = true;
                File styleFile = new File(compiler.getBaseDir(), attrValue.replace('/', File.separatorChar));
                StringWriter styleBuffer = new StringWriter();
                try {
                    FileReader in = new FileReader(styleFile);
                    char[] readBuffer = new char[2048];
                    int c;
                    while ((c = in.read(readBuffer)) > 0) {
                        styleBuffer.write(readBuffer, 0, c);
                    }
                } catch (FileNotFoundException e) {
                    compiler.reportError("stylesheet file not found: " + styleFile);
                }
                compiler.getSourceFiles().push(styleFile);
                compiler.registerStylesheet(processStylesheet(styleBuffer.toString()));
                compiler.getSourceFiles().pop();
            } else if (!name.startsWith(XMLNS_ATTRIBUTE) &&
                    !JAXXCompiler.JAXX_INTERNAL_NAMESPACE.equals(attribute.getNamespaceURI())) {
                throw new UnsupportedAttributeException(name);
            }
        }

        StringBuffer style = new StringBuffer();
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node child = children.item(i);
            switch (child.getNodeType()) {
                case Node.ELEMENT_NODE:
                    compiler.reportError("<style> tag may not contain child elements: " + tag);
                case Node.TEXT_NODE: // fall through
                case Node.CDATA_SECTION_NODE:
                    style.append(((Text) child).getData());
            }
        }

        String styleString = style.toString().trim();
        if (styleString.length() > 0) {
            if (source) {
                compiler.reportError("<style> tag has both a source attribute and an inline stylesheet");
            }
            compiler.registerStylesheet(processStylesheet(style.toString()));
        }
    }

    @Override
    public void compileSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
    }

    protected Selector processSelector(SimpleNode selector) {
        if (selector.getId() != CSSParserTreeConstants.JJTSELECTOR) {
            throw new IllegalArgumentException("argument node is not a Selector");
        }
        String javaClassName = null;
        String styleClass = null;
        String pseudoClass = null;
        String id = null;

        for (int i = 0; i < selector.jjtGetNumChildren(); i++) {
            SimpleNode child = selector.getChild(i);
            switch (child.getId()) {
                case CSSParserTreeConstants.JJTJAVACLASS:
                    if (!child.getText().trim().equals("*")) {
                        javaClassName = child.getText();
                    }
                    break;
                case CSSParserTreeConstants.JJTCLASS:
                    styleClass = child.getText().substring(1);
                    break;
                case CSSParserTreeConstants.JJTPSEUDOCLASS:
                    pseudoClass = child.getText().substring(1);
                    break;
                case CSSParserTreeConstants.JJTID:
                    id = child.getText().substring(1);
                    break;

                default:
                    throw new IllegalStateException("unexpected child of Selector node, type=" + child.getId());
            }
        }

        return new Selector(javaClassName, styleClass, pseudoClass, id);
    }

    protected Rule processRule(SimpleNode ruleNode) {
        if (ruleNode.getId() != CSSParserTreeConstants.JJTRULE) {
            throw new IllegalArgumentException("argument node is not a Rule");
        }
        SimpleNode selectorsNode = ruleNode.getChild(0);
        assert selectorsNode.getId() == CSSParserTreeConstants.JJTSELECTORS : "expected node to be of type Selectors";

        List<Selector> selectors = new ArrayList<Selector>();
        for (int i = 0; i < selectorsNode.jjtGetNumChildren(); i++) {
            SimpleNode selectorNode = selectorsNode.getChild(i);
            selectors.add(processSelector(selectorNode));
        }

        Map<String, String> properties = new HashMap<String, String>();
        for (int i = 1; i < ruleNode.jjtGetNumChildren(); i++) {
            SimpleNode declarationNode = ruleNode.getChild(i);
            if (declarationNode.getId() == CSSParserTreeConstants.JJTDECLARATION) {
                String key = declarationNode.getChild(0).getText();
                SimpleNode valueNode = declarationNode.getChild(1);
                String value = valueNode.getText();
                if (valueNode.firstToken.kind == CSSParserConstants.STRING) {
                    value = value.substring(1, value.length() - 1);
                }
                properties.put(key, value);
            }
        }
        Rule rule;
        rule = new Rule(selectors.toArray(new Selector[selectors.size()]), properties);
        return rule;
    }

    protected Stylesheet processStylesheet(String stylesheetText) throws CompilerException {
        CSSParser p = new CSSParser(new StringReader(stylesheetText));
        SimpleNode node;
        try {
            node = p.Stylesheet();
        } catch (Error e) {
            throw new CompilerException(e);
        }
        List<Rule> rules = new ArrayList<Rule>();
        for (int i = 0; i < node.jjtGetNumChildren(); i++) {
            SimpleNode ruleNode = node.getChild(i);
            Rule rule = processRule(ruleNode);
            rules.add(rule);
        }
        Stylesheet stylesheet;
        stylesheet = new Stylesheet(rules.toArray(new Rule[rules.size()]));
        return stylesheet;
    }
}
