/*
 * Copyright 2006 Ethan Nicholas. All rights reserved.
 * Use is subject to license terms.
 */
package jaxx.tags.swing;

import jaxx.CompilerException;
import jaxx.compiler.CompiledObject;
import jaxx.compiler.I18nHelper;
import jaxx.compiler.JAXXCompiler;
import jaxx.reflect.ClassDescriptorLoader;
import jaxx.runtime.swing.TabInfo;
import jaxx.tags.TagHandler;
import jaxx.types.TypeManager;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import javax.swing.Icon;
import javax.swing.JTabbedPane;
import java.awt.Color;
import java.io.IOException;

public class TabHandler implements TagHandler {

    public void compileFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        compileChildrenFirstPass(tag, compiler);
    }


    public void compileSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        if (!ClassDescriptorLoader.getClassDescriptor(JTabbedPane.class).isAssignableFrom(compiler.getOpenComponent().getObjectClass())) {
            compiler.reportError("tab tag may only appear within JTabbedPane tag");
            return;
        }

        JTabbedPaneHandler.CompiledTabbedPane tabs = (JTabbedPaneHandler.CompiledTabbedPane) compiler.getOpenComponent();

        String id = tag.getAttribute("id");
        if (id == null || id.length() == 0) {
            id = compiler.getAutoId(ClassDescriptorLoader.getClassDescriptor(TabInfo.class));
        }
        TabInfo tabInfo = new TabInfo(id);
        CompiledObject compiledTabInfo = new CompiledObject(id, ClassDescriptorLoader.getClassDescriptor(TabInfo.class), compiler);
        compiler.registerCompiledObject(compiledTabInfo);
        //id = tabInfo.getId();
        tabs.tabInfo = tabInfo;
        setAttributes(compiledTabInfo, tabs, tag, compiler);
        compileChildrenSecondPass(tag, compiler);
        tabs.tabInfo = null;
    }


    public static void setAttribute(CompiledObject compiledTabInfo, JTabbedPaneHandler.CompiledTabbedPane tabs, String name, String value, JAXXCompiler compiler) throws CompilerException {
        value = value.trim();
        TabInfo tabInfo = tabs.tabInfo;
        String id = tabInfo.getId();
        String binding = compiler.processDataBindings(value, ClassDescriptorLoader.getClassDescriptor(Object.class));
        if (binding != null) {
            compiler.registerDataBinding(binding, id + "." + name, id + ".set" + org.apache.commons.lang.StringUtils.capitalize(name) + "(" + binding + ");");
            return;
        }

        String valueCode = TypeManager.getJavaCode(value);

        // add i18n support
        if (I18nHelper.isI18nableAttribute(name, compiler)) {
            value = valueCode = I18nHelper.addI18nInvocation(id, name, valueCode, compiler);
        }

        if (name.equals("title")) {
            tabInfo.setTitle(value);
            compiledTabInfo.appendInitializationCode(id + ".setTitle(" + valueCode + ");");
            //compiledTabInfo.appendInitializationCode(id + ".setTitle(" + TypeManager.getJavaCode(value) + ");");
        } else if (name.equals("toolTipText")) {
            tabInfo.setToolTipText(value);
            compiledTabInfo.appendInitializationCode(id + ".setToolTipText(" + valueCode + ");");
            //compiledTabInfo.appendInitializationCode(id + ".setToolTipText(" + TypeManager.getJavaCode(value) + ");");
        } else if (name.equals("icon")) {
            Icon icon = (Icon) TypeManager.convertFromString(value, Icon.class);
            tabInfo.setIcon(icon);
            compiledTabInfo.appendInitializationCode(id + ".setIcon(" + TypeManager.getJavaCode(icon) + ");");
        } else if (name.equals("enabled")) {
            boolean enabled = (Boolean) TypeManager.convertFromString(value, Boolean.class);
            tabInfo.setEnabled(enabled);
            compiledTabInfo.appendInitializationCode(id + ".setEnabled(" + enabled + ");");
        } else if (name.equals("disabledIcon")) {
            Icon disabledIcon = (Icon) TypeManager.convertFromString(value, Icon.class);
            tabInfo.setDisabledIcon(disabledIcon);
            compiledTabInfo.appendInitializationCode(id + ".setDisabledIcon(" + TypeManager.getJavaCode(disabledIcon) + ");");
        } else if (name.equals("mnemonic")) {
            int mnemonic = (Character) TypeManager.convertFromString(value, char.class);
            tabInfo.setMnemonic(mnemonic);
            compiledTabInfo.appendInitializationCode(id + ".setMnemonic(" + mnemonic + ");");
        } else if (name.equals("displayedMnemonicIndex")) {
            int displayedMnemonicIndex = (Integer) TypeManager.convertFromString(value, int.class);
            tabInfo.setDisplayedMnemonicIndex(displayedMnemonicIndex);
            compiledTabInfo.appendInitializationCode(id + ".setDisplayedMnemonicIndex(" + displayedMnemonicIndex + ");");
        } else if (name.equals("foreground")) {
            Color foreground = (Color) TypeManager.convertFromString(value, Color.class);
            tabInfo.setForeground(foreground);
            compiledTabInfo.appendInitializationCode(id + ".setForeground(" + TypeManager.getJavaCode(foreground) + ");");
        } else if (name.equals("background")) {
            Color background = (Color) TypeManager.convertFromString(value, Color.class);
            tabInfo.setBackground(background);
            compiledTabInfo.appendInitializationCode(id + ".setBackground(" + TypeManager.getJavaCode(background) + ");");
        } else if (name.equals("id")) {
            // ignore, already handled
        } else {
            compiler.reportError("The <tab> tag does not support the attribute '" + name + "'");
        }
    }


    public void setAttributes(CompiledObject compiledTabInfo, JTabbedPaneHandler.CompiledTabbedPane tabs, Element tag, JAXXCompiler compiler) throws CompilerException {
        NamedNodeMap children = tag.getAttributes();
        for (int i = 0; i < children.getLength(); i++) {
            Attr attribute = (Attr) children.item(i);
            String name = attribute.getName();
            String value = attribute.getValue();
            if (!name.startsWith("xmlns") && !JAXXCompiler.JAXX_INTERNAL_NAMESPACE.equals(attribute.getNamespaceURI())) {
                setAttribute(compiledTabInfo, tabs, name, value, compiler);
            }
        }
    }


    protected void compileChildrenFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node node = children.item(i);
            int nodeType = node.getNodeType();
            if (nodeType == Node.ELEMENT_NODE) {
                Element child = (Element) node;
                compileChildTagFirstPass(child, compiler);
            } else if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
                String text = ((Text) node).getData().trim();
                if (text.length() > 0) {
                    compiler.reportError("tag '" + tag.getLocalName() + "' may not contain text ('" + ((Text) node).getData().trim() + "')");
                }
            }
        }
    }


    protected void compileChildTagFirstPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        compiler.compileFirstPass(tag);
    }


    protected void compileChildrenSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        NodeList children = tag.getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            Node node = children.item(i);
            int nodeType = node.getNodeType();
            if (nodeType == Node.ELEMENT_NODE) {
                Element child = (Element) node;
                compileChildTagSecondPass(child, compiler);
            } else if (nodeType == Node.TEXT_NODE || nodeType == Node.CDATA_SECTION_NODE) {
                String text = ((Text) node).getData().trim();
                if (text.length() > 0) {
                    compiler.reportError("tag '" + tag.getLocalName() + "' may not contain text ('" + ((Text) node).getData().trim() + "')");
                }
            }
        }
    }


    protected void compileChildTagSecondPass(Element tag, JAXXCompiler compiler) throws CompilerException, IOException {
        compiler.compileSecondPass(tag);
    }
}