/*
 * *##% Plugin maven pour i18n
 * Copyright (C) 2007 - 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%* */
package org.nuiton.i18n.plugin;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.codehaus.plexus.util.DirectoryScanner;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Locale;
import org.nuiton.util.PluginHelper;
import org.nuiton.util.PluginHelper.SortedProperties;

/**
 * Recupere les différents fichiers des parsers en un fichier de proprietes.
 *
 * @author julien
 * @goal get
 * @phase generate-resources
 */
public class Getter extends AbstractI18nPlugin {

    /*
     * (non-Javadoc)
     * @see org.apache.maven.plugin.AbstractMojo#execute()
     */
    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        init();

        if (!needGeneration()) {
            if (verbose) {
                getLog().info("Nothing to generate - all files are up to date.");
            }
            return;
        }
        if (!silent) {
            getLog().info("config - basedir : " + out.getAbsolutePath());
            getLog().info("config - locales     : " + Arrays.toString(locales));
        }

        try {
            File bundleGetters = new File(out.getAbsolutePath() + File.separatorChar + artifactId + ".properties");
            bundleGetters.createNewFile();

            DirectoryScanner ds = new DirectoryScanner();
            ds.setBasedir(out);
            ds.setIncludes(new String[]{"*.getter"});
            ds.scan();
            String[] files = ds.getIncludedFiles();

            // Fusion des fichiers propriétés des différents parsers
            for (String file : files) {
                long t0 = System.nanoTime();
                File bundleGetter = getGetterFile(out, file, false);
                concactProperties(bundleGetter, bundleGetters);
                if (genSrc) {
                    bundleGetter.delete();
                }
                if (!silent) {
                    getLog().info("import getter " + bundleGetter.getName() + " in " + PluginHelper.convertTime(System.nanoTime() - t0));
                }
            }

            // Création des bundles
            for (Locale locale : locales) {
                if (!silent && getLog().isDebugEnabled()) {
                    getLog().debug("generate bundle for locale " + locale);
                }
                File bundleOut = getI18nFile(out, artifactId, locale, false);
                copyFile(bundleGetters, bundleOut);
                if (!silent && verbose) {
                    getLog().info("generate bundle " + locale);
                }
            }

            bundleGetters.delete();
        } catch (IOException e) {
            getLog().error("File Error I/O ", e);
            throw new MojoFailureException("File Error I/O");
        }
    }

    /**
     * Concatene deux fichiers de proprietes
     *
     * @param in  le fichier entrant
     * @param out le fichier sortant
     * @throws IOException  si problème pendant la sauvegarde ou fichier non trouvé.
     */
    protected void concactProperties(File in, File out) throws IOException {
        SortedProperties propertiesIn = new SortedProperties(encoding).load(in);
        SortedProperties propertiesOut = new SortedProperties(encoding).load(out);
        propertiesOut.putAll(propertiesIn);
        propertiesOut.store(out);
    }
}
