/*
 * #%L
 * Pollen :: VoteCounting Api
 * $Id: AbstractVoteCountingStrategy.java 3710 2012-09-29 21:18:12Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.5/pollen-votecounting-api/src/main/java/org/chorem/pollen/votecounting/AbstractVoteCountingStrategy.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.votecounting;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.Sets;
import org.chorem.pollen.votecounting.model.ChoiceIdAble;
import org.chorem.pollen.votecounting.model.ChoiceScore;
import org.chorem.pollen.votecounting.model.GroupOfVoter;
import org.chorem.pollen.votecounting.model.GroupVoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteCountingResult;
import org.chorem.pollen.votecounting.model.VoteForChoice;
import org.chorem.pollen.votecounting.model.Voter;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Base abstract implementation of a {@link VoteCountingStrategy}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.4.5
 */
public abstract class AbstractVoteCountingStrategy implements VoteCountingStrategy {

    public static final BigDecimal ZERO_D = BigDecimal.valueOf(0.);

    protected final Comparator<VoteForChoice> voteValueComparator = new Comparator<VoteForChoice>() {

        @Override
        public int compare(VoteForChoice o1, VoteForChoice o2) {
            Double v1 = o1.getVoteValue();
            Double v2 = o2.getVoteValue();
            if (v1 == null) {
                v1 = Double.MAX_VALUE;
            }
            if (v2 == null) {
                v2 = Double.MAX_VALUE;
            }
            return v1.intValue() - v2.intValue();
        }
    };

    @Override
    public final GroupVoteCountingResult votecount(GroupOfVoter group) {

        Set<GroupOfVoter> groups = Sets.newHashSet();
        voteCount(group, groups);

        GroupVoteCountingResult result = GroupVoteCountingResult.newResult(
                group, groups);
        return result;
    }

    public Map<String, ChoiceScore> newEmptyChoiceScoreMap(Set<Voter> voters) {
        // get all choice Id
        Set<String> choiceIds = getAllChoiceIds(voters);

        Map<String, ChoiceScore> resultByChoice = Maps.newHashMap();

        // creates all empty result for choice
        for (String choiceId : choiceIds) {
            ChoiceScore choiceScore = ChoiceScore.newScore(choiceId, null);
            resultByChoice.put(choiceId, choiceScore);
        }
        return resultByChoice;
    }

    protected VoteCountingResult orderByValues(Collection<ChoiceScore> scores) {

        // get scores by score value
        Multimap<BigDecimal, ChoiceScore> map = Multimaps.index(
                scores, ChoiceScore.SCORE_BY_VALUE
        );

        // get all distinct score values
        List<BigDecimal> values = Lists.newArrayList(map.asMap().keySet());
        Collections.sort(values);
        Collections.reverse(values);

        // compute rank for each scores
        int rank = 0;
        for (BigDecimal value : values) {

            Collection<ChoiceScore> scoresForValue = map.get(value);
            for (ChoiceScore score : scoresForValue) {
                score.setScoreOrder(rank);
            }
            rank++;
        }
        // get all scores
        List<ChoiceScore> orderedScores = Lists.newArrayList(map.values());

        // sort them by their rank
        Collections.sort(orderedScores);

        // transform map of result to list of them (and sort them)
        VoteCountingResult result = VoteCountingResult.newResult(orderedScores);
        return result;
    }

    public Set<String> getAllChoiceIds(Set<Voter> voters) {
        ChoiceIdAble.ChoiceIdAbleById function = new ChoiceIdAble.ChoiceIdAbleById();
        Set<String> result = Sets.newHashSet();
        for (Voter voter : voters) {
            Iterable<String> transform = Iterables.transform(
                    voter.getVoteForChoices(), function);
            Iterables.addAll(result, transform);

        }
        return result;
    }

    public Map<Voter, List<Set<String>>> buildVoterSortedChoices(Set<Voter> voters) {

        Map<Voter, List<Set<String>>> voterSortedChoices = Maps.newHashMap();

        for (Voter voter : voters) {

            List<Set<String>> sortedChoices = sortVoteForChoices(
                    voter.getVoteForChoices());
            voterSortedChoices.put(voter, sortedChoices);
        }
        return voterSortedChoices;
    }

    public List<Set<String>> sortVoteForChoices(Set<VoteForChoice> voteForChoices) {
        // get sort vote for choices
        List<VoteForChoice> sortedChoices = Lists.newArrayList(voteForChoices);
        Collections.sort(sortedChoices, voteValueComparator);

        // build ranks
        List<Set<String>> result = Lists.newArrayList();

        Set<String> set = Sets.newHashSet();
        result.add(set);
        VoteForChoice lastVoteForChoice = null;
        for (VoteForChoice voteForChoice : sortedChoices) {
            if (lastVoteForChoice != null &&
                voteValueComparator.compare(lastVoteForChoice, voteForChoice) != 0) {

                // new rank found
                // register it
                result.add(set = Sets.newHashSet());
            }

            set.add(voteForChoice.getChoiceId());
            lastVoteForChoice = voteForChoice;
        }
        return result;
    }

    protected void voteCount(GroupOfVoter group, Set<GroupOfVoter> groups) {

        groups.add(group);

        // all childs of this group
        Set<Voter> voters = group.getVoters();

        // treat before all his group childs
        for (Voter voter : voters) {
            if (voter instanceof GroupOfVoter) {

                // treat group child before all
                voteCount((GroupOfVoter) voter, groups);
            }
        }

        // once here, all childs has been treated, can votecount this group
        VoteCountingResult voteCountingResult = votecount(voters);

        // store the result for this group
        group.setResult(voteCountingResult);
    }
}
