/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.pages.poll;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.tapestry5.BindingConstants;
import org.apache.tapestry5.annotations.Component;
import org.apache.tapestry5.annotations.IncludeJavaScriptLibrary;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Parameter;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.ResultDTO;
import org.chorem.pollen.business.dto.ResultListDTO;
import org.chorem.pollen.business.services.ServicePoll;
import org.chorem.pollen.business.services.ServicePollAccount;
import org.chorem.pollen.business.services.ServiceResults;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.common.PollType;
import org.chorem.pollen.common.VoteCountingType;
import org.chorem.pollen.ui.components.Chart;
import org.chorem.pollen.ui.components.FeedBack;
import org.chorem.pollen.ui.components.ImageContextLink;
import org.chorem.pollen.ui.data.Lien;
import org.chorem.pollen.ui.services.Configuration;
import org.chorem.pollen.votecounting.business.NumberMethod;
import org.chorem.pollen.votecounting.dto.VoteCountingResultDTO;
import org.slf4j.Logger;

/**
 * Classe de la page de rendu des résultats.
 *
 * @author rannou
 * @version $Id: Results.java 2864 2010-02-09 16:16:50Z schorlet $
 */
@IncludeStylesheet({"context:css/results.css", "context:css/lightbox.css"})
@IncludeJavaScriptLibrary({"${tapestry.scriptaculous}/builder.js","context:js/lightbox.js"})
public class Results {
    @Parameter(defaultPrefix = BindingConstants.MESSAGE, value = "title")
    @Property
    private String title;

    @SuppressWarnings("unused")
    @Property
    private Lien[] address;

    /** Paramètres de la page */
    private String param;

    /** Affichage des messages pour l'utilisateur */
    @Component(id = "feedback")
    private FeedBack feedback;

    /** Zone de résultats */
    @InjectComponent
    private Zone resultZone;

    /** Affichage détaillé ? */
    @Persist
    @Property
    private boolean byGroup;

    /** Indique si l'utilisateur est authorisé */
    @Property
    private boolean userAllowed = false;

    /** Type de représentation des résultats */
    @SuppressWarnings("unused")
    @Persist
    @Property
    private int type;

    /** Sondage concerné par les résultats */
    @Property
    @Persist
    private PollDTO poll;

    /** Créateur du sondage */
    @Property
    private PollAccountDTO creator;

    /** Résultats du sondage */
    @Persist
    @Property(write = false)
    private List<ResultDTO> results;

    /**
     * Résultats du sondage.
     * 
     * La map est indexée par les noms de choix.
     */
    @Property(write = false)
    private Map<String, List<String>> choicesResults;

    /** Résultat courant */
    @SuppressWarnings("unused")
    @Property
    private Map.Entry<String, List<String>> choicesResult;

    /** Résultats du sondage */
    @Persist
    @Property(write = false)
    private Map<String, List<String>> subtitles;

    /** Résultat courant */
    @Property
    private ResultDTO result;

    /** Choix courant (diagramme) */
    @SuppressWarnings("unused")
    @Property
    private List<String> choice;

    /** Localisation */
    @Inject
    private Messages messages;

    /** Page d'affichage d'une image */
    @InjectPage
    private ImageDisplay imageDisplay;

    /** Format des dates */
    @Persist
    @Property
    private DateFormat dateFormat;

    /** Locale courante */
    @Inject
    private Locale currentLocale;

    /**
     * Service contenant la configuration de l'application.
     */
    @Inject
    private Configuration conf;

    /**
     * Logs.
     */
    @Inject
    private Logger logger;

    /** Injection des services */
    @Inject
    private ServicePoll servicePoll;
    @Inject
    private ServicePollAccount servicePollAccount;
    @Inject
    private ServiceResults serviceResults;
    
    @InjectComponent
    private ImageContextLink imgContext;

    public ImageContextLink getImgContext() {
        return imgContext;
    }

    /** Récupération de la liste des résultats pour le diagramme. */
    public ArrayList<ArrayList<String>> getChoices() {
        ArrayList<ArrayList<String>> choices = new ArrayList<ArrayList<String>>();
        ArrayList<String> choiceValues = new ArrayList<String>();

        countPoll();
        for (ResultDTO result : results) {
            String name = result.getName();
            String value = result.getValue();

            if (poll.getChoiceType() == ChoiceType.DATE) { // mise en forme de la date
                Date date = new Date(Long.parseLong(name));
                name = dateFormat.format(date);
            }

            choiceValues.add(name);
            choiceValues.add(value);
        }

        choices.add(choiceValues);
        return choices;
    }

    /** Récupération de la liste des résultats pour le classement. */
    public List<ResultDTO> getRanking() {
        countPoll();
        List<ResultDTO> ranking = results;

        Collections.sort(ranking, new Comparator<ResultDTO>() {
            @Override
            public int compare(ResultDTO o1, ResultDTO o2) {
                Double result1 = Double.parseDouble(o1.getValue());
                Double result2 = Double.parseDouble(o2.getValue());
                int comp = 0; // résultat de la comparaison

                if (result1 > result2) {
                    comp = -1;
                } else if (result1 < result2) {
                    comp = 1;
                }
                if (comp == 0) {
                    comp = o1.getName().compareTo(o2.getName());
                }
                return comp;
            }
        });

        return ranking;
    }

    /** Récupération des résultats gagnants. */
    public List<ResultDTO> getTopRanking() {
        List<ResultDTO> ranking = getRanking();
        List<ResultDTO> winners = new ArrayList<ResultDTO>();

        String winValue = null;
        if (ranking.size() > 0) {
            winValue = ranking.get(0).getValue();
        } else {
            return winners;
        }

        for (ResultDTO r : ranking) {
            if (winValue.equals(r.getValue())) {
                winners.add(r);
            }
        }
        return winners;
    }

    /**
     * Changement de type de diagrammes
     *
     * @return la zone à mettre à jour
     */
    Object onType(int type) {
        this.type = type;
        return resultZone.getBody();
    }

    /**
     * Changement de type d'affichage (byGroup, normal)
     *
     * @return la zone à mettre à jour
     */
    Object onDisplayType() {
        byGroup = !byGroup;
        return resultZone.getBody();
    }

    public boolean isTextType() {
        return poll.getChoiceType() == ChoiceType.TEXT;
    }

    public boolean isDateType() {
        return poll.getChoiceType() == ChoiceType.DATE;
    }

    public boolean isImageType() {
        return poll.getChoiceType() == ChoiceType.IMAGE;
    }

    public boolean isFreePoll() {
        return poll.getPollType() == PollType.FREE;
    }

    public boolean isRestrictedPoll() {
        return poll.getPollType() == PollType.RESTRICTED;
    }

    public boolean isGroupPoll() {
        return poll.getPollType() == PollType.GROUP;
    }
    
    /** Retourne le message d'aide correspondant au type de sondage. */
    public String getHelpMessage() {
        switch (poll.getVoteCounting()) {
        case NORMAL:
            return messages.get("normalVote-help");
        case PERCENTAGE:
            return messages.get("percentageVote-help");
        case CONDORCET:
            return messages.get("condorcetVote-help");
        case NUMBER:
            return messages.get("numberVote-help");
        default:
            return "";
        }
    }

    /** Retourne le message de victoire indiquant le ou les gagnants. */
    public String getVictoryMessage() {
        if (poll.getVoteCounting() == VoteCountingType.NUMBER) {
            return null;
        }
        
        List<ResultDTO> winners = getTopRanking();

        if (winners.size() == 0) {
            return "";
        } else if (winners.size() == 1) {
            return messages.format("victory");
        } else {
            return messages.format("victories");
        }
    }

    /** Retourne la valeur du résultat courant sans le .0 final */
    public String getTrimValue() {
        String value = result.getValue();

        // le résultat peut-être un double : 1,0 -> 1 et 1,2 -> 1,2
        if (value.endsWith(".0")) {
            value = value.substring(0, value.indexOf('.'));
        }
        return value;
    }

    /** Retourne la date correspondant au résultat courant */
    public Date getResultNameAsDate() {
        return new Date(Long.valueOf(result.getName()));
    }

    /** Action réalisée lorsqu'on clique sur l'image */
    Object onDisplayImage(String resultName) {
        String choiceId = "";

        // Récupération du choix correspondant au résultat
        for (ChoiceDTO choice : poll.getChoiceDTOs()) {
            if (resultName.equals(choice.getName())) {
                choiceId = choice.getId();
            }
        }

        imageDisplay.setPoll(poll);
        imageDisplay.setChoiceId(choiceId);
        imageDisplay.setPageStyle("VoteCounting");
        return imageDisplay;
    }

    /** Dépouillement du sondage. Mise à jour des résultats. */
    private void countPoll() {
        ResultListDTO resultListDTO = null;
        
        if (byGroup) {
            resultListDTO = serviceResults.getGroupResults(poll.getPollId());
        } else {
            resultListDTO = serviceResults.getNormalResults(poll.getPollId());
        }

        results = resultListDTO.getResultDTOs();
        computeResults(resultListDTO);
        
        for (ResultDTO res : results) {
            logger.debug(res.getName() + ": " + res.getValue()
                    + ", (voteCounting=" + res.getVoteCounting() + ", byGroup="
                    + res.isByGroup() + ")");
        }
    }

    private void computeResults(ResultListDTO resultListDTO) {
        if (poll.getVoteCounting() == VoteCountingType.NUMBER) {
            // resultats des choix cachés
            choicesResults = new LinkedHashMap<String, List<String>>();
            
            // sous-titres des vrais choix (les choix pas cachés)
            subtitles = new LinkedHashMap<String, List<String>>();
            
            /*
             *  liste des vrais choix (les choix pas cachés).
             *  cette variable sera affectée à la variable results
             */
            List<ResultDTO> results2 = new ArrayList<ResultDTO>();
            
            // Ajout des résultats des choix cachés
            for (ResultDTO result : results) {
                if (result.isHidden()) {
                    String name = result.getName();
                    int indexOf = name.indexOf('#');
                    String choice = name.substring(
                            NumberMethod.HIDDEN_PREFIX.length(), indexOf);
                    
                    List<String> votes = choicesResults.get(choice);
                    if (votes == null) {
                        votes = new ArrayList<String>();
                    }

                    String votingId = name.substring(indexOf + 1);
                    String value = result.getValue();

                    votes.add(votingId);
                    votes.add(value);
                    
                    choicesResults.put(choice, votes);
                    
                } else {
                    results2.add(result);
                }
            }
            
            Set<Entry<String, List<String>>> entries = choicesResults.entrySet();
            for (Entry<String, List<String>> entry : entries) {
                String choiceName = entry.getKey();
                List<String> values = entry.getValue();
                
                // Récupération du choix correspondant au résultat
                VoteCountingResultDTO voteCountingResultDTO = resultListDTO
                        .getVoteCountingResultDTO();
                
                for (org.chorem.pollen.votecounting.dto.ChoiceDTO choice :
                            voteCountingResultDTO.getChoices()) {
                    if (choiceName.equals(choice.getName())) {
                        List<String> choiceSubtitles = subtitles.get(choice);
                        if (choiceSubtitles == null) {
                            choiceSubtitles = new ArrayList<String>();
                        }
                        
                        choiceSubtitles.add(messages.get("numberVote-total"));
                        choiceSubtitles.add(String.valueOf(choice.getValue()));
                        
                        choiceSubtitles.add(messages.get("numberVote-average"));
                        choiceSubtitles.add(String.valueOf(choice.getAverage()));
                        
                        choiceSubtitles.add(messages.get("numberVote-blank-votes"));
                        choiceSubtitles.add(String.valueOf(choice.getNbBlankVotes()));
                        
                        choiceSubtitles.add(messages.get("numberVote-total-votes"));
                        choiceSubtitles.add(String.valueOf(values.size() / 2));

                        subtitles.put(choiceName, choiceSubtitles);
                        break;
                    }
                }
                
            }
            
            results = results2;
            if (results.size() == 1) {
                results.remove(0);
            }
        }
    }
    
    /** Initialisation des objets de session */
    void onActivate(String id) {
        param = id;
        String pollId = null;

        if (id != null && !"".equals(id)) {
            pollId = id.split(":", 2)[0];
            poll = servicePoll.findPollByPollId(pollId);

            if (poll != null) {

                // Identification de l'utilisateur
                creator = servicePollAccount.findPollAccountById(poll
                        .getCreatorId());
                if (poll.isPublicResults()) {
                    userAllowed = true;
                } else if (id.split(":", 2).length == 2) {
                    String creatorId = id.split(":", 2)[1];
                    if (creatorId.equals(creator.getAccountId())) {
                        userAllowed = true;
                    }
                }
            }
        }

        // Affichage des erreurs
        if (poll == null) {
            feedback.addError(messages.get("pollNotFound"));
        } else if (!userAllowed) {
            feedback.addError(messages.get("userNotAllowed"));
        } else if (!poll.isClosed()) {
            feedback.addError(messages.get("pollNotClosed"));
        }
    }

    /**
     * Méthode appelée au moment de la désactivation de la page
     *
     * @return l'identifiant du sondage et du créateur
     */
    String onPassivate() {
        return param;
    }

    /**
     * Initialisation de l'affichage
     */
    void setupRender() {
        address = new Lien[] { new Lien("Pollen", "Index"),
                new Lien(title, null) };
        dateFormat = DateFormat.getDateTimeInstance(DateFormat.SHORT,
                DateFormat.SHORT, currentLocale);

        type = Chart.PIE;
        byGroup = isGroupPoll();
    }
}
