/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.data;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.chorem.pollen.business.dto.ChoiceDTO;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.common.ChoiceType;
import org.chorem.pollen.ui.data.uio.DateChoiceUIO;
import org.chorem.pollen.ui.data.uio.ImageChoiceUIO;

/**
 * Gestion de la copie d'un sondage existant pour en créer un nouveau.
 *
 * @author rannou
 * @version $Id: PollHelper.java 2688 2009-08-03 15:16:17Z nrannou $
 */
public class PollHelper {

    /**
     * Création d'un nouveau sondage à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return le nouveau sondage
     */
    public static PollDTO getPoll(PollDTO oldPoll) {
        PollDTO newPoll = new PollDTO();
        newPoll.setTitle(oldPoll.getTitle());
        newPoll.setDescription(oldPoll.getDescription());
        newPoll.setBeginChoiceDate(oldPoll.getBeginChoiceDate());
        newPoll.setBeginDate(oldPoll.getBeginDate());
        newPoll.setEndDate(oldPoll.getEndDate());
        newPoll.setAnonymous(oldPoll.isAnonymous());
        newPoll.setAnonymousVoteAllowed(oldPoll.isAnonymousVoteAllowed());
        newPoll.setPublicResults(oldPoll.isPublicResults());
        newPoll.setContinuousResults(oldPoll.isContinuousResults());
        newPoll.setChoiceAddAllowed(oldPoll.isChoiceAddAllowed());
        newPoll.setMaxChoiceNb(oldPoll.getMaxChoiceNb());
        newPoll.setChoiceType(oldPoll.getChoiceType());
        newPoll.setPollType(oldPoll.getPollType());
        newPoll.setVoteCounting(oldPoll.getVoteCounting());
        return newPoll;
    }

    /**
     * Création de listes de votants à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return les listes de votants
     */
    public static List<VotingListDTO> getVotingLists(PollDTO oldPoll) {
        List<VotingListDTO> votingLists = new ArrayList<VotingListDTO>();
        for (VotingListDTO oldList : oldPoll.getVotingListDTOs()) {
            VotingListDTO votingList = new VotingListDTO();
            votingList.setName(oldList.getName());
            votingList.setWeight(oldList.getWeight());
            votingList.setPollAccountDTOs(getPollAccounts(oldList));
            votingLists.add(votingList);
        }
        return votingLists;
    }

    private static List<PollAccountDTO> getPollAccounts(VotingListDTO oldList) {
        List<PollAccountDTO> pollAccounts = new ArrayList<PollAccountDTO>();
        for (PollAccountDTO oldAccount : oldList.getPollAccountDTOs()) {
            PollAccountDTO pollAccount = new PollAccountDTO();
            pollAccount.setVotingId(oldAccount.getVotingId());
            pollAccount.setEmail(oldAccount.getEmail());
            pollAccount.setWeight(oldAccount.getWeight());
            pollAccounts.add(pollAccount);
        }
        return pollAccounts;
    }

    /**
     * Création de choix de type Texte à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return les choix
     */
    public static List<ChoiceDTO> getTextChoices(PollDTO oldPoll) {
        List<ChoiceDTO> choices = new ArrayList<ChoiceDTO>();
        if (oldPoll.getChoiceType() == ChoiceType.TEXT) {
            for (ChoiceDTO oldChoice : oldPoll.getChoiceDTOs()) {
                ChoiceDTO choice = new ChoiceDTO();
                choice.setName(oldChoice.getName());
                choice.setDescription(oldChoice.getDescription());
                choices.add(choice);
            }
        }
        return choices;
    }

    /**
     * Création de choix de type Date à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return les choix
     */
    public static List<DateChoiceUIO> getDateChoices(PollDTO oldPoll) {
        List<DateChoiceUIO> choices = new ArrayList<DateChoiceUIO>();
        if (oldPoll.getChoiceType() == ChoiceType.DATE) {
            for (ChoiceDTO oldChoice : oldPoll.getChoiceDTOs()) {
                DateChoiceUIO choice = new DateChoiceUIO();
                choice.setName(oldChoice.getName());
                choice.setDescription(oldChoice.getDescription());
                Date date = new Date(Long.valueOf(choice.getName()));
                choice.setDate(date);
                choices.add(choice);
            }
        }
        return choices;
    }

    /**
     * Création de choix de type Image à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return les choix
     */
    public static List<ImageChoiceUIO> getImageChoices(PollDTO oldPoll) {
        List<ImageChoiceUIO> choices = new ArrayList<ImageChoiceUIO>();
        if (oldPoll.getChoiceType() == ChoiceType.IMAGE) {
            /*for (ChoiceDTO oldChoice : oldPoll.getChoiceDTOs()) {
                ImageChoiceUIO choice = new ImageChoiceUIO();
                choice.setName(oldChoice.getName());
                choice.setDescription(oldChoice.getDescription());
                //TODO copie des images de l'ancien sondage et initialisation UploadedFile
                choices.add(choice);
            }*/
        }
        return choices;
    }

    /**
     * Création de la règle de notification à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return la règle
     */
    public static PreventRuleDTO getNotificationPreventRule(PollDTO oldPoll) {
        PreventRuleDTO preventRule = new PreventRuleDTO();
        for (PreventRuleDTO oldRule : oldPoll.getPreventRuleDTOs()) {
            if ("vote".equals(oldRule.getScope())) {
                preventRule.setScope(oldRule.getScope());
                preventRule.setSensibility(oldRule.getSensibility());
                preventRule.setRepeated(oldRule.isRepeated());
                preventRule.setMethod(oldRule.getMethod());
            }
        }
        return preventRule;
    }

    /**
     * Création de la règle de rappel à partir d'un sondage existant.
     *
     * @param oldPoll le sondage existant
     * @return la règle
     */
    public static PreventRuleDTO getReminderPreventRule(PollDTO oldPoll) {
        PreventRuleDTO preventRule = new PreventRuleDTO();
        for (PreventRuleDTO oldRule : oldPoll.getPreventRuleDTOs()) {
            if ("rappel".equals(oldRule.getScope())) {
                preventRule.setScope(oldRule.getScope());
                preventRule.setSensibility(oldRule.getSensibility());
                preventRule.setRepeated(oldRule.isRepeated());
                preventRule.setMethod(oldRule.getMethod());
            }
        }
        return preventRule;
    }
}
