/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.services;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tapestry5.ioc.Messages;
import org.chorem.pollen.business.business.PreventRuleManager;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.dto.PollDTO;
import org.chorem.pollen.business.dto.PreventRuleDTO;
import org.chorem.pollen.business.dto.VotingListDTO;
import org.chorem.pollen.business.services.ServicePoll;
import org.chorem.pollen.business.services.ServicePreventRule;
import org.chorem.pollen.business.services.ServicePreventRuleImpl;

/**
 * Implémentation du service BackgroundWorker.
 *
 * @author rannou
 * @version $Id: BackgroundWorkerImpl.java 2736 2009-08-19 15:48:01Z nrannou $
 */
public class BackgroundWorkerImpl implements BackgroundWorker {

    /** Timer gérant les threads. */
    private Timer timer;

    /** log. */
    private static final Log log = LogFactory
            .getLog(BackgroundWorkerImpl.class);

    /** Messages internationalisés **/
    private final Messages messages;

    /** Service contenant la configuration de l'application. */
    private Configuration conf;

    /** Injection du service de gestion de sondages. */
    private ServicePoll servicePoll;

    /**
     * Constructeur exécuté au lancement de l'application.
     */
    public BackgroundWorkerImpl(Messages messages, Configuration conf,
            ServicePoll servicePoll) {
        this.messages = messages;
        this.conf = conf;
        this.servicePoll = servicePoll;

        timer = new Timer();
        executeTasks();
    }

    /**
     * Lancement des tâches en arrière-plan.
     */
    public void executeTasks() {
        if ("".equals(conf.getProperty("siteUrl"))) {
            log
                    .warn("No property siteUrl. Reminder emails would not contain it");
        }

        // Exécution de la tâche toutes les 10 min
        timer.schedule(new MailingTask(), 600000, 600000);
    }

    /**
     * Tâche gérant l'envoi de mails de rappels.
     */
    private class MailingTask extends TimerTask {

        @Override
        public void run() {
            if (log.isDebugEnabled()) {
                log.debug("MailingTask running...");
            }

            // Récupération des sondages en cours
            List<PollDTO> polls = servicePoll.findRunningPolls(true);

            for (PollDTO poll : polls) {
                Date now = new Date();

                // Parcours des preventRules de chaque sondage
                // envoi d'un email si endDate-nowDate <= preventRuleSensibility
                if (poll.getEndDate() != null) {

                    // timeValue = endDate-nowDate
                    Long timeValue = poll.getEndDate().getTime()
                            - now.getTime();

                    if (log.isDebugEnabled()) {
                        log.debug("Now: " + now.getTime() + " End: "
                                + poll.getEndDate().getTime() + " timeValue: "
                                + timeValue + "(" + timeValue / 3600000 + "h)");
                    }

                    // envoi des mails avec une sensibilité de l'ordre de l'heure
                    sendMailNotification(poll, timeValue / 3600000);
                }
            }
        }
    }

    /**
     * Envoi du mail de notification.
     *
     * @param poll sondage concerné
     */
    private void sendMailNotification(PollDTO poll, Long timeValue) {
        String voteURL = conf.getProperty("siteUrl") + "/poll/VoteFor/"
                + poll.getPollId();
        Map<String, String> data = new HashMap<String, String>();
        data.put("host", conf.getProperty("email_host"));
        data.put("port", conf.getProperty("email_port"));
        data.put("from", conf.getProperty("email_from"));

        // Mails aux votants
        for (PreventRuleDTO rule : poll.getPreventRuleDTOs()) {
            if (rule.isActive()) {
                Boolean hasRun = false;
                PreventRuleManager manager = new PreventRuleManager(rule);
                for (VotingListDTO list : poll.getVotingListDTOs()) {
                    for (PollAccountDTO account : list.getPollAccountDTOs()) {
                        if (account.getEmail() != null) {
                            String accountVoteURL = voteURL + ":"
                                    + account.getAccountId();

                            data.put("to", account.getEmail());
                            data.put("title", messages.format(
                                    "reminderEmail_subject", poll.getTitle()));
                            data.put("msg", messages.format(
                                    "reminderEmail_msg", poll.getTitle(),
                                    account.getVotingId(), accountVoteURL));

                            // Exécution de la règle avec une valeur +1
                            // Sensibilité de 2h : 2h05 KO, 1h55 OK
                            hasRun = manager.execute("rappel", timeValue
                                    .intValue() + 1, data);
                        }
                    }
                }

                // désactivation de la règle de rappel pour éviter qu'elle soit réexécutée
                if (hasRun) {
                    rule.setActive(false);
                    ServicePreventRule spr = new ServicePreventRuleImpl();
                    spr.updatePreventRule(rule);
                }
            }
        }
    }
}
