/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.ui.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.directory.Attribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.dto.PollAccountDTO;

/**
 * Classe permettant de gérer l'import de listes de votants à partir d'un
 * serveur LDAP.
 *
 * @author rannou
 * @version $Id: LDAPAccountUtil.java 2731 2009-08-18 12:22:46Z nrannou $
 */
public class LDAPAccountUtil {

    /** log. */
    private static final Log log = LogFactory.getLog(LDAPAccountUtil.class);

    /**
     * Import d'une liste de votants à partir d'un serveur LDAP.
     *
     * @param url l'URL correspondant à la recherche
     * @return la liste de votants
     */
    public static List<PollAccountDTO> importList(String url) {
        return importList(null, url, null);
    }

    /**
     * Import d'une liste de votants à partir d'un serveur LDAP.
     *
     * @param server l'adresse du serveur
     * @param context le nom du contexte pour la recherche
     * @param filter le filtre à apposer pour la recherche
     * @return la liste de votants
     */
    public static List<PollAccountDTO> importList(String server,
            String context, String filter) {
        long start = System.nanoTime();
        List<PollAccountDTO> accounts = new ArrayList<PollAccountDTO>();

        try {

            // Initialisation du contexte
            Properties env = new Properties();
            if (server != null) {
                env.put(Context.INITIAL_CONTEXT_FACTORY,
                        "com.sun.jndi.ldap.LdapCtxFactory");
                env.put(Context.PROVIDER_URL, "ldap://" + server + "/");
            }
            DirContext ictx = new InitialDirContext(env);

            // Recherche en profondeur
            SearchControls control = new SearchControls();
            control.setSearchScope(SearchControls.SUBTREE_SCOPE);

            // Création des comptes avec les résultats de la recherche
            NamingEnumeration<SearchResult> e = ictx.search(context, filter,
                    control);
            while (e.hasMore()) {
                SearchResult r = e.next();

                if (log.isDebugEnabled()) {
                    log.debug("Result: " + r.getName() + "(object: "
                            + r.getClassName() + ")");
                }

                Attribute nameAttr = r.getAttributes().get("cn");
                Attribute emailAttr = r.getAttributes().get("mail");

                if (nameAttr != null) {
                    PollAccountDTO account = new PollAccountDTO();
                    account.setVotingId(nameAttr.get().toString());
                    account.setEmail(emailAttr.get().toString());
                    accounts.add(account);

                    if (log.isDebugEnabled()) {
                        log.debug("New account - name: "
                                + nameAttr.get().toString() + ", email: "
                                + emailAttr.get().toString());
                    }
                }
            }
        } catch (javax.naming.NamingException e) {
            log.error("Exception de nommage lors de l'import depuis LDAP", e);
        }

        long duration = (System.nanoTime() - start) / 1000000000;
        if (log.isInfoEnabled()) {
            log.info(accounts.size() + " comptes importés en " + duration
                    + " sec.");
        }

        return accounts;
    }
}
