/*
 * #%L
 * Pollen :: UI (struts2)
 * $Id: AbstractPollForm.java 3715 2012-10-01 05:11:10Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.4/pollen-ui-struts2/src/main/java/org/chorem/pollen/ui/actions/poll/form/AbstractPollForm.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.ui.actions.poll.form;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.Preparable;
import com.opensymphony.xwork2.interceptor.annotations.InputConfig;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.interceptor.ParameterAware;
import org.chorem.pollen.PollenTechnicalException;
import org.chorem.pollen.bean.PollDateChoice;
import org.chorem.pollen.bean.PollImageChoice;
import org.chorem.pollen.bean.PollUri;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceImpl;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.I18nAble;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListImpl;
import org.chorem.pollen.business.persistence.Poll;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountImpl;
import org.chorem.pollen.business.persistence.PollCommentVisibility;
import org.chorem.pollen.business.persistence.PollType;
import org.chorem.pollen.business.persistence.PollVoteVisibility;
import org.chorem.pollen.business.persistence.PreventRule;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListImpl;
import org.chorem.pollen.services.PollenServiceFunctions;
import org.chorem.pollen.services.exceptions.PollNotFoundException;
import org.chorem.pollen.services.impl.PollService;
import org.chorem.pollen.services.impl.PreventRuleService;
import org.chorem.pollen.ui.actions.FileUploadAware;
import org.chorem.pollen.ui.actions.PollenActionSupportForEdition;
import org.chorem.pollen.ui.converters.DateConverter;
import org.chorem.pollen.votecounting.VoteCounting;
import org.chorem.pollen.votecounting.VoteCountingFactory;
import org.nuiton.util.StringUtil;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Created: 04/04/12
 *
 * @author fdesbois <desbois@codelutin.com>
 *         $Id: AbstractPollForm.java 3715 2012-10-01 05:11:10Z tchemit $
 */
public abstract class AbstractPollForm extends PollenActionSupportForEdition implements Preparable, ParameterAware, FileUploadAware {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(AbstractPollForm.class);

    public static final String IMAGECHOICES_THUMB_PREFIX = "imagechoicesThumb_";

    private static final Pattern TEXT_CHOICE_NAME_PATTERN =
            Pattern.compile("textChoice_(\\d+)\\.name");

    private static final Pattern DATE_CHOICE_NAME_PATTERN =
            Pattern.compile("dateChoice_(\\d+)\\.name");

    private static final Pattern IMAGE_CHOICE_DESCRIPTION_PATTERN =
            Pattern.compile("imageChoice_(\\d+)\\.description");

    protected Poll poll;

    private Map<String, String> pollTypes;

    private Map<String, String> voteCountingTypes;

    private Map<String, String> voteCountingTypesHelp;

    private Map<String, String> choiceTypes;

    private Map<String, String> pollVoteVisibilities;

    /** Text choices. */
    private List<Choice> textChoices;

    /** Image choices. */
    private List<Choice> imageChoices;

    /** Date choices. */
    private List<Choice> dateChoices;

    /** restricted Voting lists. */
    private List<VotingList> restrictedVotingList;

    /** Group voting lists. */
    private List<VotingList> groupVotingList;

    private boolean notification;

    private boolean reminder;

    private boolean limitChoice;

    private Integer maxChoices;

    private Integer reminderHourCountdown;

    /** To create a new personToList. */
    private transient Function<PersonToList, PersonToList> persontoListCreator;

    /** To create a new votingList. */
    private transient Function<VotingList, VotingList> votingListCreator;

    /** To create a new text choice. */
    private transient Function<Choice, Choice> textChoiceCreator;

    /** To create a new date choice. */
    private transient Function<Choice, Choice> dateChoiceCreator;

    /** To create a new image choice. */
    private transient Function<Choice, Choice> imageChoiceCreator;

    /**
     * Flag when there is some errors on the information panel.
     *
     * @since 1.3
     */
    protected boolean informationsError;

    /**
     * Flag when there is some errors on the options panel.
     *
     * @since 1.3
     */
    private boolean optionsError;

    /**
     * Indexed choices  retreive from parameters for the choiceType selected in
     * form. (Other choices are lost).
     *
     * @since 1.3
     */
    private Map<Integer, Choice> choices;

    /**
     * Contains the order of choices (given by the xxx.order field.
     *
     * @since 1.4
     */
    private Map<Integer, Integer> choicesOrder;

    /**
     * Indexed voting lists retreive from parameters for the pollType selected
     * in form. (Other voting lists are lost).
     *
     * @since 1.3
     */
    private Map<Integer, VotingList> votingLists;

    private PollUri pollUri;

    public abstract boolean isClone();

    protected abstract Poll savePoll(Poll poll) throws PollNotFoundException;

    public abstract boolean isEdit();

    @Override
    public void prepare() throws Exception {

        prepareFormPage();

        if (isGetMethod()) {

            // do not prepare when coming as GET
            return;
        }

        // we are after a submit

        String pollUid = getNonEmptyParameterValue("poll.pollId");
        String userId = getNonEmptyParameterValue("userId");
        if (StringUtils.isBlank(userId)) {

            // try to take the one from current user
            UserAccount userAccount = getPollenUserAccount();
            if (userAccount != null) {
                userId = userAccount.getTopiaId();
            }
        }
        UserAccount userAccount = null;
        if (StringUtils.isNotBlank(userId)) {
            // load use account to use
            userAccount = getPollService().getEntityById(
                    UserAccount.class, userId);
        }

        // get a copy (or a fresh new poll)
        poll = getPollService().getPollEditable(pollUid, userAccount, false);

        // If vote is started, prepare choices and votingLists is useless
        // because they can't be updated.
        if (!isVoteStarted()) {

            // Retrieve choiceType from parameters, the poll object will be updated after prepare
            String choiceTypeParam = getNonEmptyParameterValue("poll.choiceType");
            ChoiceType pollChoiceType = ChoiceType.valueOf(choiceTypeParam);
            poll.setChoiceType(pollChoiceType);

            choicesOrder = Maps.newTreeMap();

            // build poll choices

            switch (pollChoiceType) {

                case TEXT:
                    choices = buildTextChoices();
                    break;
                case DATE:
                    choices = buildDateChoices();
                    break;
                case IMAGE:
                    choices = buildImageChoices();
                    break;
            }
            PollType pollType;
            String pollTypeParam = getNonEmptyParameterValue("poll.pollType");
            pollType = PollType.valueOf(pollTypeParam);
            switch (pollType) {

                case FREE:

                    // empty voting list
                    votingLists = Maps.newTreeMap();
                    break;
                case RESTRICTED:

                    // restricted voting list
                    votingLists = buildVotingLists(pollType);
                    break;
                case GROUP:

                    // group voting lists
                    votingLists = buildVotingLists(pollType);
                    break;
            }
        }
    }

    @Override
    public String input() throws Exception {

        UserAccount userAccount = getPollenUserAccount();
        String pollUid = pollUri == null ? null : pollUri.getPollId();
        PollService service = getPollService();

        poll = service.getPollEditable(pollUid, userAccount, isClone());

        if (poll.isClosed()) {
            addFlashWarning(_("pollen.warning.poll.is.closed.so.read.only"));
        }
        List<Choice> pollChoices = poll.getChoice();

        if (isClone()) {

            if (ChoiceType.IMAGE == poll.getChoiceType()) {

                // recopy images to tmp

                File tmpDir = getConfiguration().getTemporaryDirectory();
                for (Choice choice : pollChoices) {
                    String choiceName = choice.getName();

                    // image from poll to clone
                    File imageChoiceFile = service.getPollChoiceImageFile(
                            pollUid,
                            choiceName
                    );

                    // new image
                    File newImageChoiceFile = File.createTempFile(
                            choiceName, null, tmpDir);

                    if (log.isInfoEnabled()) {
                        log.info("Copy image from " + imageChoiceFile +
                                 " to " + newImageChoiceFile);
                    }
                    FileUtils.copyFile(imageChoiceFile,
                                       newImageChoiceFile);

                    ((PollImageChoice) choice).setLocation(
                            newImageChoiceFile.getAbsolutePath());
                }
            }
        }

        List<VotingList> pollVotingLists = poll.getVotingList();

        loadChoicesAndvotingLists(poll,
                                  pollChoices,
                                  pollVotingLists,
                                  !isVoteStarted());

        setLimitChoice(poll.getMaxChoiceNb() > 0);

        if (isLimitChoice()) {
            setMaxChoices(poll.getMaxChoiceNb());
        } else {
            // set default max choices
            setMaxChoices(1);
        }
        PreventRule reminder = poll.getPreventRuleByScope(
                PreventRuleService.SCOPE_REMINDER);
        if (reminder != null) {
            setReminder(true);
            setReminderHourCountdown(reminder.getSensibility());
        } else {
            setReminder(false);
            // set default reminderHourCountdown
            setReminderHourCountdown(2);
        }

        PreventRule notification = poll.getPreventRuleByScope(
                PreventRuleService.SCOPE_VOTE);
        if (notification != null) {
            setNotification(true);
        }

        if (isVoteStarted()) {
            addFlashMessage(_("pollen.information.poll.form.voteStarted"));
        }

        return INPUT;
    }

    public String inputAfterValidationError() throws Exception {

        Collection<Choice> pollChoices =
                isVoteStarted() ? poll.getChoice() : choices.values();
        Collection<VotingList> pollVotingLists =
                isVoteStarted() ? poll.getVotingList() : votingLists.values();

        if (ChoiceType.IMAGE == poll.getChoiceType()) {

            // recopy images for new choices, the one uploaded will be
            // destroyed, by the upload interceptor

            File tmpDir = getConfiguration().getTemporaryDirectory();

            for (Choice choice : pollChoices) {

                PollImageChoice imageChoice = (PollImageChoice) choice;

                String choiceId = choice.getTopiaId();
                String location = imageChoice.getLocation();

                boolean needCopy = false;
                if (!isImageUploadEmpty(imageChoice)) {

                    // there is an image in the choice

                    if (StringUtils.isBlank(choiceId)) {

                        // this is a new choice
                        needCopy = true;
                    } else {

                        // check that name has not changed
                        Choice oldChoice = poll.getChoiceByTopiaId(choiceId);

                        needCopy = ObjectUtils.notEqual(oldChoice.getName(), imageChoice.getName());
                    }
                }

                if (needCopy) {

                    // not persisted choice with an upload
                    // let's copy it somewhere safe

                    File uploadedImage = new File(location);
                    File copyImage = File.createTempFile(
                            uploadedImage.getName(), null, tmpDir);

                    if (log.isInfoEnabled()) {
                        log.info("Copy image from " + uploadedImage +
                                 " to " + copyImage);
                    }
                    FileUtils.copyFile(uploadedImage, copyImage);

                    imageChoice.setLocation(copyImage.getAbsolutePath());

                    // generate also the thumb file
                    getPollService().generateThumbIfNeeded(copyImage);
                }
            }
        }

        loadChoicesAndvotingLists(poll, pollChoices, pollVotingLists, true);
        return INPUT;
    }

    @Override
    @InputConfig(methodName = "inputAfterValidationError")
    public String execute() throws Exception {

        if (isGetMethod()) {

            // input method
            return input();
        }

        // Save choices and votingLists only if vote is not started
        if (!isVoteStarted()) {
            //TODO-tchemit comment me 2012-06-04 A merge would be nicer but more complex to code
            // Clear previous collections to save those from the form
            poll.clearChoice();
            poll.clearVotingList();
            for (Integer index : choices.keySet()) {
                Choice choice = choices.get(index);
                poll.addChoice(choice);
            }

            if (!poll.isPollFree()) {

                for (Integer index : votingLists.keySet()) {
                    VotingList votingList = votingLists.get(index);
                    poll.addVotingList(votingList);
                }
            }
        }

        if (isLimitChoice()) {

            // push back filled value
            poll.setMaxChoiceNb(getMaxChoices());
        } else {
            // reset value
            poll.setMaxChoiceNb(0);
        }

        if (poll.isAnonymous()) {

            // force to not have choice to vote anonymously
            poll.setAnonymousVoteAllowed(false);
        }

        //TODO-tchemit comment me 2012-06-04 A merge would be nicer but more complex to code
        poll.clearPreventRule();

        PreventRuleService preventRuleService = getPreventRuleService();
        if (isNotification()) {

            // add a notification rule

            PreventRule rule = preventRuleService.createAddVotePreventRule();
            poll.addPreventRule(rule);
        }

        if (isReminder()) {

            // add a reminder rule

            PreventRule rule = preventRuleService.createRemindPreventRule(
                    getReminderHourCountdown()
            );

            poll.addPreventRule(rule);
        }

        // do save poll
        poll = savePoll(poll);

        // compute fresh pollUri
        pollUri = PollUri.newPollUri(poll.getAdminId());

        // remove all stuff from session
        getPollenSession().clearDynamicData();

        return SUCCESS;
    }

    @Override
    public void validate() {

        if (isGetMethod()) {

            // input method
            return;
        }
        validateInformations();

        validateOptions();
    }

    @Override
    public void addFile(int index, File file) {
        getParameters().put("imageChoice_" + index + ".newLocation",
                            new String[]{file.getAbsolutePath()});
    }

    @Override
    public void addFileContentType(int index, String contentType) {
        getParameters().put("imageChoice_" + index + ".newContentType",
                            new String[]{contentType});
        // not used here
    }

    @Override
    public void addFileName(int index, String fileName) {
        getParameters().put("imageChoice_" + index + ".newName", new String[]{fileName});
    }

    public boolean isInformationsError() {
        return informationsError;
    }

    public boolean isOptionsError() {
        return optionsError;
    }

    public int getSelectedTab() {
        int result;
        if (isInformationsError()) {
            result = 0;
        } else {
            if (isOptionsError()) {
                result = 1;
            } else {
                result = 0;
            }
        }
        return result;
    }

    @Override
    public PollUri getPollUri() {
        return pollUri;
    }

    public void setPollUri(PollUri pollUri) {
        this.pollUri = pollUri;
    }

    public Poll getPoll() {
        return poll;
    }

    public Map<String, String> getPollTypes() {
        return pollTypes;
    }

    public Map<String, String> getVoteCountingTypes() {
        return voteCountingTypes;
    }

    public Map<String, String> getChoiceTypes() {
        return choiceTypes;
    }

    public Map<String, String> getPollVoteVisibilities() {
        return pollVoteVisibilities;
    }

    public List<Choice> getTextChoices() {
        return textChoices;
    }

    public List<Choice> getImageChoices() {
        return imageChoices;
    }

    public List<Choice> getDateChoices() {
        return dateChoices;
    }

    public List<VotingList> getRestrictedVotingList() {
        return restrictedVotingList;
    }

    public List<VotingList> getGroupVotingList() {
        return groupVotingList;
    }

    public boolean isNotification() {
        return notification;
    }

    public void setNotification(boolean notification) {
        this.notification = notification;
    }

    public Integer getMaxChoices() {
        return maxChoices;
    }

    public void setMaxChoices(Integer maxChoices) {
        this.maxChoices = maxChoices;
    }

    public boolean isReminder() {
        return reminder;
    }

    public void setReminder(boolean reminder) {
        this.reminder = reminder;
    }

    public boolean isLimitChoice() {
        return limitChoice;
    }

    public void setLimitChoice(boolean limitChoice) {
        this.limitChoice = limitChoice;
    }

    public Integer getReminderHourCountdown() {
        return reminderHourCountdown;
    }

    public void setReminderHourCountdown(Integer reminderHourCountdown) {
        this.reminderHourCountdown = reminderHourCountdown;
    }

    public boolean isPollCommentVisible() {
        PollCommentVisibility pollCommentVisibility =
                poll.getPollCommentVisibility();
        return pollCommentVisibility == PollCommentVisibility.EVERYBODY;
    }

    public void setPollCommentVisible(boolean newValue) {
        if (newValue) {
            poll.setPollCommentVisibility(PollCommentVisibility.EVERYBODY);
        } else {
            poll.setPollCommentVisibility(PollCommentVisibility.NOBODY);
        }
    }

    public String getActionLabel() {
        return isEdit() ? _("pollen.action.editPoll") :
               _("pollen.action.createPoll");
    }

    public String getPageTitle() {
        return isEdit() ? getPoll().getTitle() :
               _("pollen.title.createPoll");
    }

    public boolean isVoteStarted() {
        return isEdit() && poll.sizeVote() > 0;
    }

    public boolean isCreatorUserAccountDefined() {
        PollAccount creator = poll.getCreator();
        return creator.getUserAccount() != null;
    }

    public String getImageChoiceName(Choice choice) {
        String name = choice.getName();
        try {
            String result = URLEncoder.encode(IMAGECHOICES_THUMB_PREFIX + name,
                                              Charsets.UTF_8.name());
            return result;
        } catch (UnsupportedEncodingException e) {
            throw new PollenTechnicalException(
                    "Could not encode name " + name, e);
        }
    }

    public String getVoteCountingHelp(String type) {
        return voteCountingTypesHelp.get(type);
    }

    public void prepareFormPage() throws Exception {

        getPollenSession().removeDynamicDataWithPrefix(IMAGECHOICES_THUMB_PREFIX);

        pollTypes = decorateToName(PollType.values());
        choiceTypes = decorateToName(ChoiceType.values());
        pollVoteVisibilities = decorateToName(PollVoteVisibility.values());
        voteCountingTypes = Maps.newTreeMap();
        voteCountingTypesHelp = Maps.newTreeMap();

        VoteCountingFactory voteCountingFactory = getVoteCountingFactory();
        for (VoteCounting strategy : voteCountingFactory) {
            String strategyName = strategy.getName(getLocale());
            String strategyHelp = strategy.getHelp(getLocale());
            voteCountingTypes.put(strategy.getId() + "", strategyName);
            voteCountingTypesHelp.put(strategy.getId() + "", strategyHelp);
        }

        textChoices = Lists.newArrayList();
        imageChoices = Lists.newArrayList();
        dateChoices = Lists.newArrayList();

        restrictedVotingList = Lists.newArrayList();
        groupVotingList = Lists.newArrayList();
    }

    public Function<PersonToList, PersonToList> getPersontoListCreator() {
        if (persontoListCreator == null) {
            persontoListCreator = PollenServiceFunctions.newPersonToListCreator();
        }
        return persontoListCreator;
    }

    public Function<VotingList, VotingList> getVotingListCreator() {
        if (votingListCreator == null) {
            votingListCreator = PollenServiceFunctions.newVotingListCreator(
                    getPersontoListCreator());
        }
        return votingListCreator;
    }

    public Function<Choice, Choice> getTextChoiceCreator() {
        if (textChoiceCreator == null) {
            textChoiceCreator = PollenServiceFunctions.newTextChoiceCreator();
        }
        return textChoiceCreator;
    }

    public Function<Choice, Choice> getDateChoiceCreator() {
        if (dateChoiceCreator == null) {
            dateChoiceCreator = PollenServiceFunctions.newDateChoiceCreator();
        }
        return dateChoiceCreator;
    }

    public Function<Choice, Choice> getImageChoiceCreator() {
        if (imageChoiceCreator == null) {
            imageChoiceCreator = PollenServiceFunctions.newImageChoiceCreator();
        }
        return imageChoiceCreator;
    }

    private <E extends Enum<E> & I18nAble> Map<String, String> decorateToName(E... values) {
        Map<String, String> result = Maps.newLinkedHashMap();
        for (E value : values) {
            result.put(value.name(), getText(value.getI18nKey()));
        }
        return result;
    }

    protected void validateInformations() {

        // -- Title : required -- //
        if (StringUtils.isBlank(poll.getTitle())) {
            addInformationsError("poll.title",
                                 _("pollen.error.poll.required.title"));
        }

        if (isVoteStarted()) {

            // no validation on choices if vote is started

        } else {

            // -- Choice -- //
            if (MapUtils.isEmpty(choices)) {

                // poll must have at least one choice
                addInformationsError("poll.choices",
                                     _("pollen.error.poll.required.one.choice"));
            } else {
                switch (poll.getChoiceType()) {

                    case TEXT:
                        validateTextChoices();
                        break;
                    case DATE:
                        validateDateChoices();
                        break;
                    case IMAGE:
                        validateImageChoices();
                        break;
                }
            }
        }
    }

    protected void validateTextChoices() {

        String choicePrefix = getChoiceFieldPrefix(ChoiceType.TEXT);

        Set<String> choiceNames = Sets.newHashSet();

        for (Map.Entry<Integer, Choice> entry : choices.entrySet()) {
            Integer choiceIndex = entry.getKey();
            Choice choice = entry.getValue();
            String choiceErrorField = choicePrefix + choiceIndex + ".name";
            String choiceName = choice.getName();
            if (StringUtils.isBlank(choiceName)) {
                // no name
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.choice.name.required"));
                continue;
            }
            if (choiceNames.contains(choiceName)) {
                // duplicated names
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.choice.already.used.name"));
                continue;
            }
            choiceNames.add(choiceName);
        }
    }

    protected void validateDateChoices() {

        String choicePrefix = getChoiceFieldPrefix(ChoiceType.DATE);

        Set<String> choiceNames = Sets.newHashSet();

        for (Map.Entry<Integer, Choice> entry : choices.entrySet()) {
            Integer choiceIndex = entry.getKey();
            PollDateChoice choice = (PollDateChoice) entry.getValue();
            String choiceErrorField = choicePrefix + choiceIndex + ".name";
            if (StringUtils.isBlank(choice.getName())) {
                // no name
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.dateChoice.required"));
                continue;
            }
            if (choice.getDate() == null) {

                // format error
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.dateChoice.badDateFormat"));
                continue;
            }

            // date is valid
            // use now the date.toString() as value to ensure unicity
            String choiceValue = choice.getDate().toString();

            if (choiceNames.contains(choiceValue)) {
                // duplicated names
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.dateChoice.already.used"));
                continue;
            }
            choiceNames.add(choiceValue);
        }
    }

    protected void validateImageChoices() {

        String choicePrefix = getChoiceFieldPrefix(ChoiceType.IMAGE);

        Set<String> choiceNames = Sets.newHashSet();

        for (Map.Entry<Integer, Choice> entry : choices.entrySet()) {
            Integer choiceIndex = entry.getKey();
            Choice choice = entry.getValue();
            String choiceErrorField = choicePrefix + choiceIndex + ".name";
            String choiceName = choice.getName();
            if (StringUtils.isBlank(choiceName)) {
                // no name
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.imageChoice.upload.required"));
                continue;
            }
            if (choiceNames.contains(choiceName)) {
                // duplicated names
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.imageChoice.already.used"));
                continue;
            }
            choiceNames.add(choiceName);
            String contentTypeField = choicePrefix + choiceIndex + ".newContentType";
            String contentTypes = getNonEmptyParameterValue(contentTypeField);
            if (contentTypes != null && !contentTypes.startsWith("image/")) {
                // Bad content Type
                addInformationsError(
                        choiceErrorField,
                        _("pollen.error.poll.imageChoice.upload.badContentType"));
                //Remove the file and its thumb
                PollImageChoice imageChoice = (PollImageChoice) choice;
                File imageFile = new File(imageChoice.getLocation());
                File imageThumbFile = getPollService().getImageThumbFile(imageFile);
                FileUtils.deleteQuietly(imageThumbFile);
                FileUtils.deleteQuietly(imageFile);
                // The choice has no more location
                String choiceId = imageChoice.getTopiaId();
                if (StringUtils.isBlank(choiceId)) {
                    imageChoice.setLocation(null);
                    // As the name of the choice is the fileName, remove it too.
                    imageChoice.setName(null);
                } else {
                    // in case of update, get back the old values for name and location and keep the old choice
                    PollImageChoice pollChoice = (PollImageChoice) poll.getChoiceByTopiaId(choiceId);
                    imageChoice.setName(pollChoice.getName());
                    imageChoice.setLocation(pollChoice.getLocation());
                }
                continue;
            }
        }
    }

    protected void validateOptions() {

        if (isVoteStarted()) {

            // no validation on votingLists if vote is started

        } else {

            // -- VotingList -- //

            if (poll.isPollFree()) {

                // nothing to validate

            } else {

                Set<String> groups = Sets.newHashSet();
                Set<String> voters = Sets.newHashSet();
                Set<String> emails = Sets.newHashSet();

                for (Map.Entry<Integer, VotingList> entry : votingLists.entrySet()) {
                    validateVotingList(entry.getKey(),
                                       entry.getValue(),
                                       groups,
                                       voters,
                                       emails);
                }
            }
        }

        String creatorEmail = poll.getCreator().getEmail();
        if (StringUtils.isNotBlank(creatorEmail) &&
            !StringUtil.isEmail(creatorEmail)) {

            addOptionsError("poll.creator.email",
                            _("pollen.error.email.invalid"));
        }

        Date currentTime = serviceContext.getCurrentTime();

        if (validateEndDate(poll.getBeginChoiceDate(), poll.getEndChoiceDate())) {

            addOptionsError(
                    "poll.endChoiceDate",
                    _("pollen.error.poll.endChoiceDate.before.beginChoiceDate"));
        }

        if (validateEndDate(currentTime, poll.getEndChoiceDate())) {

            addOptionsError(
                    "poll.endChoiceDate",
                    _("pollen.error.poll.endChoiceDate.before.now"));
        }

        if (validateEndDate(poll.getBeginDate(), poll.getEndDate())) {

            addOptionsError("poll.endDate",
                            _("pollen.error.poll.endDate.before.beginDate"));
        }


        if (validateEndDate(currentTime, poll.getEndDate())) {

            addOptionsError(
                    "poll.endDate",
                    _("pollen.error.poll.endDate.before.now"));
        }

        if (validateEndDate(poll.getEndChoiceDate(), poll.getEndDate())) {

            addOptionsError("poll.endChoiceDate",
                            _("pollen.error.poll.endChoiceDate.after.endDate"));
        }

        if (isLimitChoice()) {

            // validate maxChoices

            if (getMaxChoices() == null) {
                // maxChoices == null
                addOptionsError("maxChoices",
                                _("pollen.error.poll.maxChoice.required"));
            } else if (getMaxChoices() < 1) {
                // maxChoices <= 0
                addOptionsError("maxChoices",
                                _("pollen.error.poll.maxChoice.greater.than.0"));
            }
        }

        if (isReminder()) {

            // validate reminderHourCountdown

            if (getReminderHourCountdown() == null) {

                // reminderHourCountdown == null
                addOptionsError("reminderHourCountdown",
                                _("pollen.error.poll.reminderHourCountdown.required"));
            } else if (getReminderHourCountdown() < 1) {

                // reminderHourCountdown <= 0
                addOptionsError("reminderHourCountdown",
                                _("pollen.error.poll.reminderHourCountdown.greater.than.0"));
            }
        }
    }

    protected void validateVotingList(int votingListNumber,
                                      VotingList votingList,
                                      Set<String> groups,
                                      Set<String> voters,
                                      Set<String> emails) {

        PollType votingListType = poll.getPollType();
        String fieldNamePrefix = "votingList" + votingListType + "_" +
                                 votingListNumber;

        if (poll.isPollGroup()) {

            // group poll

            // check there is at least one group
            // check no doublon on group names
            // check there is at least one voter on each group
            // check no doublon on voter names
            // check no doublon on voter emails

            // validate votingList name
            String votingListName = votingList.getName();

            if (StringUtils.isEmpty(votingListName)) {

                addOptionsError(
                        fieldNamePrefix,
                        _("pollen.error.poll.required.votingList.name"));
            } else {

                // check no votingList name doublon
                boolean add = groups.add(votingListName);
                if (!add) {

                    // name doublon
                    addOptionsError(
                            fieldNamePrefix,
                            _("pollen.error.poll.votingList.name.doublon"));
                }
            }

            // validate votingList weight

            if (votingList.getWeight() == 0) {

                // no weight filled (can be a bad conversion)
                addOptionsError(
                        fieldNamePrefix,
                        _("pollen.error.poll.votingList.weight.not.valid"));
            }

        }

        // check there is at least one voter

        List<PersonToList> personToLists =
                votingList.getPollAccountPersonToList();

        if (CollectionUtils.isEmpty(personToLists)) {

            // no personToList found for unique votingList 0
            addOptionsError(fieldNamePrefix,
                            _("pollen.error.poll.required.one.personToList"));
        } else {

            // check no doublon on voter names

            for (int i = 0; i < personToLists.size(); i++) {

                validatePersonList(i,
                                   fieldNamePrefix,
                                   personToLists.get(i),
                                   voters,
                                   emails);
            }
        }
    }

    protected void validatePersonList(int personToListNumber,
                                      String votingListFieldNamePrefix,
                                      PersonToList personToList,
                                      Set<String> voters,
                                      Set<String> emails) {

        String fieldNamePrefix = votingListFieldNamePrefix +
                                 "PersonToList_" + personToListNumber;

        PollAccount pollAccount = personToList.getPollAccount();

        // -- validate votingId -- //

        String votingId = pollAccount.getVotingId();

        if (StringUtils.isEmpty(votingId)) {

            // voter name mandatory
            addOptionsError(
                    fieldNamePrefix,
                    _("pollen.error.poll.personToList.votingId.required"));
        } else {

            boolean add = voters.add(votingId);
            if (!add) {

                // voter name already used
                addOptionsError(
                        fieldNamePrefix,
                        _("pollen.error.poll.personToList.votingId.doublon"));
            }
        }

        // -- validate email -- //

        String email = pollAccount.getEmail();

        if (StringUtils.isEmpty(email)) {

            // voter email mandatory
            addOptionsError(fieldNamePrefix,
                            _("pollen.error.email.required"));
        } else {

            boolean validEmail = StringUtil.isEmail(email);
            if (!validEmail) {

                // not a valid email
                addOptionsError(fieldNamePrefix,
                                _("pollen.error.email.invalid"));
            } else {

                boolean add = emails.add(email);
                if (!add) {

                    // email already used
                    addOptionsError(
                            fieldNamePrefix,
                            _("pollen.error.poll.personToList.email.doublon"));
                }
            }
        }

        // -- validate weight -- //

        if (personToList.getWeight() == 0) {

            // no weight filled (can be a bad conversion)
            addOptionsError(
                    fieldNamePrefix,
                    _("pollen.error.poll.personToList.weight.not.valid"));
        }
    }

    protected boolean validateEndDate(Date begin, Date end) {
        return begin != null
               && end != null
               && end.before(begin);
    }

    public Object getDateChoiceValue(PollDateChoice choice) {
        Date date = choice.getDate();
        Object result;
        if (date == null) {
            result = choice.getName();
        } else {
            result = date;
        }
        return result;
    }

    @Override
    public void addFieldError(String fieldName, String errorMessage) {
        super.addFieldError(fieldName, errorMessage);
        if (log.isDebugEnabled()) {
            log.debug("VALIDATION [" + fieldName + "] : " + errorMessage);
        }
    }

    protected void addInformationsError(String fieldName, String errorMessage) {
        addFieldError(fieldName, errorMessage);
        informationsError = true;
    }

    protected void addOptionsError(String fieldName, String errorMessage) {
        addFieldError(fieldName, errorMessage);
        optionsError = true;
    }

    protected void loadChoicesAndvotingLists(Poll poll,
                                             Collection<Choice> pollChoices,
                                             Collection<VotingList> pollVotinLists,
                                             boolean fillLists) throws IOException {

        Preconditions.checkNotNull(poll);
        Preconditions.checkNotNull(poll.getChoiceType());
        Preconditions.checkNotNull(poll.getPollType());

        if (CollectionUtils.isNotEmpty(pollChoices)) {

            //push back choices

            switch (poll.getChoiceType()) {

                case TEXT:
                    getTextChoices().addAll(pollChoices);
                    break;
                case DATE:
                    getDateChoices().addAll(pollChoices);
                    break;
                case IMAGE:
                    getImageChoices().addAll(pollChoices);

                    PollService service = getPollService();

                    // if images are not still saved in a poll, then thumb
                    // does not exists, must create a temporary one
                    for (Choice choice : pollChoices) {
                        PollImageChoice imageChoice = (PollImageChoice) choice;
                        String choiceId = imageChoice.getTopiaId();
                        File imageChoiceFile = null;

                        if (StringUtils.isBlank(choiceId)) {

                            // new choice,

                            if (!isImageUploadEmpty(imageChoice)) {

                                // ok there is an upload
                                imageChoiceFile =
                                        new File(imageChoice.getLocation());
                            }

                        } else {

                            // already persisted choice, check choice has not changed

                            Choice oldChoice =
                                    poll.getChoiceByTopiaId(choiceId);

                            if (ObjectUtils.equals(
                                    imageChoice.getName(),
                                    oldChoice.getName())) {

                                // name has not changed, keep persisted choice
                                imageChoiceFile = service.getPollChoiceImageFile(
                                        poll.getPollId(), choice.getName());
                            } else {

                                // use new choice image location
                                imageChoiceFile =
                                        new File(imageChoice.getLocation());
                            }

                        }

                        if (imageChoiceFile != null) {
                            File thumbFile = service.getImageThumbFile(
                                    imageChoiceFile);

                            // keep in session the location of this thumb (do not
                            // want to expose the full path location in url)
                            String key = getImageChoiceName(choice);
                            getPollenSession().putDynamicData(key, thumbFile);
                        }
                    }

                    break;
            }
        }

        if (CollectionUtils.isNotEmpty(pollVotinLists)) {

            // push back voting lists

            switch (poll.getPollType()) {

                case FREE:
                    // not voting lists
                    break;
                case RESTRICTED:
                    getRestrictedVotingList().addAll(pollVotinLists);
                    break;
                case GROUP:
                    getGroupVotingList().addAll(pollVotinLists);
                    break;
            }
        }

        if (fillLists) {

            // refill lists

            int defaultMaxChoices = 5;
            int defaultMaxVoting = 5;

            fillLists(defaultMaxChoices, defaultMaxVoting);
        }
    }

    protected void fillLists(int defaultMaxChoices, int defaultMaxVoting) {

        PollenServiceFunctions.fillChoiceList(textChoices,
                                              defaultMaxChoices,
                                              getTextChoiceCreator());

        PollenServiceFunctions.fillChoiceList(dateChoices,
                                              defaultMaxChoices,
                                              getDateChoiceCreator());

        PollenServiceFunctions.fillChoiceList(imageChoices,
                                              defaultMaxChoices,
                                              getImageChoiceCreator());

        if (CollectionUtils.isEmpty(restrictedVotingList)) {
            restrictedVotingList.add(getVotingListCreator().apply(null));
        }
        for (VotingList votingList : restrictedVotingList) {
            PollenServiceFunctions.fillVotingList(votingList,
                                                  getPersontoListCreator(),
                                                  defaultMaxVoting);
        }
        if (CollectionUtils.isEmpty(groupVotingList)) {
            groupVotingList.add(getVotingListCreator().apply(null));
        }
        for (VotingList votingList : groupVotingList) {
            PollenServiceFunctions.fillVotingList(votingList,
                                                  getPersontoListCreator(),
                                                  defaultMaxVoting);
        }
    }

    protected boolean isTextChoiceEmpty(Choice textChoice) {
        boolean result =
                StringUtils.isBlank(textChoice.getName()) &&
                StringUtils.isBlank(textChoice.getDescription());
        return result;
    }

    protected boolean isDateChoiceEmpty(PollDateChoice dateChoice) {
        boolean result =
                StringUtils.isBlank(dateChoice.getName()) &&
                StringUtils.isBlank(dateChoice.getDescription());
        return result;
    }

    protected boolean isImageChoiceEmpty(PollImageChoice imageChoice) {
        boolean result =
                StringUtils.isBlank(imageChoice.getLocation()) &&
                StringUtils.isBlank(imageChoice.getDescription());
        return result;
    }

    protected boolean isImageUploadEmpty(PollImageChoice imageChoice) {
        boolean result =
                StringUtils.isBlank(imageChoice.getLocation());
        return result;
    }

    protected boolean isPersonToListEmpty(PersonToList personToList) {
        PollAccount pollAccount = personToList.getPollAccount();
        boolean result =
                StringUtils.isBlank(pollAccount.getVotingId()) &&
                StringUtils.isBlank(pollAccount.getEmail());
        return result;
    }

    protected Map<Integer, Choice> buildTextChoices() {
        Map<Integer, Choice> result = Maps.newTreeMap();

        String fieldPrefix = getChoiceFieldPrefix(ChoiceType.TEXT);

        int maxNumber = 0;

        for (String paramName : getParameters().keySet()) {

            Matcher matcher = TEXT_CHOICE_NAME_PATTERN.matcher(paramName);
            if (matcher.matches()) {

                // found a text choice name

                String paramValue = getNonEmptyParameterValue(paramName);

                // can keep this none empty text choice name

                Integer choiceNumber = Integer.valueOf(matcher.group(1));

                if (choiceNumber > maxNumber) {
                    maxNumber = choiceNumber;
                }
                String choicePrefix = fieldPrefix + choiceNumber;
                Choice choice = createChoice(new ChoiceImpl(),
                                             choicePrefix,
                                             paramValue);
                if (!isTextChoiceEmpty(choice)) {

                    // This is not an empty choice keep it
                    result.put(choiceNumber, choice);

                    // keep order
                    int order = getIntegerValue(choicePrefix + ".order");
                    choicesOrder.put(order, choiceNumber);
                }
            }
        }
        result = reindexChoiceMap(result, maxNumber);

        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbTextChoices (from request) = " + size);
        }
        logChoice(result);
        return result;
    }

    protected Map<Integer, Choice> buildDateChoices() {
        Map<Integer, Choice> result = Maps.newTreeMap();

        String fieldPrefix = getChoiceFieldPrefix(ChoiceType.DATE);

        int maxNumber = 0;
        for (String paramName : getParameters().keySet()) {

            Matcher matcher = DATE_CHOICE_NAME_PATTERN.matcher(paramName);
            if (matcher.matches()) {

                // found a text choice name

                String paramValue = getNonEmptyParameterValue(paramName);

                // can keep this none empty text choice name

                Integer choiceNumber = Integer.valueOf(matcher.group(1));
                if (choiceNumber > maxNumber) {
                    maxNumber = choiceNumber;
                }
                String choicePrefix = fieldPrefix + choiceNumber;

                PollDateChoice choice = createChoice(new PollDateChoice(),
                                                     choicePrefix,
                                                     paramValue);
                if (StringUtils.isNotBlank(paramValue)) {
                    Date date = DateConverter.convertFromString(paramValue);
                    choice.setDate(date);
                }

                if (!isDateChoiceEmpty(choice)) {

                    // This is not an empty choice keep it
                    result.put(choiceNumber, choice);

                    // keep order
                    int order = getIntegerValue(choicePrefix + ".order");
                    choicesOrder.put(order, choiceNumber);
                }
            }
        }
        result = reindexChoiceMap(result, maxNumber);

        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbDateChoices (from request)  = " + size);
        }
        logChoice(result);
        return result;
    }

    protected Map<Integer, Choice> buildImageChoices() {
        Map<Integer, Choice> result = Maps.newTreeMap();

        String fieldPrefix = getChoiceFieldPrefix(ChoiceType.IMAGE);

        int maxNumber = 0;

        Map<String, String> parametersToSwitch = Maps.newTreeMap();

        for (String paramName : getParameters().keySet()) {

            Matcher matcher = IMAGE_CHOICE_DESCRIPTION_PATTERN.matcher(paramName);
            if (matcher.matches()) {

                // found an image choice description
                // Note: We can not use as for other choices (text and date)
                // the name since there is no name parameter submitted...
                // the upload filed will implies a such parameter...

                Integer choiceNumber = Integer.valueOf(matcher.group(1));
                if (choiceNumber > maxNumber) {
                    maxNumber = choiceNumber;
                }

                String choicePrefix = fieldPrefix + choiceNumber;

                PollImageChoice choice = createChoice(new PollImageChoice(),
                                                      choicePrefix,
                                                      null);
                String location = getNonEmptyParameterValue(
                        choicePrefix + ".newLocation");
                if (StringUtils.isNotBlank(location)) {

                    // found a real upload file (not jus the location of an
                    // already choice uploaded choice
                    choice.setLocation(location);

                    // ok so now use also newName
                    String paramValue = getNonEmptyParameterValue(
                            choicePrefix + ".newName");
                    choice.setName(paramValue);

                    parametersToSwitch.put(choicePrefix + ".newLocation",
                                           choicePrefix + ".location");

                    parametersToSwitch.put(choicePrefix + ".newName",
                                           choicePrefix + ".name");

                } else {

                    // try with old location
                    location = getNonEmptyParameterValue(
                            choicePrefix + ".location");
                    choice.setLocation(location);

                    String paramValue = getNonEmptyParameterValue(
                            choicePrefix + ".name");
                    choice.setName(paramValue);
                }

                if (!isImageChoiceEmpty(choice)) {

                    // This is not an empty choice keep it
                    result.put(choiceNumber, choice);

                    // keep order
                    int order = getIntegerValue(choicePrefix + ".order");
                    choicesOrder.put(order, choiceNumber);
                }
            }
        }

        // switch new uploaded values
        for (Map.Entry<String, String> entry : parametersToSwitch.entrySet()) {
            String paramNameSource = entry.getKey();
            String paramNameTarget = entry.getValue();
            String[] valueToSwitch = getParameters().remove(paramNameSource);
            getParameters().put(paramNameTarget, valueToSwitch);
        }
        result = reindexChoiceMap(result, maxNumber);
        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbImageChoices  (from request) = " + size);
        }
        logChoice(result);
        return result;
    }

    protected Map<Integer, VotingList> buildVotingLists(PollType pollType) {
        Map<Integer, VotingList> result = Maps.newTreeMap();

        int maxNumber = 0;

        String votingListPrefix = "votingList" + pollType.name();

        // get all votingList_xxx parameters
        Set<String> votingListParameterNames = Sets.filter(
                getParameters().keySet(),
                new StringStartWithPredicate(votingListPrefix));

        Pattern votingListPattern = Pattern.compile(
                "(" + votingListPrefix + ")_(\\d+)\\.name");

        for (String paramName : votingListParameterNames) {

            Matcher matcher = votingListPattern.matcher(paramName);

            if (matcher.matches()) {

                // found a voting list name

                int votingListNumber = Integer.valueOf(matcher.group(2));

                buildVotingList(paramName,
                                votingListPrefix + "_" + votingListNumber,
                                votingListNumber,
                                result
                );
                maxNumber = Math.max(maxNumber, votingListNumber);
            }
        }

        result = reindexMap(result, maxNumber);

        int size = result.size();
        if (log.isInfoEnabled()) {
            log.info("nbVotingList [" + pollType + "] (from request) = " + size);
        }

        // add personToList maps to session (but just now, since votingList
        // could have been reindex)
        for (Map.Entry<Integer, VotingList> entry : result.entrySet()) {
            VotingList votingList = entry.getValue();

            if (!votingList.isPollAccountPersonToListEmpty()) {
                List<PersonToList> personToList =
                        votingList.getPollAccountPersonToList();

                Map<Integer, PersonToList> personToListMap = Maps.newTreeMap();
                int index = 0;
                for (PersonToList toList : personToList) {
                    personToListMap.put(index++, toList);
                }
            }
        }

        return result;
    }

    private double getDoubleValue(String parameterName) {
        String parameterValue = getNonEmptyParameterValue(parameterName);
        double result = 0;
        if (StringUtils.isNotEmpty(parameterValue)) {

            try {
                result = Double.valueOf(parameterValue);
            } catch (NumberFormatException e) {
                //bad conversion, will be treated later
                if (log.isDebugEnabled()) {
                    log.debug("Bad double conversion from param [" +
                              parameterName + "] : " + parameterValue);
                }
            }
        }
        return result;
    }

    private int getIntegerValue(String parameterName) {
        String parameterValue = getNonEmptyParameterValue(parameterName);
        int result = 0;
        if (StringUtils.isNotEmpty(parameterValue)) {

            try {
                result = Integer.valueOf(parameterValue);
            } catch (NumberFormatException e) {
                //bad conversion, will be treated later
                if (log.isDebugEnabled()) {
                    log.debug("Bad double conversion from param [" +
                              parameterName + "] : " + parameterValue);
                }
            }
        }
        return result;
    }

    private int buildVotingList(String votingListParameterName,
                                String votingListPrefix,
                                int votingListNumber,
                                Map<Integer, VotingList> result) {

        String paramValue = getNonEmptyParameterValue(votingListParameterName);

        VotingList votingList = new VotingListImpl();

        votingList.setName(paramValue);

        double weight = getDoubleValue(votingListPrefix + ".weight");
        votingList.setWeight(weight);

        String topiaId = getNonEmptyParameterValue(votingListPrefix + ".topiaId");
        votingList.setTopiaId(topiaId);

        result.put(votingListNumber, votingList);

        String personToListPrefix = votingListPrefix + "PersonToList_";

        // get all personToList parameters
        Set<String> votingListParameterNames = Sets.filter(
                getParameters().keySet(),
                new StringStartWithPredicate(personToListPrefix));

        Pattern personToListNamePattern = Pattern.compile(
                personToListPrefix + "(\\d+)\\.votingId");

        Map<Integer, PersonToList> personToLists = Maps.newTreeMap();

        int maxPersonToListNumber = 0;

        // let's build personToList list
        for (String personToListNameParameter : votingListParameterNames) {

            Matcher matcher = personToListNamePattern.matcher(
                    personToListNameParameter);

            if (matcher.matches()) {

                int personToListNumber = buildPersonToList(
                        personToListPrefix,
                        personToListNameParameter,
                        matcher,
                        votingListNumber,
                        personToLists);

                maxPersonToListNumber = Math.max(maxPersonToListNumber,
                                                 personToListNumber);
            }
        }

        personToLists = reindexMap(personToLists, maxPersonToListNumber);

        for (PersonToList personToList : personToLists.values()) {
            votingList.addPollAccountPersonToList(personToList);
        }

        return votingListNumber;
    }

    private int buildPersonToList(String personToListPrefix,
                                  String paramName,
                                  Matcher personToListMatcher,
                                  int votingListNumber,
                                  Map<Integer, PersonToList> result) {

        int personToListNumber = Integer.valueOf(personToListMatcher.group(1));

        PersonToList personToList = new PersonToListImpl();

        PollAccount account = new PollAccountImpl();
        personToList.setPollAccount(account);

        account.setVotingId(getNonEmptyParameterValue(paramName));

        String prefix = personToListPrefix + personToListNumber;

        double weight = getDoubleValue(prefix + ".weight");
        personToList.setWeight(weight);

        String topiaId = getNonEmptyParameterValue(prefix + ".topiaId");
        personToList.setTopiaId(topiaId);

        String email = getNonEmptyParameterValue(prefix + ".email");
        account.setEmail(email);

        String accountId = getNonEmptyParameterValue(prefix + ".accountId");
        account.setAccountId(accountId);

        if (!isPersonToListEmpty(personToList)) {

            // can keep this not empty personToList
            result.put(personToListNumber, personToList);
        }
        return personToListNumber;
    }

    private <C extends Choice> C createChoice(C choice,
                                              String prefix,
                                              String name) {
        String description = getNonEmptyParameterValue(prefix + ".description");
        String topiaId = getNonEmptyParameterValue(prefix + ".topiaId");
        choice.setName(name);
        choice.setDescription(description);
        choice.setTopiaId(topiaId);
        return choice;
    }

    private void logChoice(Map<Integer, Choice> result) {
        for (Map.Entry<Integer, Choice> e : result.entrySet()) {
            Integer choiceId = e.getKey();
            Choice choice = e.getValue();
            if (log.isInfoEnabled()) {
                log.info("Choice [" + choiceId + "] = " +
                         choice.getName() + " -- " +
                         choice.getDescription());
            }
        }
    }

    protected <T> Map<Integer, T> reindexMap(Map<Integer, T> result, int maxNumber) {
        Map<Integer, T> result2;

        if (maxNumber != result.size() - 1) {

            // means there is a hole inside the result (a empty choice was
            // submitted)

            // le'ts remove this
            List<Integer> numbers = Lists.newArrayList(result.keySet());

            Collections.sort(numbers);

            result2 = Maps.newTreeMap();
            int i = 0;
            for (Integer number : numbers) {
                T choice = result.get(number);
                result2.put(i++, choice);
            }
        } else {
            result2 = result;
        }
        return result2;
    }

    protected <T> Map<Integer, T> reindexChoiceMap(Map<Integer, T> result, int maxNumber) {
        Map<Integer, T> result2 = Maps.newTreeMap();

        for (Integer choiceOrder : choicesOrder.keySet()) {
            Integer choiceNumber = choicesOrder.get(choiceOrder);
            T choice = result.get(choiceNumber);
            if (choice != null) {
                result2.put(choiceOrder, choice);
            }
        }

//        if (maxNumber != result.size() - 1) {
//
//            // means there is a hole inside the result (a empty choice was
//            // submitted)
//
//            // le'ts remove this
//            List<Integer> numbers = Lists.newArrayList(result.keySet());
//
//            Collections.sort(numbers);
//
//            result2 = Maps.newTreeMap();
//            int i = 0;
//            for (Integer number : numbers) {
//                T choice = result.get(number);
//                result2.put(i++, choice);
//            }
//        } else {
//            result2 = result;
//        }
        return result2;
    }


    protected String getNonEmptyParameterValue(String paramName) {
        String[] paramValues = getParameters().get(paramName);
        String result = null;
        if (paramValues != null && paramValues.length == 1) {
            String paramValue = paramValues[0];
            if (StringUtils.isNotEmpty(paramValue)) {
                result = paramValue;
            }
        }
        return result;
    }

    private String getChoiceFieldPrefix(ChoiceType choiceType) {
        String result = null;
        switch (choiceType) {

            case TEXT:
                result = "textChoice_";
                break;
            case DATE:
                result = "dateChoice_";
                break;
            case IMAGE:
                result = "imageChoice_";
                break;
        }
        return result;
    }

    private static class StringStartWithPredicate implements Predicate<String> {
        private final String prefix;

        public StringStartWithPredicate(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public boolean apply(String input) {
            return input.startsWith(prefix);
        }
    }
}
