/*
 * #%L
 * Pollen :: Services
 * $Id: PollenServiceFunctions.java 3712 2012-09-30 12:57:25Z tchemit $
 * $HeadURL: http://svn.chorem.org/svn/pollen/tags/pollen-1.5.5/pollen-services/src/main/java/org/chorem/pollen/services/PollenServiceFunctions.java $
 * %%
 * Copyright (C) 2009 - 2012 CodeLutin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package org.chorem.pollen.services;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import org.chorem.pollen.bean.PollDateChoice;
import org.chorem.pollen.bean.PollImageChoice;
import org.chorem.pollen.business.persistence.Choice;
import org.chorem.pollen.business.persistence.ChoiceImpl;
import org.chorem.pollen.business.persistence.ChoiceType;
import org.chorem.pollen.business.persistence.PersonToList;
import org.chorem.pollen.business.persistence.PersonToListImpl;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountImpl;
import org.chorem.pollen.business.persistence.VotingList;
import org.chorem.pollen.business.persistence.VotingListImpl;

import java.util.Date;
import java.util.List;

/**
 * Usefull functions for the service (and higher) layers.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 1.3
 */
public class PollenServiceFunctions {

    protected PollenServiceFunctions() {
        // no instance for helper class
    }

    public static void fillChoiceList(List<Choice> choiceList, int max,
                                      Function<Choice, Choice> function) {
        while (choiceList.size() < max) {
            Choice newChoice = function.apply(null);
            choiceList.add(newChoice);
        }
    }

    public static void fillVotingList(VotingList votingList,
                                      Function<PersonToList, PersonToList> persontoListCreator,
                                      int max) {
        while (votingList.sizePollAccountPersonToList() < max) {
            PersonToList newPersontoList = persontoListCreator.apply(null);
            votingList.addPollAccountPersonToList(newPersontoList);
        }
    }

    public static final Function<Choice, String> CHOICE_TO_NAME = new Function<Choice, String>() {
        @Override
        public String apply(Choice input) {
            return input.getName();
        }
    };

    public static final Function<ChoiceType, String> CHOICE_TYPE_TO_PROPERTY_NAME = new Function<ChoiceType, String>() {
        @Override
        public String apply(ChoiceType choiceType) {
            String result;
            switch (choiceType) {
                case DATE:
                    result = PollDateChoice.PROPERTY_DATE;
                    break;
                case IMAGE:
                    result = PollImageChoice.PROPERTY_LOCATION;
                    break;
                case TEXT:
                default:
                    result = Choice.PROPERTY_NAME;
            }
            return result;
        }
    };

    public static TextChoiceCreator newTextChoiceCreator() {
        return new TextChoiceCreator();
    }

    public static ImageChoiceCreator newImageChoiceCreator() {
        return new ImageChoiceCreator();
    }

    public static DateChoiceCreator newDateChoiceCreator() {
        return new DateChoiceCreator();
    }

    public static Function<Choice, Choice> newChoiceCreator(ChoiceType choiceType) {
        Function<Choice, Choice> result;
        switch (choiceType) {

            case DATE:
                result = newDateChoiceCreator();
                break;

            case IMAGE:
                result = newImageChoiceCreator();
                break;

            default:
            case TEXT:
                result = newTextChoiceCreator();
        }
        return result;
    }

    public static Function<VotingList, VotingList> newVotingListCreator(
            Function<PersonToList, PersonToList> persontoListCreator) {
        return new VotingListCreator(persontoListCreator);
    }

    public static Function<PersonToList, PersonToList> newPersonToListCreator() {
        return new PersonToListCreator();
    }

    public static class TextChoiceCreator implements Function<Choice, Choice> {

        @Override
        public Choice apply(Choice input) {
            Choice result = new ChoiceImpl();
            if (input != null) {
                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setDescription(input.getDescription());
            }
            return result;
        }
    }

    public static class ImageChoiceCreator implements Function<Choice, Choice> {

        @Override
        public Choice apply(Choice input) {
            PollImageChoice result = new PollImageChoice();
            if (input != null) {
                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setDescription(input.getDescription());
                result.setLocation(input.getName());
            }
            return result;
        }
    }

    public static class DateChoiceCreator implements Function<Choice, Choice> {

        @Override
        public Choice apply(Choice input) {
            PollDateChoice result = new PollDateChoice();
            if (input != null) {
                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setDescription(input.getDescription());
                result.setDate(new Date(Long.valueOf(input.getName())));
            }
            return result;
        }
    }

    public static class VotingListCreator implements Function<VotingList, VotingList> {

        protected final Function<PersonToList, PersonToList> persontoListCreator;

        protected VotingListCreator(Function<PersonToList, PersonToList> persontoListCreator) {
            this.persontoListCreator = persontoListCreator;
        }

        @Override
        public VotingList apply(VotingList input) {
            VotingList result = new VotingListImpl();
            result.setPollAccountPersonToList(
                    Lists.<PersonToList>newArrayList());
            if (input == null) {

                // new votingList
                result.setWeight(1);
            } else {

                result.setTopiaId(input.getTopiaId());
                result.setName(input.getName());
                result.setWeight(input.getWeight());

                if (!input.isPollAccountPersonToListEmpty()) {

                    // copy person to lists
                    result.addAllPollAccountPersonToList(Lists.transform(
                            input.getPollAccountPersonToList(),
                            persontoListCreator));
                }
            }
            return result;
        }
    }

    public static class PersonToListCreator implements Function<PersonToList, PersonToList> {

        @Override
        public PersonToList apply(PersonToList input) {
            PersonToList result = new PersonToListImpl();
            if (input != null) {
                result.setTopiaId(input.getTopiaId());
                result.setWeight(input.getWeight());
                PollAccount pollAccount = input.getPollAccount();
                if (pollAccount != null) {
                    PollAccount pollAccount2 = new PollAccountImpl();
                    pollAccount2.setTopiaId(pollAccount.getTopiaId());
                    pollAccount2.setAccountId(pollAccount.getAccountId());
                    pollAccount2.setVotingId(pollAccount.getVotingId());
                    result.setPollAccount(pollAccount2);
                }
            } else {
                result.setWeight(1);
            }
            return result;
        }
    }

}
