/* *##% Pollen
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. ##%*/

package org.chorem.pollen.business.services;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.pollen.business.converters.DataPollAccountConverter;
import org.chorem.pollen.business.dto.PollAccountDTO;
import org.chorem.pollen.business.persistence.PollAccount;
import org.chorem.pollen.business.persistence.PollAccountDAO;
import org.chorem.pollen.business.persistence.PollenModelDAOHelper;
import org.chorem.pollen.business.persistence.UserAccount;
import org.chorem.pollen.business.persistence.UserAccountDAO;
import org.chorem.pollen.business.utils.ContextUtil;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;

/**
 * Implémentation du service de gestion des comptes.
 *
 * @author kmorin
 * @author rannou
 * @version $Id: ServicePollAccountImpl.java 2737 2009-08-20 13:00:52Z nrannou $
 */
public class ServicePollAccountImpl implements ServicePollAccount {
    private TopiaContext rootContext = ContextUtil.getInstance().getContext();
    private PollAccountDAO pollAccountDAO = null;
    private DataPollAccountConverter converter = new DataPollAccountConverter();

    /** log. */
    private static final Log log = LogFactory
            .getLog(ServicePollAccountImpl.class);

    public ServicePollAccountImpl() {
    }

    @Override
    public String createPollAccount(PollAccountDTO pollAccountDTO) {
        TopiaContext transaction = null;
        String topiaId;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);
            PollAccount pollAccountEntity = pollAccountDAO.create();

            // Identifiant du compte
            String id = UUID.randomUUID().toString();
            pollAccountDTO.setAccountId(id.replaceAll("-", ""));

            converter.setTransaction(transaction);
            converter.populatePollAccountEntity(pollAccountDTO,
                    pollAccountEntity);

            topiaId = pollAccountEntity.getTopiaId();
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }
            return topiaId;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public PollAccount createPollAccount(String votingId, String email,
            String userId) {
        TopiaContext transaction = null;
        String topiaId;
        PollAccount pollAccountEntity = null;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);
            pollAccountEntity = pollAccountDAO.create();

            // Identifiant du compte
            String id = UUID.randomUUID().toString();
            pollAccountEntity.setAccountId(id.replaceAll("-", ""));

            converter.setTransaction(transaction);
            converter.populatePollAccountEntity(votingId, email, userId,
                    pollAccountEntity);

            topiaId = pollAccountEntity.getTopiaId();

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity created: " + topiaId);
            }

            return pollAccountEntity;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public List<PollAccount> createPollAccounts(
            List<PollAccountDTO> pollAccounts, TopiaContext trans) {
        if (pollAccounts.isEmpty()) {
            return null;
        }

        List<PollAccount> result = null;
        try {
            pollAccountDAO = PollenModelDAOHelper.getPollAccountDAO(trans);

            converter.setTransaction(trans);

            result = new ArrayList<PollAccount>();
            for (PollAccountDTO pollAccount : pollAccounts) {
                PollAccount entity = pollAccountDAO.create();

                // Identifiants du compte
                String id = UUID.randomUUID().toString();
                pollAccount.setAccountId(id.replaceAll("-", ""));
                pollAccount.setId(entity.getTopiaId());

                converter.populatePollAccountEntity(pollAccount, entity);
                result.add(entity);
            }

            return result;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, trans);
            return null;
        }
    }

    @Override
    public boolean deletePollAccount(String pollAccountId) {
        TopiaContext transaction = null;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);
            PollAccount pollAccountEntity = pollAccountDAO
                    .findByTopiaId(pollAccountId);
            pollAccountDAO.delete(pollAccountEntity);
            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity deleted: " + pollAccountId);
            }

            return true;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        }
    }

    @Override
    public boolean updatePollAccount(PollAccountDTO pollAccountDTO) {
        TopiaContext transaction = null;
        boolean resu = false;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);
            PollAccount pollAccountEntity = pollAccountDAO
                    .findByTopiaId(pollAccountDTO.getId());

            if (pollAccountEntity != null) {
                converter.setTransaction(transaction);
                converter.populatePollAccountEntity(pollAccountDTO,
                        pollAccountEntity);

                pollAccountDAO.update(pollAccountEntity);
                transaction.commitTransaction();
                resu = true;
            }

            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity updated: " + pollAccountDTO.getId());
            }

            return resu;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return false;
        }
    }

    @Override
    public PollAccountDTO findPollAccountById(String pollAccountId) {
        TopiaContext transaction = null;
        PollAccountDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);

            PollAccount pollAccountEntity = pollAccountDAO
                    .findByTopiaId(pollAccountId);

            if (pollAccountEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPollAccountDTO(pollAccountEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public PollAccountDTO findPollAccountByAccountId(String pollAccountId) {
        TopiaContext transaction = null;
        PollAccountDTO result = null;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);

            PollAccount pollAccountEntity = pollAccountDAO
                    .findByAccountId(pollAccountId);

            if (pollAccountEntity != null) {
                converter.setTransaction(transaction);
                result = converter.createPollAccountDTO(pollAccountEntity);
            }

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entity found: "
                        + ((result == null) ? "null" : result.getId()));
            }

            return result;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public List<PollAccountDTO> findPollAccountsByUser(String userId) {
        TopiaContext transaction = null;
        List<PollAccountDTO> results = null;
        List<PollAccount> pollAccountEntities = null;
        UserAccountDAO userDAO = null;
        try {
            transaction = rootContext.beginTransaction();

            userDAO = PollenModelDAOHelper.getUserAccountDAO(transaction);
            UserAccount user = userDAO.findByTopiaId(userId);

            pollAccountEntities = new ArrayList<PollAccount>();
            pollAccountEntities.addAll(user.getPollAccount());

            converter.setTransaction(transaction);
            results = converter.createPollAccountDTOs(pollAccountEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }

    @Override
    public List<PollAccountDTO> selectPollAccounts(
            Map<String, Object> properties) {
        TopiaContext transaction = null;
        List<PollAccountDTO> results = null;
        List<PollAccount> pollAccountEntities = null;
        try {
            transaction = rootContext.beginTransaction();

            pollAccountDAO = PollenModelDAOHelper
                    .getPollAccountDAO(transaction);

            if (properties == null) {
                pollAccountEntities = pollAccountDAO.findAll();
                if (log.isWarnEnabled()) {
                    log
                            .warn("Attention : tous les comptes ont été sélectionnés !");
                }
            } else {
                pollAccountEntities = pollAccountDAO
                        .findAllByProperties(properties);
            }
            converter.setTransaction(transaction);
            results = converter.createPollAccountDTOs(pollAccountEntities);

            transaction.commitTransaction();
            transaction.closeContext();

            if (log.isDebugEnabled()) {
                log.debug("Entities found: "
                        + ((results == null) ? "null" : results.size()));
            }

            return results;
        } catch (TopiaException e) {
            ContextUtil.doCatch(e, transaction);
            return null;
        }
    }
}