/**
 * *##% Callao PeriodServiceImpl
 * Copyright (C) 2009 CodeLutin
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>. ##%*
 */

package org.chorem.callao.service;

import java.util.*;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.chorem.callao.entity.CallaoDAOHelper;
import org.chorem.callao.entity.Period;
import org.chorem.callao.entity.PeriodDAO;
import org.chorem.callao.entity.TimeSpan;
import org.chorem.callao.entity.TimeSpanDAO;
import org.chorem.callao.service.dto.PeriodDTO;
import org.chorem.callao.service.dto.TimeSpanDTO;
import org.chorem.callao.service.convertObject.ConvertPeriod;
import org.chorem.callao.service.utils.ContextCallao;
import org.chorem.callao.service.utils.DateUtil;
import org.chorem.callao.service.utils.ServiceHelper;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.util.PeriodDates;

/**
 * Gestion des périodes.
 * Chaque période doit posséder 12 timespans mensuels.
 * Une période ne peut être supprimée et débloquée.
 *
 * @author Rémi Chapelet
 */
public class PeriodServiceImpl { //implements PeriodService {
	
	private static final Log log = LogFactory.getLog(PeriodServiceImpl.class);
	
	private TopiaContext rootContext = ContextCallao.getInstance().getContext();

    private TimeSpanServiceImpl timeSpanServiceImpl = new TimeSpanServiceImpl();

    private ConvertPeriod convertPeriod = new ConvertPeriod();

    private DateUtil dateUtil = new DateUtil();

        
    //@Override
    public String addTimeSpan(Date beginTimeSpan, Date endTimeSpan)
    {
        return null;
    }

    //@Override
    public String removeTimeSpan(String timespan)
    {
        return null;
    }

    /**
     * Permet de créer une période principale. Elle a une durée de un an,
     * composées de 12 périodes mensuelles. Elle correspond à l'exercice
     * comptable. Pour créer une nouvelle période, la précédente doit être
     * obligatoirement clôturée.
     * @param beginTimeSpan date début de période
     * @param endTimeSpan date fin de période
     * @param lock Vrai si la période est bloquée
     * @return
     */
    //@Override
    public String createPeriod(Date beginTimeSpan, Date endTimeSpan,boolean lock)
    {
        // Par défaut lock est à false
        lock = false;
        String result = ServiceHelper.RESPOND_ERROR;

        try {
            boolean correctPeriod = false;
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
            // Creation de la periodDate
            PeriodDates newPeriod = new PeriodDates(beginTimeSpan, endTimeSpan);

            // Creation de la période
            Period period = periodDAO.create();
            period.setBeginPeriod(newPeriod.getFromDate());
            period.setEndPeriod(newPeriod.getThruDate());
            period.setLocked(lock);
            // Vérification si la période est correcte
            correctPeriod = isCorrectPeriod(period);
            if (correctPeriod) {
                // Si elle est correcte : Création BDD
                topiaContext.commitTransaction();                
                if (log.isInfoEnabled()) {
                    log.info("Ajout exercice du " + beginTimeSpan.toString() +
                            " au " + endTimeSpan.toString());
                }
                /**
                 * Ajout des timeSpans mensuels
                 * Par défaut, il prend seulement la date beginTimeSpan, et calcule
                 * la date de fin, c'est pourquoi il est laissé deux fois "beginTimeSpan"
                 */
                TimeSpanServiceImpl timeSpanImpl = new TimeSpanServiceImpl();
                boolean stop = false; // En cas d'erreur lors de la création d'un timeSpan.

                List<Date> monthList = newPeriod.getMonths();

                int max = monthList.size();
                log.debug("la periode fait:"+max);
                for (int i=0;i < max;i++)
                {
                    // Création timeSpan
                    result = timeSpanImpl.createTimeSpan(monthList.get(i),monthList.get(i),period,false);
                    if (result.equals(ServiceHelper.RESPOND_ERROR))
                    {
                        stop = true;
                        if (log.isErrorEnabled()) {
                            log.error("Erreur lors de la création des "+max+" timeSpans");
                        }
                        result = ServiceHelper.PERIOD_CREATE_TIMESPANS;
                    }
                    // Supprimer la periode si stop = true
                }
                if ( stop == false )
                {
                    result = ServiceHelper.RESPOND_SUCCESS;
                }
            }
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return result;
    }
    

    /**
     * Permet de créer une période à partir d'une période DTO.
     * @param periodDTO période à créer au format DTO.
     * @return
     */
    public String createPeriod (PeriodDTO periodDTO)
    {
        String result = createPeriod(periodDTO.getBeginPeriod(),periodDTO.getEndPeriod(),false);
        // Tous les timeSpans sont automatiquement créés avec la méthode createPeriod
        return result;
    }

    /**
     * Permet de vérifier si la période est correcte (aucun chevauchement avec
     * les autres périodes, période(s) précédente(s) bloquée(s), formée de 1 à 24
     * mois).
     * @param period période à tester si elle est correcte
     * @return
     */
    private boolean isCorrectPeriod(Period period)
    {
        // Calcul du nombre de mois pour que la période soit valide (12 mois)
        int number_months = dateUtil.numberOfMonths(period.getBeginPeriod(),period.getEndPeriod());
        boolean result = true;

        if (number_months == 0 && number_months >= 24) // Periode de 1mois a 24 mois
        {
            if (log.isErrorEnabled()) {
                log.error("Période du " + period.getBeginPeriod() +
                            " au " + period.getEndPeriod()+
                            "  : L'exercice doit faire au minimum 1 mois et au maximum 24 mois");
            }
            result = false;
        }else{
            // Période(s) précédente(s) bloquée(s).
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
                // Recherche les périodes qui ne sont pas bloquées.
                List<Period> listePeriod = periodDAO.findAllByLocked(false);
                // Recherche la dernière période, les dates doivent se suivre
                Date date_search = period.getBeginPeriod();
                Period previousPeriod = periodDAO.findByEndPeriod(dateUtil.previousDay(date_search));
                // Nombre de périodes dans la base
                int number_period = periodDAO.findAll().size();
                // Fermeture BDD
                topiaContext.closeContext();
                // Si il y a au moins une période trouvée
                if ( !(listePeriod.isEmpty()) )
                {
                    if (log.isErrorEnabled()) {
                        log.error("Il existe au moins une période précédente non bloquée !");
                    }
                    result = false;
                }
                // Si la période précédente correspond au jour précédent
                // et vérifie si ce n'est pas la première période
                if ( previousPeriod == null && number_period != 0 )
                {
                    if (log.isErrorEnabled()) {
                        log.error("Votre période doit être collée " +
                                "aux autres périodes !"+date_search.toString()+"  PPP   "
                                +dateUtil.previousDay(date_search).toString());
                    }
                    result = false;
                }
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }

    /**
     * Cette méthode permet de bloquer une période. Elle doit vérifier que
     * tous les timeSpans mensuels sont bloqués.
     * @param 
     * @return
     */
    //@Override
    public String blockPeriod(Period period)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        // La période ne doit pas être nulle
        if ( period != null )
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                TimeSpanDAO timeSpanDAO = CallaoDAOHelper.getTimeSpanDAO(topiaContext);
                // Recherche tous les timeSpans de la période non bloquées
                List<TimeSpan> listeTimeSpan = timeSpanDAO.findAllByPeriod(period);
                boolean existTimeSpanNotBlock = false;
                for (TimeSpan timeSpan : listeTimeSpan)
                {
                    // Si timeSpan non bloqué
                    if (!timeSpan.getLocked())
                    {
                        existTimeSpanNotBlock = true;
                        if (log.isErrorEnabled()) {
                            log.error("Le timeSpan du "+
                                    timeSpan.getBeginTimeSpan().toString()+" au "
                                    + timeSpan.getEndTimeSpan().toString()+
                                    " non bloqué !");
                        }
                        result = ServiceHelper.PERIOD_TIMESPAN_NOT_BLOCK;
                    }
                }
                // Si il n'existe pas de timeSpan non bloqué
                if ( !(existTimeSpanNotBlock) )
                {
                    // Bloque la période
                    period.setLocked(true);
                    // Chargement du DAO
                    PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
                    periodDAO.update(period);
                    // Création BDD
                    topiaContext.commitTransaction();
                    result = ServiceHelper.RESPOND_SUCCESS;
                    if (log.isInfoEnabled()) {
                            log.info("Période "+
                                    period.getBeginPeriod().toString()+" au "
                                    + period.getEndPeriod().toString()+
                                    " bloquée avec succès.");
                    }
                }
                // Fermeture BDD
                topiaContext.closeContext();
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }

    /**
     * Permet de bloquer une période. Avec la périodeDTO passée en paramètre, on
     * effectue une recherche avec la date de début pour récupérer la période
     * dans la base de données. Il est possible de l'identifier avec son topiaId,
     * mais il y a plus de risque que cette mentien ne soit pas renseignée dans
     * l'objet periodDTO.
     * @param periodDTO
     * @return
     */
    public String blockPeriod(PeriodDTO periodDTO)
    {
        // Recherche de la période
        Period period = searchPeriodWithDate(periodDTO.getBeginPeriod());
        String result = blockPeriod(period);
        return result;
    }


    /**
     * Permet de bloquer TOUS les timeSpans de la période
     * @param period
     * @return
     */
    public String blockAllTimeSpanOfPeriod(Period period)
    {
        String result = ServiceHelper.RESPOND_ERROR; 
        if ( period == null )
        {
            if (log.isErrorEnabled()) {
                log.error("La période ne peut bloquer ses périodes mensuelles, " +
                        "il faut une période en entrée.");
            }
        } else {
            // On bloque chaque timeSpan
            boolean ErrorBlockTimeSpan = false;
            List<TimeSpanDTO> listTimeSpanDTO = timeSpanServiceImpl.searchListTimeSpanDTO(period);
            // tri la liste des timeSpans dans l'ordre croissant des dates
            Collections.sort(listTimeSpanDTO);
            for (TimeSpanDTO timeSpanDTO : listTimeSpanDTO)
            {
                result = timeSpanServiceImpl.blockTimeSpan(timeSpanDTO);
                if ( result.equals(ServiceHelper.RESPOND_ERROR) )
                {
                    ErrorBlockTimeSpan = true;
                }
            }                                          
            // Si il n'y a pas eu d'erreurs pour bloquer les timespans
            if ( !ErrorBlockTimeSpan )
            {
                result = ServiceHelper.RESPOND_SUCCESS;
            } else {
                if (log.isErrorEnabled()) {
                    log.error("Impossible de bloquer tous les timeSpans.");
                }
                result = ServiceHelper.PERIOD_ALL_TIMESPAN;
            }
        }
        return result;
    }
   

    /**
     * Cette méthode est semblable à searchPeriodWithDateFirst. Elle permet
     * de rechercher une période sans forcément connaître la date de départ. On
     * donne une date quelconque, et elle retourne la période correspondante.
     * @param d
     * @return
     */
    public Period searchPeriodWithDate (Date d)
    {
        Period period = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
            // Recherche de toutes les périodes
            List<Period> listePeriod = periodDAO.findAll();
            // Fermeture BDD
            topiaContext.closeContext();
            // Recherche de la période
            for (Period periodSearch : listePeriod)
            {
                // Recherche si la date d est entre la date de début et fin de la période
                boolean periodBetweenDate = dateUtil.betweenDate(d, periodSearch.getBeginPeriod(), periodSearch.getEndPeriod());
                // Si la date est comprise dans les limites de la période
                if (periodBetweenDate)
                {
                    period = periodSearch;
                }
            }
        }catch (TopiaException e) {
            log.error(e);
        }
        return period;
    }


    /**
     * Permet de rechercher une période au format DTO avec une date. Cette date
     * peut être quelconque (pas obligatoire d'être le premier jour de la période
     * par exemple).
     * @param d date comprise dans la période qu'on souhaite rechercher
     * @return
     */
    public PeriodDTO searchPeriodDTOWithDate (Date d)
    {
        Period period = searchPeriodWithDate(d);
        // Converti la période en PeriodDTO
        PeriodDTO periodDTO = convertPeriod.periodEntityToDto(period);        
        return periodDTO;
    }


    /**
     * Permet de renvoyer tous les périodes en format DTO, avec tous les timeSpans
     * en format DTO également.
     * @return
     */
    public List<PeriodDTO> getAllPeriod ()
    {
        ArrayList<PeriodDTO> listAllPeriodDTO = new ArrayList<PeriodDTO>();
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
            // Recherche de toutes les périodes
            List<Period> listePeriod = periodDAO.findAll();
            // Transforme les périodes en DTO
            for (Period period : listePeriod)
            {
                PeriodDTO periodDTO = convertPeriod.periodEntityToDto(period);
                listAllPeriodDTO.add(periodDTO);
            }
            // Fermeture BDD
            topiaContext.closeContext();
            }catch (TopiaException e) {
            log.error(e);
        }
        return listAllPeriodDTO;
    }


    /**
     * Recherche une période précise, avec son identifiant topiaId.
     * @param topiaId identifiant de la période
     * @return
     */
    public Period searchPeriodWithTopiaId (String topiaId)
    {
        Period periodSearch = null;
        try {
            // Acces BDD
            TopiaContext topiaContext = rootContext.beginTransaction();
            // Chargement du DAO
            PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
            // Creation du journal
            periodSearch = periodDAO.findByTopiaId(topiaId);
            // Fermeture BDD
            topiaContext.closeContext();
        }catch (TopiaException e) {
            log.error(e);
        }
        return periodSearch;
    }


    /**
     * Efface une période dans la base de données. Tous les timeSpans sont
     * également effacés. Cette méthode est uniquement utilisée pour les tests
     * unitaires.
     * @param period
     * @return
     */
    public String removePeriod (String topiaId)
    {
        String result = ServiceHelper.RESPOND_ERROR;
        Period periodDelete = searchPeriodWithTopiaId(topiaId);
        // Si la période n'existe pas
        if (periodDelete == null)
        {
            if (log.isWarnEnabled()) {
                log.warn("La période "+topiaId+" n'existe pas !");
            }
            result = ServiceHelper.PERIOD_NOT_EXIST;
        }else // Sinon on efface la période
        {
            try {
                // Acces BDD
                TopiaContext topiaContext = rootContext.beginTransaction();
                // Chargement du DAO
                PeriodDAO periodDAO = CallaoDAOHelper.getPeriodDAO(topiaContext);
                // Débloque la période
                periodDelete.setLocked(false);
                // Mise à jour dans la bdd
                periodDAO.update(periodDelete);
                // Création BDD
                topiaContext.commitTransaction();
                /**
                 * Supprime les timeSpans de la période
                 */
                List<TimeSpanDTO> listTimeSpanDTO = timeSpanServiceImpl.searchListTimeSpanDTO(periodDelete);
                // tri la liste des timeSpans dans l'ordre décroissant des dates
                Collections.sort(listTimeSpanDTO,Collections.reverseOrder());
                // Débloque les timeSpans
                for (TimeSpanDTO timeSpanDTO : listTimeSpanDTO)
                {
                    TimeSpan timeSpan = timeSpanServiceImpl.searchTimeSpanByDate(timeSpanDTO);
                    // Si le timeSpan est bloqué
                    if (timeSpanDTO.isLocked())
                    {
                        // débloque le timeSpan
                        timeSpanServiceImpl.unblockTimeSpan(timeSpan);
                    }                    
                }
                // Efface les timeSpans
                for (TimeSpanDTO timeSpanDTO : listTimeSpanDTO)
                {
                    TimeSpan timeSpan = timeSpanServiceImpl.searchTimeSpanByDate(timeSpanDTO);
                    // efface le timeSpan
                    timeSpanServiceImpl.removeTimeSpan(timeSpan);
                }

                /**
                 * Supprime la période
                 */
                periodDAO.delete(periodDelete);
                // Création BDD
                topiaContext.commitTransaction();
                // Fermeture BDD
                topiaContext.closeContext();
                result = ServiceHelper.RESPOND_SUCCESS;
            }catch (TopiaException e) {
                log.error(e);
            }
        }
        return result;
    }


    /**
     * Permet d'effacer une période au format DTO.
     * @param periodDTO la période à supprimer
     * @return
     */
    public String removePeriod (PeriodDTO periodDTO)
    {
        String result = removePeriod (periodDTO.getId());
        return result;
    }


    //@Override
	public String[] getMethods() {
		// TODO Auto-generated method stub
		return null;
	}

	//@Override
	public void destroy() {
		// TODO Auto-generated method stub

	}

	//@Override
	public void init(TopiaContext arg0) {
		// TODO Auto-generated method stub
	}


}
