package fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.frequency;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.type.WeightUnit;
import fr.ifremer.tutti.ui.swing.content.operation.catches.FrequencyConfigurationMode;
import fr.ifremer.tutti.ui.swing.content.operation.catches.benthos.BenthosBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.computable.ComputableData;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIModel;
import fr.ifremer.tutti.util.Weights;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import java.util.Map;
import java.util.Set;

/**
 * @author Tony Chemit - chemit@codelutin.com
 * @since 0.2
 */
public class BenthosFrequencyUIModel extends AbstractTuttiTableUIModel<BenthosBatchRowModel, BenthosFrequencyRowModel, BenthosFrequencyUIModel> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(BenthosFrequencyUIModel.class);

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_BATCH = "batch";

    public static final String PROPERTY_CONFIGURATION_MODE = "configurationMode";

    public static final String PROPERTY_STEP = "step";

    private static final String PROPERTY_MIN_STEP = "minStep";

    private static final String PROPERTY_MAX_STEP = "maxStep";

    public static final String PROPERTY_CAN_GENERATE = "canGenerate";

    public static final String PROPERTY_AUTO_GEN_MODE = "autoGenMode";

    public static final String PROPERTY_RAFALE_MODE = "rafaleMode";

    public static final String PROPERTY_SIMPLE_COUNTING_MODE = "simpleCountingMode";

    public static final String PROPERTY_SIMPLE_COUNT = "simpleCount";

    public static final String PROPERTY_LENGTH_STEP_CARACTERISTIC = "lengthStepCaracteristic";

    public static final String PROPERTY_LENGTH_STEP_CARACTERISTIC_UNIT = "lengthStepCaracteristicUnit";

    public static final String PROPERTY_TOTAL_NUMBER = "totalNumber";

    public static final String PROPERTY_TOTAL_COMPUTED_OR_NOT_WEIGHT = "totalComputedOrNotWeight";

    public static final String PROPERTY_TOTAL_WEIGHT = "totalWeight";

    public static final String PROPERTY_TOTAL_COMPUTED_WEIGHT = "totalComputedWeight";

    public static final String PROPERTY_EMPTY_ROWS = "emptyRows";

    public static final String PROPERTY_NEXT_EDITABLE_ROW_INDEX = "nextEditableRowIndex";

    public static final String PROPERTY_CAN_EDIT_LENGTH_STEP = "canEditLengthStep";

    /**
     * Fill mode.
     *
     * @since 0.2
     */
    protected FrequencyConfigurationMode configurationMode;

    /**
     * Batch that contains frequencies.
     *
     * @since 0.2
     */
    protected BenthosBatchRowModel batch;

    /**
     * Default step to increment length step.
     *
     * @since 0.2
     */
    protected Float step;

    /**
     * Min step to auto generate length steps.
     *
     * @since 0.2
     */
    protected Float minStep;

    /**
     * Max step to auto generate length steps.
     *
     * @since 0.2
     */
    protected Float maxStep;

    /**
     * Length step caracteristic.
     *
     * @since 0.3
     */
    protected Caracteristic lengthStepCaracteristic;

    /**
     * Number in case of simple counting mode
     *
     * @since 1.0
     */
    protected Integer simpleCount;

    /**
     * Sum of the number of each valid row
     *
     * @since 2.3
     */
    protected Integer totalNumber;

    /**
     * Sum of the weight of each valid row, or sample weight
     *
     * @since 3.8
     */
    protected ComputableData<Float> totalComputedOrNotWeight = new ComputableData<Float>();

    /**
     * The index of the next editable row (null if none).
     *
     * @since 2.5
     */
    protected Integer nextEditableRowIndex;

    protected Set<BenthosFrequencyRowModel> emptyRows;

    /**
     * Sample categories model.
     *
     * @since 2.4
     */
    protected final SampleCategoryModel sampleCategoryModel;

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    /**
     * To store all caches used by the screen
     *
     * @since 3.11
     */
    protected final BenthosFrequencyUIModelCache cache = new BenthosFrequencyUIModelCache();

    /**
     * Can edit length step? (only if no row is filled).
     * see https://forge.codelutin.com/issues/5694
     *
     * @since 3.11
     */
    protected boolean canEditLengthStep = true;

    /**
     * To store graph series.
     *
     * @since 3.11
     */
    protected final XYSeriesCollection dataset;

    public BenthosFrequencyUIModel(WeightUnit weightUnit,
                                   SampleCategoryModel sampleCategoryModel) {
        super(BenthosBatchRowModel.class, null, null);
        this.weightUnit = weightUnit;
        this.sampleCategoryModel = sampleCategoryModel;
        this.totalComputedOrNotWeight.addPropagateListener(PROPERTY_TOTAL_WEIGHT, this);
        setEmptyRows(Sets.<BenthosFrequencyRowModel>newHashSet());

        XYSeries series = new XYSeries("", true, false);

        dataset = new XYSeriesCollection(series);
        dataset.setIntervalPositionFactor(0);
        dataset.setIntervalWidth(0);
    }

    public void reloadRows() {

        setEmptyRows(Sets.<BenthosFrequencyRowModel>newHashSet());

        XYSeries series = dataset.getSeries(0);
        series.clear();

        cache.loadCache(rows);

        recomputeRowsValidateState();

        if (rows != null) {

            for (BenthosFrequencyRowModel row : rows) {

                if (row.isValid()) {

                    Float lengthStep = row.getLengthStep();
                    series.addOrUpdate(lengthStep, row.getNumber());

                }

            }

        }

        recomputeTotalNumberAndWeight();

    }

    public boolean isRowValid(BenthosFrequencyRowModel row) {

        // lengthStepCaracteristic conditions : not null
        boolean valid = row.getLengthStepCaracteristic() != null;

        if (valid) {

            // lengthStep conditions : not null and positive + not found in more than one row
            Float lengthStep = row.getLengthStep();
            valid = lengthStep != null
                    && lengthStep > 0
                    && numberOfRows(lengthStep) < 2;
        }

        if (valid) {

            // number conditions : not null and positive number
            Integer number = row.getNumber();
            valid = number != null && number > 0;

        }

        if (valid) {

            // weight conditions
            Float weight = row.getWeight();
            valid = getNbRowsWithWeight() == 0 || (weight != null && weight > 0);

        }

        return valid;
    }

    @Override
    protected BenthosBatchRowModel newEntity() {
        return new BenthosBatchRowModel(weightUnit, sampleCategoryModel);
    }

    public FrequencyConfigurationMode getConfigurationMode() {
        return configurationMode;
    }

    public void setConfigurationMode(FrequencyConfigurationMode configurationMode) {
        Object oldValue = getConfigurationMode();
        this.configurationMode = configurationMode;
        firePropertyChange(PROPERTY_CONFIGURATION_MODE, oldValue, configurationMode);
        firePropertyChange(PROPERTY_AUTO_GEN_MODE, null, isAutoGenMode());
        firePropertyChange(PROPERTY_RAFALE_MODE, null, isRafaleMode());
        firePropertyChange(PROPERTY_SIMPLE_COUNTING_MODE, null, isSimpleCountingMode());
    }

    public Float getStep() {
        return step;
    }

    public void setStep(Float step) {
        Object oldValue = getStep();
        this.step = step;
        firePropertyChange(PROPERTY_STEP, oldValue, step);
    }

    public Caracteristic getLengthStepCaracteristic() {
        return lengthStepCaracteristic;
    }

    public void setLengthStepCaracteristic(Caracteristic lengthStepCaracteristic) {
        Object oldValue = getLengthStepCaracteristic();
        this.lengthStepCaracteristic = lengthStepCaracteristic;
        firePropertyChange(PROPERTY_LENGTH_STEP_CARACTERISTIC, oldValue, lengthStepCaracteristic);
        firePropertyChange(PROPERTY_CAN_GENERATE, null, isCanGenerate());
        firePropertyChange(PROPERTY_LENGTH_STEP_CARACTERISTIC_UNIT, null, getLengthStepCaracteristicUnit());
    }

    public String getLengthStepCaracteristicUnit() {
        return lengthStepCaracteristic == null ? null : lengthStepCaracteristic.getUnit();
    }

    public Float getLengthStepCaracteristicPrecision() {
        return lengthStepCaracteristic == null ? null : lengthStepCaracteristic.getPrecision();
    }

    public Float getMinStep() {
        return minStep;
    }

    public void setMinStep(Float minStep) {
        Object oldValue = getMinStep();
        this.minStep = minStep;
        firePropertyChange(PROPERTY_MIN_STEP, oldValue, minStep);
        firePropertyChange(PROPERTY_CAN_GENERATE, null, isCanGenerate());
    }

    public Float getMaxStep() {
        return maxStep;
    }

    public void setMaxStep(Float maxStep) {
        Object oldValue = getMaxStep();
        this.maxStep = maxStep;
        firePropertyChange(PROPERTY_MAX_STEP, oldValue, maxStep);
        firePropertyChange(PROPERTY_CAN_GENERATE, null, isCanGenerate());
    }

    public Integer getSimpleCount() {
        return simpleCount;
    }

    public void setSimpleCount(Integer simpleCount) {
        Object oldValue = getSimpleCount();
        this.simpleCount = simpleCount;
        firePropertyChange(PROPERTY_SIMPLE_COUNT, oldValue, simpleCount);
    }

    public boolean isCanEditLengthStep() {
        return canEditLengthStep;
    }

    public void setCanEditLengthStep(boolean canEditLengthStep) {
        Object oldValue = isCanEditLengthStep();
        this.canEditLengthStep = canEditLengthStep;
        firePropertyChange(PROPERTY_CAN_EDIT_LENGTH_STEP, oldValue, canEditLengthStep);
    }

    public Integer getNextEditableRowIndex() {
        return nextEditableRowIndex;
    }

    public void setNextEditableRowIndex(Integer nextEditableRowIndex) {
        Object oldValue = getNextEditableRowIndex();
        this.nextEditableRowIndex = nextEditableRowIndex;
        firePropertyChange(PROPERTY_NEXT_EDITABLE_ROW_INDEX, oldValue, nextEditableRowIndex);
    }

    public boolean isAutoGenMode() {
        return FrequencyConfigurationMode.AUTO_GEN == configurationMode;
    }

    public boolean isRafaleMode() {
        return FrequencyConfigurationMode.RAFALE == configurationMode;
    }

    public boolean isSimpleCountingMode() {
        return FrequencyConfigurationMode.SIMPLE_COUNTING == configurationMode;
    }

    public boolean isCanGenerate() {
        return minStep != null && maxStep != null && maxStep > minStep && lengthStepCaracteristic != null;
    }

    public BenthosBatchRowModel getBatch() {
        return batch;
    }

    public void setBatch(BenthosBatchRowModel batch) {
        this.batch = batch;
        firePropertyChange(PROPERTY_BATCH, null, batch);
    }

    public float getLengthStep(float lengthStep) {
        int intValue = (int) (lengthStep * 10);
        int intStep = (int) (step * 10);
        int correctIntStep = intValue - (intValue % intStep);
        float result = correctIntStep / 10f;
        return result;
    }

    public Integer getTotalNumber() {
        return totalNumber;
    }

    public void setTotalNumber(Integer totalNumber) {
        Object oldValue = getTotalNumber();
        this.totalNumber = totalNumber;
        firePropertyChange(PROPERTY_TOTAL_NUMBER, oldValue, totalNumber);
    }

    public ComputableData<Float> getTotalComputedOrNotWeight() {
        return totalComputedOrNotWeight;
    }

    public void setTotalComputedOrNotWeight(ComputableData<Float> totalComputedOrNotWeight) {
        Object oldValue = getTotalComputedOrNotWeight();
        this.totalComputedOrNotWeight = totalComputedOrNotWeight;
        firePropertyChange(PROPERTY_TOTAL_COMPUTED_OR_NOT_WEIGHT, oldValue, totalComputedOrNotWeight);
    }

    public Float getTotalWeight() {
        return totalComputedOrNotWeight.getData();
    }

    public void setTotalWeight(Float totalWeight) {
        Object oldValue = getTotalWeight();
        this.totalComputedOrNotWeight.setData(totalWeight);
        firePropertyChange(PROPERTY_TOTAL_WEIGHT, oldValue, totalWeight);
    }

    public Float getTotalComputedWeight() {
        return totalComputedOrNotWeight.getComputedData();
    }

    public void setTotalComputedWeight(Float totalComputedWeight) {
        Object oldValue = getTotalComputedWeight();
        this.totalComputedOrNotWeight.setComputedData(totalComputedWeight);
        firePropertyChange(PROPERTY_TOTAL_COMPUTED_WEIGHT, oldValue, totalComputedWeight);
    }

    public boolean isTotalWeightSameAsComputedWeight() {
        Float totalWeight = getTotalWeight();
        Float computedWeight = getTotalComputedWeight();
        return totalWeight != null && computedWeight != null
               && Weights.isEqualWeight(totalWeight, computedWeight);
    }

    public Set<BenthosFrequencyRowModel> getEmptyRows() {
        return emptyRows;
    }

    public void setEmptyRows(Set<BenthosFrequencyRowModel> emptyRows) {
        this.emptyRows = emptyRows;
        firePropertyChange(PROPERTY_EMPTY_ROWS, null, emptyRows);
    }

    public int getNbRowsWithWeight() {
        return cache.getNbRowsWithWeight();
    }

    public boolean isAllRowsWithWeight() {
        return cache.getNbRowsWithWeight() == rows.size();
    }

    public boolean isSomeRowsWithWeightAndOtherWithout() {

        boolean result;

        if (CollectionUtils.isEmpty(rows)) {

            // no row
            result = false;

        } else {

            // there is some rows

            int nbNoneEmptyRows = 0;
            int nbNoneEmptyRowsWithWeight = 0;

            for (BenthosFrequencyRowModel row : rows) {

                if (row.isEmpty()) {

                    // no value on rows, no check on it
                    continue;
                }

                nbNoneEmptyRows++;

                if (row.getWeight() != null) {

                    nbNoneEmptyRowsWithWeight++;

                }

            }

            result = nbNoneEmptyRowsWithWeight > 0 && nbNoneEmptyRows != nbNoneEmptyRowsWithWeight;

        }

        return result;

    }

    public void updateEmptyRow(BenthosFrequencyRowModel row) {
        if (row.isValid() && row.getNumber() == null && row.getWeight() == null) {
            emptyRows.add(row);
        } else {
            emptyRows.remove(row);
        }
        firePropertyChange(PROPERTY_EMPTY_ROWS, null, emptyRows);
    }

    public int computeTotalNumber() {
        int result = 0;
        if (rows != null) {
            for (BenthosFrequencyRowModel row : rows) {
                if (!row.isValid()) {
                    continue;
                }
                if (row.getNumber() != null) {
                    result += row.getNumber();
                }
            }
        }
        return result;
    }

    public void recomputeTotalNumberAndWeight() {

        int computeTotalNumber = computeTotalNumber();
        Float computeTotalWeight = cache.computeTotalWeight();
        setTotalNumber(computeTotalNumber);
        setTotalComputedWeight(computeTotalWeight);

    }

    public void recomputeCanEditLengthStep() {

        boolean result = true;

        for (BenthosFrequencyRowModel row : rows) {

            if (row.isEmpty()) {
                // la ligne est vide
                continue;
            }
            if (row.getLengthStep() == null || row.getNumber() == null) {
                // la ligne n'est pas complete
                continue;
            }

            // une ligne non vide et complete a ete trouvee
            // on ne peut plus editer
            result = false;

        }

        setCanEditLengthStep(result);

    }

    public int numberOfRows(float lengthStep) {
        return cache.numberOfRows(lengthStep);
    }

    public Map<Float, BenthosFrequencyRowModel> getRowCache() {
        return cache.getRowCache();
    }

    protected final void recomputeRowValidState(BenthosFrequencyRowModel row) {

        // recompute row valid state
        boolean valid = isRowValid(row);

        // apply it to row
        row.setValid(valid);

        if (valid) {
            removeRowInError(row);
        } else {
            addRowInError(row);
        }

    }

    public void recomputeRowsValidateState() {

        if (log.isInfoEnabled()) {
            log.info("Revalidate all rows");
        }

        for (BenthosFrequencyRowModel r : rows) {
            recomputeRowValidState(r);
        }

    }

    public void setDataSetIntervalWidth(float step) {
        dataset.setIntervalWidth(step);
    }
}
