package fr.ifremer.tutti.ui.swing.content.operation.catches.species.split;

/*
 * #%L
 * Tutti :: UI
 * $Id: SplitSpeciesBatchUIHandler.java 1564 2014-01-31 10:56:26Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.4.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/split/SplitSpeciesBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.nuiton.jaxx.application.swing.util.Cancelable;
import fr.ifremer.tutti.persistence.entities.data.SampleCategory;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModelEntry;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.List;
import java.util.Map;

/**
 * Handler of {@link SplitSpeciesBatchUI}.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class SplitSpeciesBatchUIHandler
        extends AbstractTuttiTableUIHandler<SplitSpeciesBatchRowModel, SplitSpeciesBatchUIModel, SplitSpeciesBatchUI> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SplitSpeciesBatchUIHandler.class);

    public SplitSpeciesBatchUIHandler() {
        super(SplitSpeciesBatchRowModel.PROPERTY_SELECTED,
              SplitSpeciesBatchRowModel.PROPERTY_CATEGORY_VALUE,
              SplitSpeciesBatchRowModel.PROPERTY_WEIGHT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public SplitSpeciesBatchTableModel getTableModel() {
        return (SplitSpeciesBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SplitSpeciesBatchRowModel row) {
        return row.isSelected();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SplitSpeciesBatchRowModel> rowMonitor,
                                             SplitSpeciesBatchRowModel row) {
        if (rowMonitor.wasModified()) {

            if (row.isValid()) {
                if (log.isInfoEnabled()) {
                    log.info("Change row that was modified and valid");
                }
            }

            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             SplitSpeciesBatchRowModel oldRow,
                                             int newRowIndex,
                                             SplitSpeciesBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);
        if (newRow != null) {

            // Recompute the valid state of the row
            recomputeRowValidState(newRow);

            // Need to recompute the sample weight
            computeSampleWeight();
        }
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SplitSpeciesBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        // Recompute the valid state of the row
        recomputeRowValidState(row);

        // Need to recompute the sample weight
        computeSampleWeight();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<SplitSpeciesBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInit(SplitSpeciesBatchUI ui) {
        super.beforeInit(ui);
        SampleCategoryModel sampleCategoryModel =
                getDataContext().getSampleCategoryModel();

        SplitSpeciesBatchUIModel model = new SplitSpeciesBatchUIModel(
                getConfig().getSpeciesWeightUnit(), sampleCategoryModel);

        ui.setContextValue(model);
    }

    @Override
    public void afterInit(SplitSpeciesBatchUI ui) {

        initUI(ui);

        SplitSpeciesBatchUIModel model = getModel();

        // when category changed, remove selected category
        model.addPropertyChangeListener(SplitSpeciesBatchUIModel.PROPERTY_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                SplitSpeciesBatchUIModel source =
                        (SplitSpeciesBatchUIModel) evt.getSource();

                // unselect previous selected category
                source.setSelectedCategory(null);

                // fill comboBox with new list
                List<SampleCategoryModelEntry> data = (List<SampleCategoryModelEntry>) evt.getNewValue();
                SplitSpeciesBatchUIHandler.this.ui.getCategoryComboBox().setModel(new DefaultComboBoxModel(data.toArray()));
            }
        });

        // when selected category changed, regenerate the table model + add inside some default rows
        model.addPropertyChangeListener(SplitSpeciesBatchUIModel.PROPERTY_SELECTED_CATEGORY, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                SplitSpeciesBatchUIModel source =
                        (SplitSpeciesBatchUIModel) evt.getSource();

                // when selected category change, sample total weight is reset
                source.setSampleWeight(null);

                SampleCategoryModelEntry newValue =
                        (SampleCategoryModelEntry) evt.getNewValue();
                generateTableModel(newValue);
            }
        });

        ui.getCategoryComboBox().setRenderer(newListCellRender(SampleCategoryModelEntry.class));

        ui.getCategoryComboBox().addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                JComboBox comboBox = (JComboBox) e.getSource();
                getModel().setSelectedCategory((SampleCategoryModelEntry) comboBox.getSelectedItem());
            }
        });

        generateTableModel(null);

        initTable(getTable());

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getCategoryComboBox();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model
        getModel().setValid(false);
        getModel().setSelectedCategory(null);

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        parent.getHandler().setSpeciesSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isDebugEnabled()) {
            log.debug("Cancel UI " + ui);
        }
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void editBatch(SpeciesBatchRowModel batch) {

        // get possible the last used
        List<SampleCategoryModelEntry> categories = Lists.newArrayList();

        SampleCategoryModelEntry bestSampleCategory = null;
        if (batch != null) {

            // get sample category model
            SampleCategoryModel sampleCategoryModel =
                    getModel().getSampleCategoryModel();

            // set all categories (the no more available will be removed later)
            categories.addAll(sampleCategoryModel.getCategory());

            SampleCategory<?> lastCategory = batch.getFinestCategory();

            Preconditions.checkNotNull(
                    lastCategory,
                    "Can't split a species batch with no sample category.");

            int firstOrder = lastCategory.getCategoryDef().getOrder();

            for (Integer sampleCategoryId : sampleCategoryModel.getSamplingOrder()) {
                SampleCategory<?> sampleCategory = batch.getSampleCategoryById(sampleCategoryId);
                int order = sampleCategory.getCategoryDef().getOrder();
                if (order < firstOrder || sampleCategory.isValid()) {

                    // remove category if before the finest one
                    // remove category if not filled
                    categories.remove(sampleCategory.getCategoryDef());
                }
            }

            bestSampleCategory = getDataContext().getBestFirstSpeciesSampleCategory(categories,
                                                                                    batch.getSpecies());
        }

        SplitSpeciesBatchUIModel model = getModel();

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setSampleWeight(null);
        model.setCategory(categories);
        model.setSelectedCategory(bestSampleCategory);

        // keep batch (will be used to push back editing entry)
        model.setBatch(batch);
    }

    public void editBatch(SpeciesBatchRowModel batch,
                          int sampleCategoryId) {

        Preconditions.checkNotNull(batch);
        Preconditions.checkNotNull(sampleCategoryId);

        List<SpeciesBatchRowModel> rows = batch.getChildBatch();
        Preconditions.checkNotNull(rows);

        // get possible the last used
        List<SampleCategoryModelEntry> categories = Lists.newArrayList();

        // get sample category model
        SampleCategoryModel sampleCategoryModel =
                getModel().getSampleCategoryModel();

        // set only the given category
        SampleCategoryModelEntry selectedCategory =
                sampleCategoryModel.getCategoryById(sampleCategoryId);
        categories.add(selectedCategory);

        SplitSpeciesBatchUIModel model = getModel();

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setSampleWeight(null);
        model.setCategory(categories);
        model.setSelectedCategory(selectedCategory);

        // keep batch (will be used to push back editing entry)
        model.setBatch(batch);

        // add existing rows
        Map<Serializable, SplitSpeciesBatchRowModel> rowsByValue = Maps.uniqueIndex(model.getRows(), new Function<SplitSpeciesBatchRowModel, Serializable>() {
            @Override
            public Serializable apply(SplitSpeciesBatchRowModel input) {
                return input.getCategoryValue();
            }
        });
        for (SpeciesBatchRowModel row : rows) {

            SampleCategory<?> sampleCategory = row.getSampleCategoryById(sampleCategoryId);
            Serializable categoryValue = sampleCategory.getCategoryValue();
            SplitSpeciesBatchRowModel splitRow = rowsByValue.get(categoryValue);
            splitRow.setWeight(sampleCategory.getCategoryWeight());
            splitRow.setSelected(true);
            splitRow.setEditable(false);
        }

        computeSampleWeight();

        getTableModel().fireTableDataChanged();
    }

    public void save() {

        if (log.isDebugEnabled()) {
            log.debug("Save UI " + ui);
        }

        EditCatchesUI parent = getParentContainer(EditCatchesUI.class);
        SplitSpeciesBatchUIModel model = getModel();
        if (model.isValid()) {
            if (model.isSplitMode()) {
                parent.getSpeciesTabContent().getHandler().splitBatch(
                        model.getSelectedCategory(),
                        model.getRows(),
                        model.getSampleWeight());
            } else {
                parent.getSpeciesTabContent().getHandler().addSampleCategoryBatch(
                        model.getBatch(),
                        model.getSelectedCategory(),
                        model.getRows(),
                        model.getSampleWeight());
            }
        }

        // close dialog
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void computeSampleWeight() {

        Float result = null;
        List<SplitSpeciesBatchRowModel> rows = getTableModel().getRows();
        for (SplitSpeciesBatchRowModel row : rows) {
            if (row.isSelected()) {
                Float weight = row.getWeight();
                if (weight != null) {
                    if (result == null) {
                        result = 0f;
                    }
                    result += weight;
                }
            }
        }
        getModel().setSampleWeight(result);
    }

    protected void generateTableModel(SampleCategoryModelEntry category) {

        // when generate a new table model, then reset previous rows from model
        getModel().setRows(null);

        Caracteristic data = null;

        JXTable table = getTable();

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        { // Selection

            addBooleanColumnToModel(columnModel,
                                    SplitSpeciesBatchTableModel.SELECTED,
                                    table);
        }

        boolean editableCategoryValue = false;
        if (category != null) {

            if (!category.getCaracteristic().isQualitativeValueEmpty()) {

                // qualitative category
                data = category.getCaracteristic();
            } else {
                editableCategoryValue = true;
                addFloatColumnToModel(columnModel,
                                      SplitSpeciesBatchTableModel.EDITABLE_CATEGORY_VALUE,
                                      TuttiUI.DECIMAL1_PATTERN,
                                      table);
            }

            if (data != null) {

                if (log.isDebugEnabled()) {
                    log.debug("Got " + data.sizeQualitativeValue() + " qualitative data to add");
                }
                addColumnToModel(columnModel,
                                 null,
                                 newTableCellRender(CaracteristicQualitativeValue.class),
                                 SplitSpeciesBatchTableModel.READ_ONLY_CATEGORY_VALUE);
            }
            { // Weight

                addFloatColumnToModel(columnModel,
                                      SplitSpeciesBatchTableModel.WEIGHT,
                                      getConfig().getSpeciesWeightUnit(),
                                      table);
            }
        }

        // create table model
        SplitSpeciesBatchTableModel tableModel =
                new SplitSpeciesBatchTableModel(columnModel,
                                                getModel(),
                                                editableCategoryValue,
                                                getModel().isSplitMode());

        // remove all listener on tables we could add before
        uninstallTableSaveOnRowChangedSelectionListener();
        uninstallTableKeyListener();

        if (log.isDebugEnabled()) {
            log.debug("Install new table model " + tableModel);
        }
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        // install table listeners
        installTableSaveOnRowChangedSelectionListener();
        installTableKeyListener(columnModel, table);

        // fill datas

        List<SplitSpeciesBatchRowModel> rows = Lists.newArrayList();

        if (data != null) {

            // add a row for each qualitative value
            for (CaracteristicQualitativeValue qualitativeValue : data.getQualitativeValue()) {
                if (log.isDebugEnabled()) {
                    log.debug("Add QV: " + qualitativeValue);
                }
                SplitSpeciesBatchRowModel newRow = tableModel.createNewRow();
                newRow.setCategoryValue(qualitativeValue);
                rows.add(newRow);
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will add " + rows.size() + " rows in table model " +
                      "(can add a first empty row? " + editableCategoryValue + ").");
        }

        getModel().setRows(rows);
    }

}
