package fr.ifremer.tutti.ui.swing.content.protocol;

/*
 * #%L
 * Tutti :: UI
 * $Id: EditProtocolUIHandler.java 1611 2014-02-21 10:36:09Z tchemit $
 * $HeadURL: https://svn.codelutin.com/tutti/tags/tutti-3.2.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/protocol/EditProtocolUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModel;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryModelEntry;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.TuttiDecorator;
import fr.ifremer.tutti.ui.swing.action.CloneProtocolAction;
import fr.ifremer.tutti.ui.swing.action.EditProtocolAction;
import fr.ifremer.tutti.ui.swing.action.ImportProtocolAction;
import fr.ifremer.tutti.ui.swing.content.operation.catches.SpeciesBatchRowHelper;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUI;
import fr.ifremer.tutti.ui.swing.util.species.SelectSpeciesUIModel;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.bean.BeanDoubleList;
import jaxx.runtime.swing.editor.bean.BeanDoubleListModel;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.swing.editor.bean.BeanUIUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.autocomplete.ComboBoxCellEditor;
import org.jdesktop.swingx.autocomplete.ObjectToStringConverter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.jdesktop.swingx.table.TableColumnExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.jaxx.application.swing.util.CloseableUI;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JTabbedPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static org.nuiton.i18n.I18n.t;


/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.3
 */
public class EditProtocolUIHandler extends AbstractTuttiUIHandler<EditProtocolUIModel, EditProtocolUI> implements CloseableUI {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EditProtocolUIHandler.class);

    public static String getTitle(boolean exist) {

        String result;
        if (exist) {
            result = t("tutti.editProtocol.title.edit.protocol");
        } else {
            result = t("tutti.editProtocol.title.create.protocol");
        }
        return result;
    }

    protected SelectSpeciesUI dialog;

    protected SampleCategoryModel sampleCategoryModel;

    protected List<BeanDoubleList<Caracteristic>> allDoubleLists;

    public JXTable getSpeciesTable() {
        return ui.getSpeciesTable();
    }

    public JXTable getBenthosTable() {
        return ui.getBenthosTable();
    }

    public EditProtocolSpeciesTableModel getSpeciesTableModel() {
        return (EditProtocolSpeciesTableModel) getSpeciesTable().getModel();
    }

    public EditProtocolSpeciesTableModel getBenthosTableModel() {
        return (EditProtocolSpeciesTableModel) getBenthosTable().getModel();
    }

    public boolean isSpeciesSelected(Object selectedItem) {
        return selectedItem != null && selectedItem instanceof Species;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<EditProtocolUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInit(EditProtocolUI ui) {
        super.beforeInit(ui);

        this.sampleCategoryModel = getContext().getDataContext().getSampleCategoryModel();

        incrementsMessage("Chargement des réferentiels");

        getDataContext().resetValidationDataContext();

        EditProtocolUIModel model = new EditProtocolUIModel();

        // load cache data

        List<Species> allSpecies = Lists.newArrayList(getDataContext().getSpecies());
        model.setAllSpecies(allSpecies);

        Multimap<String, Species> allSpeciesByTaxonId =
                TuttiEntities.splitByReferenceTaxonId(allSpecies);
        model.setAllSpeciesByTaxonId(allSpeciesByTaxonId);

        Map<String, Species> allReferentSpeciesByTaxonId = TuttiEntities.splitByTaxonId(
                getDataContext().getReferentSpecies());
        model.setAllReferentSpeciesByTaxonId(allReferentSpeciesByTaxonId);

        List<Caracteristic> caracteristics = Lists.newArrayList(getDataContext().getCaracteristics());
        model.setCaracteristics(caracteristics);

        Map<String, Caracteristic> allCaracteristic = TuttiEntities.splitById(caracteristics);
        model.setAllCaracteristic(allCaracteristic);

        // can't load directly model from database here, since we want to
        // fill only the model with rows (transformed from speciesProtocol)
        // As we still don't have the table model at this point, wait the
        // afterUI method to fill model

        listModelIsModify(model);
        this.ui.setContextValue(model);
    }

    @Override
    public void afterInit(EditProtocolUI ui) {

        initUI(this.ui);

        EditProtocolUIModel model = getModel();

        TuttiProtocol protocol;

        if (getContext().isProtocolFilled()) {

            // load existing protocol

            incrementsMessage("Chargement du protocole");

            protocol = getDataContext().getProtocol();

            if (EditProtocolAction.CLEAN_PROTOCOL_ENTRY.getContextValue(this.ui) != null) {

                // clean protocol
                protocol = EditProtocolAction.CLEAN_PROTOCOL_ENTRY.getContextValue(this.ui);

                EditProtocolAction.CLEAN_PROTOCOL_ENTRY.removeContextValue(this.ui);

                model.setCleaned(true);
                this.ui.getSaveWarning().setText(t("tutti.editProtocol.warn.clean"));
            }
            model.fromBean(protocol);

        } else if ((protocol = ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.getContextValue(this.ui)) != null) {

            incrementsMessage("Import du protocole");

            // import protocol

            ImportProtocolAction.IMPORT_PROTOCOL_ENTRY.removeContextValue(this.ui);

            model.fromBean(protocol);
            model.setImported(true);
            this.ui.getSaveWarning().setText(t("tutti.editProtocol.warn.import"));

        } else if ((protocol = CloneProtocolAction.CLONE_PROTOCOL_ENTRY.getContextValue(this.ui)) != null) {

            incrementsMessage("Clone du protocole");

            // clone protocol

            CloneProtocolAction.CLONE_PROTOCOL_ENTRY.removeContextValue(this.ui);

            model.fromBean(protocol);
            model.setCloned(true);
            this.ui.getSaveWarning().setText(t("tutti.editProtocol.warn.clone"));

        } else {

            incrementsMessage("Création d'un nouveau protocol");

            // create new protocol

            if (log.isDebugEnabled()) {
                log.debug("Will create a new protocol");
            }
        }

        if (model.getLengthClassesPmfmId() == null) {
            model.setLengthClassesPmfmId(Lists.<String>newArrayList());
        }

        SwingValidator validator = this.ui.getValidator();
        listenValidatorValid(validator, model);

        registerValidators(validator);

        Collection<Species> referents =
                model.getAllReferentSpeciesByTaxonId().values();

        initBeanFilterableComboBox(this.ui.getSpeciesComboBox(),
                                   Lists.newArrayList(referents), null);
        initBeanFilterableComboBox(this.ui.getBenthosComboBox(),
                                   Lists.newArrayList(referents), null);

        List<EditProtocolSpeciesRowModel> speciesRows;
        List<EditProtocolSpeciesRowModel> benthosRows;

        incrementsMessage("Préparation des interfaces graphiques");

        // build species and benthos rows
        if (protocol == null) {
            speciesRows = Lists.newArrayList();
            benthosRows = Lists.newArrayList();
        } else {

            speciesRows = toRows(protocol.getSpecies());
            benthosRows = toRows(protocol.getBenthos());

            if (log.isDebugEnabled()) {
                log.debug("Will edit protocol with " +
                          speciesRows.size() + " species and " +
                          benthosRows.size() + " benthos declared.");
            }
        }

        // set to model (will propagate to tableModel)
        model.setSpeciesRow(speciesRows);
        model.setBenthosRow(benthosRows);

        this.ui.getSpeciesComboBox().reset();
        this.ui.getBenthosComboBox().reset();

        {
            // create species table model

            JXTable table = getSpeciesTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            TableColumnExt speciesColumn = addColumnToModel(columnModel,
                                                            null,
                                                            newTableCellRender(Species.class),
                                                            EditProtocolSpeciesTableModel.SPECIES_ID);
            speciesColumn.setSortable(true);
            DecoratorService.SpeciesDecorator speciesDecorator = new DecoratorService.SpeciesDecorator();
            speciesColumn.putClientProperty(SpeciesBatchRowHelper.SPECIES_DECORATOR, speciesDecorator);
            speciesColumn.setCellRenderer(newTableCellRender(speciesDecorator));

            TableColumnExt speciesSurveyCodeColumn = addColumnToModel(columnModel,
                                                                      null,
                                                                      null,
                                                                      EditProtocolSpeciesTableModel.SURVEY_CODE_ID);
            speciesSurveyCodeColumn.setSortable(true);

            addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

            for (SampleCategoryModelEntry sampleCategoryModelEntry : sampleCategoryModel.getCategory()) {

                if (sampleCategoryModelEntry.getOrder() == 0) {
                    // first category is not editable
                    continue;
                }

                MandatorySampleCategoryColumnIdentifier identifier = MandatorySampleCategoryColumnIdentifier.newId(
                        EditProtocolSpeciesRowModel.PROPERTY_MANDATORY_SAMPLE_CATEGORY_ID,
                        sampleCategoryModelEntry.getCategoryId(),
                        sampleCategoryModelEntry.getLabel(),
                        sampleCategoryModelEntry.getLabel()
                );

                addBooleanColumnToModel(columnModel, identifier, table);
            }

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);

            initTable(table,
                      columnModel,
                      speciesColumn,
                      speciesRows,
                      new ListSelectionListener() {
                          @Override
                          public void valueChanged(ListSelectionEvent e) {
                              ListSelectionModel source = (ListSelectionModel) e.getSource();
                              getModel().setRemoveSpeciesEnabled(!source.isSelectionEmpty());
                          }
                      });
        }

        {
            // create benthos table model

            JXTable table = getBenthosTable();

            DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

            TableColumnExt speciesColumn = addColumnToModel(columnModel,
                                                            null,
                                                            newTableCellRender(Species.class),
                                                            EditProtocolSpeciesTableModel.SPECIES_ID);
            speciesColumn.setSortable(true);
            DecoratorService.SpeciesDecorator speciesDecorator = new DecoratorService.SpeciesDecorator();
            speciesColumn.putClientProperty(SpeciesBatchRowHelper.SPECIES_DECORATOR, speciesDecorator);
            speciesColumn.setCellRenderer(newTableCellRender(speciesDecorator));

            TableColumnExt speciesSurveyCodeColumn = addColumnToModel(columnModel,
                                                                      null,
                                                                      null,
                                                                      EditProtocolSpeciesTableModel.SURVEY_CODE_ID);
            speciesSurveyCodeColumn.setSortable(true);

            addLengthClassesColumnToModel(columnModel, model.getLengthClassesPmfmId());

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.WEIGHT_ENABLED, table);
            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.COUNT_IF_NO_FREQUENCY_ENABLED, table);

            for (SampleCategoryModelEntry sampleCategoryModelEntry : sampleCategoryModel.getCategory()) {

                if (sampleCategoryModelEntry.getOrder() == 0) {
                    // first category is not editable
                    continue;
                }

                MandatorySampleCategoryColumnIdentifier identifier = MandatorySampleCategoryColumnIdentifier.newId(
                        EditProtocolSpeciesRowModel.PROPERTY_MANDATORY_SAMPLE_CATEGORY_ID,
                        sampleCategoryModelEntry.getCategoryId(),
                        sampleCategoryModelEntry.getLabel(),
                        sampleCategoryModelEntry.getLabel()
                );

                addBooleanColumnToModel(columnModel, identifier, table);
            }

            addBooleanColumnToModel(columnModel, EditProtocolSpeciesTableModel.CALCIFY_SAMPLE_ENABLED, table);

            initTable(table,
                      columnModel,
                      speciesColumn,
                      benthosRows,
                      new ListSelectionListener() {
                          @Override
                          public void valueChanged(ListSelectionEvent e) {
                              ListSelectionModel source = (ListSelectionModel) e.getSource();
                              getModel().setRemoveBenthosEnabled(!source.isSelectionEmpty());
                          }
                      });
        }

        allDoubleLists = Lists.newArrayList(
                this.ui.getLengthClassesList(),
                this.ui.getGearUseFeatureList(),
                this.ui.getIndividualObservationList(),
                this.ui.getVesselUseFeatureList()
        );

        initDoubleList(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID,
                       this.ui.getLengthClassesList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getLengthClassesPmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_GEAR_USE_FEATURE_PMFM_ID,
                       this.ui.getGearUseFeatureList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getGearUseFeaturePmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_VESSEL_USE_FEATURE_PMFM_ID,
                       this.ui.getVesselUseFeatureList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getVesselUseFeaturePmfmId());

        initDoubleList(EditProtocolUIModel.PROPERTY_INDIVIDUAL_OBSERVATION_PMFM_ID,
                       this.ui.getIndividualObservationList(),
                       Lists.newArrayList(model.getCaracteristics()),
                       model.getIndividualObservationPmfmId());

        // if new protocol can already cancel his creation
        model.setModify(model.isCreate() || model.isCleaned());

        this.ui.getCaracteristicPane().addChangeListener(new ChangeListener() {

            public void stateChanged(ChangeEvent e) {
                JTabbedPane tabPanel = (JTabbedPane) e.getSource();
                int selectedIndex = tabPanel.getSelectedIndex();
                BeanDoubleList<Caracteristic> selectedDoubleList;
                log.debug("selected " + selectedIndex);
                switch (selectedIndex) {
                    case 0:
                        selectedDoubleList = EditProtocolUIHandler.this.ui.getLengthClassesList();
                        break;

                    case 1:
                        selectedDoubleList = EditProtocolUIHandler.this.ui.getGearUseFeatureList();
                        break;

                    case 2:
                        selectedDoubleList = EditProtocolUIHandler.this.ui.getIndividualObservationList();
                        break;

                    case 3:
                        selectedDoubleList = EditProtocolUIHandler.this.ui.getVesselUseFeatureList();
                        break;

                    default:
                        selectedDoubleList = null;
                }
                if (selectedDoubleList != null) {
                    selectedDoubleList.getHandler().refreshFilteredElements();
                }
            }
        });

        dialog = new SelectSpeciesUI(this.ui);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getNameField();
    }

    public List<EditProtocolSpeciesRowModel> toRows(List<SpeciesProtocol> speciesProtocols) {
        BeanFilterableComboBox<Species> speciesComboBox = ui.getSpeciesComboBox();
        Preconditions.checkNotNull(speciesComboBox.getData());
        BeanFilterableComboBox<Species> benthosComboBox = ui.getBenthosComboBox();
        Preconditions.checkNotNull(benthosComboBox.getData());

        EditProtocolUIModel model = getModel();

        Map<String, Species> allReferentSpeciesByTaxonId = model.getAllReferentSpeciesByTaxonId();
        Map<String, Caracteristic> allCaracteristic = model.getAllCaracteristic();

        List<Caracteristic> lengthClassesPmfmId = Lists.newArrayList();

        Set<Species> speciesSet = Sets.newHashSet();
        List<EditProtocolSpeciesRowModel> result = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(speciesProtocols)) {
            for (SpeciesProtocol speciesProtocol : speciesProtocols) {
                Integer taxonId = speciesProtocol.getSpeciesReferenceTaxonId();

                if (taxonId == null) {

                    continue;
                }
                String taxonIdStr = String.valueOf(taxonId);

                // remove all synonyms from available synonym list
                Collection<Species> allSynonyms = model.getAllSynonyms(taxonIdStr);
                model.getAllSynonyms().removeAll(allSynonyms);

                // get species referent taxon
                Species species = allReferentSpeciesByTaxonId.get(taxonIdStr);

                // make sure it exists
                Preconditions.checkNotNull(species, "Espèce inconnue : " + taxonIdStr);
                speciesSet.add(species);

                EditProtocolSpeciesRowModel row = EditProtocolSpeciesTableModel.newRow(sampleCategoryModel);
                row.setSpecies(species);
                String lengthStepPmfmId = speciesProtocol.getLengthStepPmfmId();

                Caracteristic lengthStepPmfm = allCaracteristic.get(lengthStepPmfmId);
                if (lengthStepPmfmId != null &&
                    !lengthClassesPmfmId.contains(lengthStepPmfm) &&
                    !model.containsLengthClassesPmfmId(lengthStepPmfmId)) {
                    if (log.isInfoEnabled()) {
                        log.info("Found a new lengthStep pmfm: " + lengthStepPmfmId);
                    }
                    lengthClassesPmfmId.add(lengthStepPmfm);
                }
                row.setLengthStepPmfm(lengthStepPmfm);
                row.fromBean(speciesProtocol);

                // make sure to get a clean copy of the list
                row.setMandatorySampleCategoryId(Lists.newArrayList(speciesProtocol.getMandatorySampleCategoryId()));
                result.add(row);
            }

            // remove once for all in comboboxes
            speciesComboBox.removeItems(speciesSet);
            benthosComboBox.removeItems(speciesSet);
        }

        if (CollectionUtils.isNotEmpty(lengthClassesPmfmId)) {
            // detect some new length step pmfp to add in protocol

            // add it to model
            model.addAllLengthClassesPmfmId(TuttiEntities.collecIds(lengthClassesPmfmId));

            // add it to ui (no binding here, must do it manually)
            BeanDoubleList<Caracteristic> lengthClassesList = ui.getLengthClassesList();
            lengthClassesList.getModel().addToSelected(lengthClassesPmfmId);
        }
        return result;
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
        clearValidators();
        ui.getTabPanel().setSelectedIndex(0);
    }

    @Override
    public boolean quitUI() {
        boolean result = quitScreen(
                getModel().isValid(),
                getModel().isModify(),
                t("tutti.editProtocol.askCancelEditBeforeLeaving.cancelSaveProtocol"),
                t("tutti.editProtocol.askSaveBeforeLeaving.saveProtocol"),
                ui.getSaveButton().getAction()
        );
        return result;
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void addDoubleListListeners() {
        EditProtocolUIModel model = getModel();

        for (BeanDoubleList<Caracteristic> list : allDoubleLists) {
            String id = (String) list.getClientProperty("_updateListenerId");
            UpdateSelectedList updateListener = (UpdateSelectedList) list.getClientProperty("_updateListener");
            model.addPropertyChangeListener(id, updateListener);
        }
    }

    public void removeDoubleListListeners() {
        EditProtocolUIModel model = getModel();

        for (BeanDoubleList<Caracteristic> list : allDoubleLists) {
            String id = (String) list.getClientProperty("_updateListenerId");
            UpdateSelectedList updateListener = (UpdateSelectedList) list.getClientProperty("_updateListener");
            model.removePropertyChangeListener(id, updateListener);
        }
    }

    public Species openSelectOtherSpeciesDialog(String title, List<Species> species) {
        SelectSpeciesUIModel model = dialog.getModel();
        model.setSpecies(species);
        model.setSelectedSpecies(null);

        openDialog(dialog, title, new Dimension(400, 130));

        return model.getSelectedSpecies();
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void initDoubleList(String propertyId,
                                  BeanDoubleList<Caracteristic> widget,
                                  List<Caracteristic> availableCaracteristics,
                                  List<String> selectedCaracteristics) {

        initBeanList(widget, availableCaracteristics,
                     Lists.<Caracteristic>newArrayList());

        UpdateSelectedList listener = new UpdateSelectedList(
                widget,
                getModel().getAllCaracteristic());
        widget.putClientProperty("_updateListener", listener);
        widget.putClientProperty("_updateListenerId", propertyId);
        listener.select(selectedCaracteristics);

        // add a filter to keep only in universe everything except what is selected by other double lists
        List<BeanDoubleList<Caracteristic>> list = Lists.newArrayList(allDoubleLists);
        list.remove(widget);
        widget.getHandler().addFilter(new SelectValuePredicate(list));
    }

    protected void selectLengthClasses(List<String> ids, JComboBox comboBox) {

        Map<String, Caracteristic> allCaracteristic = getModel().getAllCaracteristic();
        List<Caracteristic> selection = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(ids)) {
            for (String id : ids) {
                selection.add(allCaracteristic.get(id));
            }
        }

        List<Caracteristic> dataToList = Lists.newArrayList(selection);

        // add a null value at first position
        if (!dataToList.isEmpty() && dataToList.get(0) != null) {
            dataToList.add(0, null);
        }
        SwingUtil.fillComboBox(comboBox, dataToList, null);
    }

    protected void addLengthClassesColumnToModel(TableColumnModel model,
                                                 List<String> selectedIds) {

        Decorator<Caracteristic> decorator =
                getDecorator(Caracteristic.class, null);

        final JComboBox comboBox = new JComboBox();

        getModel().addPropertyChangeListener(EditProtocolUIModel.PROPERTY_LENGTH_CLASSES_PMFM_ID, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                List<String> ids = (List<String>) evt.getNewValue();
                selectLengthClasses(ids, comboBox);
            }
        });
        comboBox.setRenderer(newListCellRender(decorator));

        selectLengthClasses(selectedIds, comboBox);
        ObjectToStringConverter converter = BeanUIUtil.newDecoratedObjectToStringConverter(decorator);
        BeanUIUtil.decorate(comboBox, converter);
        ComboBoxCellEditor editor = new ComboBoxCellEditor(comboBox);

        addColumnToModel(model,
                         editor,
                         newTableCellRender(decorator),
                         EditProtocolSpeciesTableModel.LENGTH_STEP_PMFM_ID);
    }

    protected void initTable(final JXTable table,
                             DefaultTableColumnModelExt columnModel,
                             TableColumnExt speciesColumn,
                             List<EditProtocolSpeciesRowModel> rows,
                             ListSelectionListener selectionListener) {

        final EditProtocolSpeciesTableModel tableModel =
                new EditProtocolSpeciesTableModel(sampleCategoryModel, columnModel);
        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        JTableHeader tableHeader = table.getTableHeader();

        // by default do not authorize to change column orders
        tableHeader.setReorderingAllowed(false);

        addHighlighters(table);

        // always scroll to selected row
        SwingUtil.scrollToTableSelection(table);

        // add selection listener
        table.getSelectionModel().addListSelectionListener(selectionListener);

        // when model change, then rebuild the species comparator + set model as modified
        tableModel.addTableModelListener(new TableModelListener() {
            @Override
            public void tableChanged(TableModelEvent e) {
                getModel().setModify(true);

                int type = e.getType();
                if (type == TableModelEvent.DELETE ||
                    type == TableModelEvent.INSERT ||
                    e.getLastRow() == Integer.MAX_VALUE) {

                    // get species column
                    TableColumnExt tableColumn =
                            (TableColumnExt) table.getColumns().get(0);

                    // get column comparator
                    TuttiDecorator.TuttiDecoratorComparator<Species> comparator =
                            (TuttiDecorator.TuttiDecoratorComparator<Species>)
                                    tableColumn.getComparator();

                    // get column comparator
                    TuttiDecorator<Species> decorator =
                            SpeciesBatchRowHelper.getSpeciesColumnDecorator(tableColumn);

                    boolean comparatorNull = comparator == null;
                    if (comparatorNull) {

                        // first time coming here, add the comparator
                        comparator = decorator.getCurrentComparator();
                    }

                    // init comparator with model species list
                    comparator.init(decorator, tableModel.getSpeciesList());

                    if (comparatorNull) {

                        // affect it to colum
                        tableColumn.setComparator(comparator);
                    }
                }
            }
        });

        // create popup to change species decorator

        SpeciesBatchRowHelper.installSpeciesColumnComparatorPopup(
                table,
                speciesColumn,
                null,
                t("tutti.species.refTaxCode.tip"),
                t("tutti.species.name.tip")
        );

        // at the very end, set rows to model
        tableModel.setRows(rows);
    }

    protected static class UpdateSelectedList implements PropertyChangeListener {

        private final BeanDoubleListModel<Caracteristic> model;

        private final Map<String, Caracteristic> caracteristicMap;

        public UpdateSelectedList(BeanDoubleList<Caracteristic> doubleList,
                                  Map<String, Caracteristic> caracteristicMap) {
            this.model = doubleList.getModel();
            this.caracteristicMap = caracteristicMap;
        }

        private boolean valueIsAdjusting;

        @Override
        public void propertyChange(PropertyChangeEvent evt) {

            if (!valueIsAdjusting) {

                valueIsAdjusting = true;

                try {
                    List<String> selectedIds = (List<String>) evt.getNewValue();
                    if (log.isInfoEnabled()) {
                        log.info("[" + evt.getPropertyName() + "] selected ids: " + selectedIds);
                    }

                    select(selectedIds);
                } finally {
                    valueIsAdjusting = false;
                }
            }
        }

        public void select(List<String> selectedIds) {

            List<Caracteristic> selection = Lists.newArrayList();
            if (CollectionUtils.isNotEmpty(selectedIds)) {
                for (String selectedId : selectedIds) {
                    Caracteristic e = caracteristicMap.get(selectedId);
                    Preconditions.checkNotNull(e, "Could not find caracteristic with id: " + selectedId);
                    selection.add(e);
                }
            }
            model.setSelected(selection);
        }
    }

    protected static class SelectValuePredicate implements Predicate<Caracteristic> {

        protected final List<BeanDoubleList<Caracteristic>> lists;

        public SelectValuePredicate(List<BeanDoubleList<Caracteristic>> lists) {
            this.lists = lists;
        }

        @Override
        public boolean apply(Caracteristic input) {

            boolean result = true;
            for (BeanDoubleList<Caracteristic> list : lists) {
                if (list.getModel().getSelected().contains(input)) {
                    result = false;
                    break;
                }
            }
            return result;
        }
    }

//    public static class SpeciesDecoratorListener implements ActionListener {
//
//        protected final JXTable table;
//
//        protected final ButtonGroup buttonGroup;
//
//        protected final TuttiDecorator<Species> decorator;
//
//        protected final TableColumnExt column;
//
//        public SpeciesDecoratorListener(JXTable table,
//                                        ButtonGroup buttonGroup) {
//            this.table = table;
//            this.buttonGroup = buttonGroup;
//            this.column = (TableColumnExt) table.getColumn(0);
//            this.decorator = TuttiUIUtil.getSpeciesColumnDecorator(column);
//        }
//
//        @Override
//        public void actionPerformed(ActionEvent e) {
//
//            JRadioButtonMenuItem source = (JRadioButtonMenuItem) e.getSource();
//            buttonGroup.setSelected(source.getModel(), true);
//
//            Integer index =
//                    (Integer) source.getClientProperty(SPECIES_DECORATOR_INDEX);
//
//            if (log.isInfoEnabled()) {
//                log.info("Selected decorator context index: " + index);
//            }
//
//            decorator.setContextIndex(index);
//
//            column.setComparator(decorator.getCurrentComparator());
//
//
//            EditProtocolSpeciesTableModel tableModel =
//                    (EditProtocolSpeciesTableModel) table.getModel();
//
//            // keep selected rows
//            int[] rowIndexes = table.getSelectedRows();
//            List<EditProtocolSpeciesRowModel> rowsToReSelect = Lists.newArrayList();
//            for (int viewRowIndex : rowIndexes) {
//                int modelRowIndex = table.convertRowIndexToModel(viewRowIndex);
//                EditProtocolSpeciesRowModel row = tableModel.getEntry(modelRowIndex);
//                rowsToReSelect.add(row);
//            }
//
//            // fire model (will reload the comparator)
//            tableModel.fireTableDataChanged();
//
//            // reselect rows
//            for (EditProtocolSpeciesRowModel row : rowsToReSelect) {
//                int modelRowIndex = tableModel.getRowIndex(row);
//                int viewRowIndex = table.convertRowIndexToView(modelRowIndex);
//                table.addRowSelectionInterval(viewRowIndex, viewRowIndex);
//            }
//        }
//    }
//
//    private static class ShowSpeciesDecoratorPopupListener extends MouseAdapter {
//        private final JPopupMenu popup;
//
//        public ShowSpeciesDecoratorPopupListener(JPopupMenu popup) {
//            this.popup = popup;
//        }
//
//        @Override
//        public void mouseClicked(MouseEvent e) {
//            JTableHeader source = (JTableHeader) e.getSource();
//            Point point = e.getPoint();
//            int columnIndex = source.columnAtPoint(point);
//
//            boolean rightClick = SwingUtilities.isRightMouseButton(e);
//            if (columnIndex == 0 && rightClick) {
//                e.consume();
//                popup.show(source, e.getX(), e.getY());
//            }
//        }
//    }
}
