package fr.ifremer.tutti.ui.swing.content.db;

/*
 * #%L
 * Tutti :: UI
 * $Id: DbManagerUIHandler.java 1223 2013-09-25 13:31:19Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.7/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/db/DbManagerUIHandler.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Maps;
import fr.ifremer.tutti.ui.swing.TuttiDbUpdaterCallBack;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.config.TuttiApplicationConfig;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiUIHandler;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.updater.ApplicationInfo;
import org.nuiton.updater.ApplicationUpdater;
import org.nuiton.util.Version;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Map;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 1.0
 */
public class DbManagerUIHandler extends AbstractTuttiUIHandler<TuttiUIContext, DbManagerUI> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(DbManagerUIHandler.class);

    public DbManagerUIHandler(TuttiUIContext context, DbManagerUI ui) {
        super(context, ui);
    }

    @Override
    public void beforeInitUI() {
    }

    public void updateMessage() {

        boolean dbExist = getContext().isDbExist();
        boolean dbLoaded = getContext().isDbLoaded();


        if (log.isInfoEnabled()) {
            log.info("Rebuild information text... (dbExist?" + dbExist +
                     "/ dbLoaded?" + dbLoaded + ")");
        }

        String message;

        if (dbExist) {

            TuttiApplicationConfig config = getConfig();

            String jdbcUrl = config.getServiceConfig().getPersistenceConfig().getJdbcUrl();

            Map<String, String> caracteristics = Maps.newLinkedHashMap();

            String title;

            caracteristics.put(_("tutti.dbManager.caracteristic.url"), jdbcUrl);

            if (dbLoaded) {

                Version dbVersion = getContext().getPersistenceService().getDbVersion();

                // db loaded
                title = n_("tutti.dbManager.info.db.loaded");

                caracteristics.put(_("tutti.dbManager.caracteristic.schemaVersion"), dbVersion.toString());

            } else {

                // no db loaded
                title = n_("tutti.dbManager.info.no.db.loaded");
            }

            // get referential version

            if (getContext().checkUpdateDbReachable()) {
                String urlDb = config.getUpdateDbUrl();

                ApplicationUpdater up = new ApplicationUpdater();
                Map<String, ApplicationInfo> dbVersions = up.getVersions(urlDb, config.getDataDirectory());

                ApplicationInfo updateDbVersion =
                        dbVersions.get(TuttiDbUpdaterCallBack.DB_UPDATE_NAME);

                String currentReferentialVersion = updateDbVersion.oldVersion;
                String newReferentialVersion = updateDbVersion.newVersion;
                caracteristics.put(_("tutti.dbManager.caracteristic.referentialVersion"), currentReferentialVersion);
                if (newReferentialVersion != null) {
                    caracteristics.put(_("tutti.dbManager.caracteristic.lastReferentialVersion"), newReferentialVersion);
                }
            }

            StringBuilder caracteristicsToString = new StringBuilder("<ul>");
            for (Map.Entry<String, String> entry : caracteristics.entrySet()) {
                caracteristicsToString.append("<li>");
                caracteristicsToString.append(entry.getKey());
                caracteristicsToString.append(" : <strong>");
                caracteristicsToString.append(entry.getValue());
                caracteristicsToString.append("</strong></li>");
            }
            caracteristicsToString.append("</ul>");

            message = _(title, caracteristicsToString);

        } else {

            // db does not exist
            message = _("tutti.dbManager.info.no.db.exist");
        }
        String result = "<html><body>" + message + "</body></html>";
        getUI().getInformationArea().setText(result);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        getModel().addPropertyChangeListener(TuttiUIContext.PROPERTY_DB_EXIST, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                Boolean dbExist = (Boolean) evt.getNewValue();
                String mnemonic;
                if (dbExist) {
                    mnemonic = _("tutti.dbManager.action.reinstallDb.mnemonic");
                } else {
                    mnemonic = _("tutti.dbManager.action.installDb.mnemonic");
                }
                if (StringUtils.isNotBlank(mnemonic)) {
                    ui.getInstallOrReinstallDbButton().setMnemonic(mnemonic.charAt(0));
                }

            }
        });

        updateMessage();

        SwingUtilities.invokeLater(
                new Runnable() {
                    @Override
                    public void run() {
                        getContext().getMainUI().getBody().repaint();
                    }
                }
        );
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getUpgradeDbButton();
    }

    @Override
    public void onCloseUI() {
    }

    @Override
    public SwingValidator<TuttiUIContext> getValidator() {
        return null;
    }

    public String getInstallButtonText(boolean dbExist) {
        String result;
        if (dbExist) {
            result = _("tutti.dbManager.action.reinstallDb");
        } else {
            result = _("tutti.dbManager.action.installDb");
        }
        return result;
    }

    public String getInstallButtonTip(boolean dbExist) {
        String result;
        if (dbExist) {
            result = _("tutti.dbManager.action.reinstallDb.tip");
        } else {
            result = _("tutti.dbManager.action.installDb.tip");
        }
        return result;
    }
}
