package fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.IndividualObservationBatch;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.service.ValidationService;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.create.CreateIndividualObservationBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.create.CreateIndividualObservationBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.service.WeightUnit;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.caracteristics.CaracteristicMapCellComponent;
import fr.ifremer.tutti.ui.swing.util.caracteristics.CaracteristicMapColumnUIHandler;
import fr.ifremer.tutti.ui.swing.util.caracteristics.CaracteristicMapEditorUI;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.CaracteristicColumnIdentifier;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ComponentAdapter;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.decorator.Decorator;
import org.nuiton.validator.NuitonValidatorResult;

import javax.swing.JComponent;
import java.awt.Color;
import java.awt.Component;
import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class IndividualObservationBatchUIHandler
        extends AbstractTuttiBatchTableUIHandler<IndividualObservationBatchRowModel, IndividualObservationBatchUIModel, IndividualObservationBatchUI>
        implements CaracteristicMapColumnUIHandler {

    private static final Log log =
            LogFactory.getLog(IndividualObservationBatchUIHandler.class);

    protected ValidationService validationService = getContext().getValidationService();

    /**
     * Weight unit.
     *
     * @since 2.5
     */
    protected final WeightUnit weightUnit;

    public IndividualObservationBatchUIHandler(TuttiUI<?, ?> parentUi, IndividualObservationBatchUI ui) {
        super(parentUi, ui,
              IndividualObservationBatchRowModel.PROPERTY_SPECIES,
              IndividualObservationBatchRowModel.PROPERTY_WEIGHT,
              IndividualObservationBatchRowModel.PROPERTY_SIZE,
              IndividualObservationBatchRowModel.PROPERTY_LENGTH_STEP_CARACTERISTIC,
              IndividualObservationBatchRowModel.PROPERTY_CARACTERISTICS,
              IndividualObservationBatchRowModel.PROPERTY_DEFAULT_CARACTERISTICS,
//              IndividualObservationBatchRowModel.PROPERTY_CALCIFIED_PIECE_SAMPLING_CODE,
//              IndividualObservationBatchRowModel.PROPERTY_SAMPLING_CODE,
              IndividualObservationBatchRowModel.PROPERTY_COMMENT,
              IndividualObservationBatchRowModel.PROPERTY_ATTACHMENT);

        weightUnit = getConfig().getIndividualObservationWeightUnit();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        IndividualObservationBatchUIModel model = getModel();

        List<IndividualObservationBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get individualObservation batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<IndividualObservationBatch> batches =
                        persistenceService.getAllIndividualObservationBatch(bean.getId());

                for (IndividualObservationBatch aBatch : batches) {

                    // set the surveycode
                    if (getDataContext().isProtocolFilled()) {
                        // get the surveycode from the species list of the model
                        List<Species> speciesList = getDataContext().getReferentSpeciesWithSurveyCode();
                        int i = speciesList.indexOf(aBatch.getSpecies());
                        if (i > -1) {
                            aBatch.setSpecies(speciesList.get(i));
                        }
                    }

                    IndividualObservationBatchRowModel entry = loadBatch(aBatch);
                    rows.add(entry);
                }
            }
        }
        model.setRows(rows);
        recomputeBatchActionEnable();
    }

    protected IndividualObservationBatchRowModel loadBatch(IndividualObservationBatch aBatch) {

        IndividualObservationBatchRowModel newRow =
                new IndividualObservationBatchRowModel(
                        weightUnit,
                        aBatch,
                        getModel().getDefaultCaracteristic());

        List<Attachment> attachments =
                persistenceService.getAllAttachments(newRow.getObjectType(),
                                                     newRow.getObjectId());

        newRow.addAllAttachment(attachments);

        return newRow;
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public IndividualObservationBatchTableModel getTableModel() {
        return (IndividualObservationBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(IndividualObservationBatchRowModel row) {
        IndividualObservationBatch batch = row.toBean();
        NuitonValidatorResult validator = validationService.validateIndividualObservationBatch(batch);
        boolean result = !validator.hasErrorMessagess();
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 IndividualObservationBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);

        saveSelectedRowIfNeeded();

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<IndividualObservationBatchRowModel> rowMonitor,
                                             IndividualObservationBatchRowModel row) {

        if (row != null && row.isValid() && rowMonitor.wasModified()) {

            // monitored bean was modified, save it
            if (log.isInfoEnabled()) {
                log.info("Row " + row + " was modified, will save it");
            }

            rowMonitor.setBean(null);
            saveRow(row);
            rowMonitor.setBean(row);

            // clear modified flag on the monitor
            rowMonitor.clearModified();
        }
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          IndividualObservationBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             IndividualObservationBatchRowModel oldRow,
                                             int newRowIndex,
                                             IndividualObservationBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);

        // when selected row has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void addHighlighters(JXTable table) {
        super.addHighlighters(table);

        // paint in a special color for comment cell (with not null value)
        Color cellWithValueColor = getConfig().getColorCellWithValue();

        Highlighter commentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(IndividualObservationBatchTableModel.COMMENT),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                String value = (String) adapter.getValue();
                                return StringUtils.isNotBlank(value);
                            }
                        }), cellWithValueColor);
        table.addHighlighter(commentHighlighter);

        // paint in a special color for attachment cell (when some attachments)

        Highlighter attachmentHighlighter = TuttiUIUtil.newBackgroundColorHighlighter(
                new HighlightPredicate.AndHighlightPredicate(
                        new HighlightPredicate.IdentifierHighlightPredicate(IndividualObservationBatchTableModel.ATTACHMENT),
                        // for not null value
                        new HighlightPredicate() {
                            @Override
                            public boolean isHighlighted(Component renderer, ComponentAdapter adapter) {
                                Collection attachments = (Collection) adapter.getValue();
                                return CollectionUtils.isNotEmpty(attachments);
                            }
                        }
                ), cellWithValueColor);
        table.addHighlighter(attachmentHighlighter);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<IndividualObservationBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("beforeInit: " + ui);
        }

        // get the default caracteristics
        List<Caracteristic> defaultCaracteristic =
                getDataContext().getDefaultIndividualObservationCaracteristics();

        // create model
        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);
        IndividualObservationBatchUIModel model =
                new IndividualObservationBatchUIModel(catchesUIModel,
                                                      defaultCaracteristic);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isDebugEnabled()) {
            log.debug("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // can show / hide some columns in model
        table.setColumnControlVisible(true);

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();
        Decorator<Caracteristic> caracteristicDecorator =
                getDecorator(Caracteristic.class, DecoratorService.CARACTERISTIC_PARAMETER_ONLY_WITH_UNIT);
        Decorator<Caracteristic> caracteristicTipDecorator =
                getDecorator(Caracteristic.class, DecoratorService.CARACTERISTIC_WITH_UNIT);

        Decorator<CaracteristicQualitativeValue> caracteristicQualitativeDecorator =
                getDecorator(CaracteristicQualitativeValue.class, null);

        { // Species column

            Decorator<Species> speciesDecorator = getDecorator(
                    Species.class, DecoratorService.FROM_PROTOCOL);
            addComboDataColumnToModel(columnModel,
                                      IndividualObservationBatchTableModel.SPECIES,
                                      speciesDecorator, getDataContext().getReferentSpeciesWithSurveyCode());
        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  IndividualObservationBatchTableModel.WEIGHT,
                                  weightUnit);
        }

        { // Size column

            addFloatColumnToModel(columnModel,
                                  IndividualObservationBatchTableModel.SIZE,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Length step caracteristic column

            addComboDataColumnToModel(columnModel,
                                      IndividualObservationBatchTableModel.LENGTH_STEP_CARACTERISTIC,
                                      getDecorator(Caracteristic.class, null),
                                      getDataContext().getLengthStepCaracteristics());

        }

        List<Caracteristic> defaultCaracteristic =
                getModel().getDefaultCaracteristic();


        for (Caracteristic caracteristic : defaultCaracteristic) {
            String header = caracteristicDecorator.toString(caracteristic);
            String headerTip = caracteristicTipDecorator.toString(caracteristic);

            CaracteristicColumnIdentifier id = CaracteristicColumnIdentifier.newCaracteristicId(
                    caracteristic,
                    IndividualObservationBatchRowModel.PROPERTY_DEFAULT_CARACTERISTICS,
                    header,
                    headerTip
            );

            switch (caracteristic.getCaracteristicType()) {

                case NUMBER:

                    addFloatColumnToModel(columnModel,
                                          id,
                                          TuttiUI.DECIMAL3_PATTERN);

                    break;
                case QUALITATIVE:
                    List<CaracteristicQualitativeValue> values =
                            caracteristic.getQualitativeValue();
                    addComboDataColumnToModel(
                            columnModel,
                            id,
                            caracteristicQualitativeDecorator,
                            values);
                    break;
                case TEXT:

                    addColumnToModel(columnModel, id);

                    break;
            }
        }

        { // Other caracteristics column

            Set<Caracteristic> caracteristicsToSkip = Collections.unmodifiableSet(
                    Sets.newHashSet(getModel().getDefaultCaracteristic()));

            addColumnToModel(columnModel,
                             CaracteristicMapCellComponent.newEditor(ui, caracteristicsToSkip),
                             CaracteristicMapCellComponent.newRender(getContext()),
                             IndividualObservationBatchTableModel.OTHER_CARACTERISTICS);

        }

//        { // Calcified piece sampling code column
//
//            addColumnToModel(columnModel,
//                             IndividualObservationBatchTableModel.CALCIFIED_PIECE_SAMPLING_CODE);
//        }
//
//        { // Sampling code column
//
//            addColumnToModel(columnModel,
//                             IndividualObservationBatchTableModel.SAMPLING_CODE);
//        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             IndividualObservationBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             IndividualObservationBatchTableModel.ATTACHMENT);
        }

        // create table model
        IndividualObservationBatchTableModel tableModel =
                new IndividualObservationBatchTableModel(weightUnit,
                                                         columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
        recomputeBatchActionEnable();
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getTable();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
    }

    @Override
    public CaracteristicMapEditorUI getCaracteristicMapEditor() {
        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        return parent.getIndividualObservationCaracteristicMapEditor();
    }

    @Override
    public void showCaracteristicMapEditor(Species species) {
        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.EDIT_CARACTERISTICS_CARD, species);
    }

    @Override
    public void hideCaracteristicMapEditor() {
        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createBatch() {

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        CreateIndividualObservationBatchUI createBatchEditor = parent.getIndividualObservationTabCreateBatch();

        createBatchEditor.getHandler().openUI(getModel());
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.CREATE_BATCH_CARD);
    }

    public void addBatch(CreateIndividualObservationBatchUIModel model) {
        if (model.isValid()) {

            IndividualObservationBatchTableModel tableModel = getTableModel();

            IndividualObservationBatchRowModel newRow = tableModel.createNewRow();
            newRow.setSpecies(model.getSpecies());
            newRow.setWeight(model.getWeight());
            newRow.setSize(model.getSize());
            newRow.setLengthStepCaracteristic(model.getLengthStepCaracteristic());
            newRow.getDefaultCaracteristics().putAll(model.getCaracteristics());

            recomputeRowValidState(newRow);

            saveRow(newRow);

            tableModel.addNewRow(newRow);
            AbstractSelectTableAction.doSelectCell(getTable(),
                                                   tableModel.getRowCount() - 1,
                                                   0);
        }

        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void recomputeBatchActionEnable() {

        int rowIndex = getTable().getSelectedRow();

        boolean enableRemove = false;

        if (rowIndex != -1) {

            // there is a selected row
            enableRemove = true;
        }
        IndividualObservationBatchUIModel model = getModel();
        model.setRemoveBatchEnabled(enableRemove);
    }

    protected void saveRow(IndividualObservationBatchRowModel row) {

        IndividualObservationBatch entityToSave = row.toEntity();

        CaracteristicMap caracteristics = new CaracteristicMap();
        entityToSave.setCaracteristics(caracteristics);

        // push back not null extra caracteristics
        for (Map.Entry<Caracteristic, Serializable> entry : row.getCaracteristics().entrySet()) {
            Serializable value = entry.getValue();
            if (value != null) {
                caracteristics.put(entry.getKey(), value);
            }
        }

        // push back not null default caracteristics
        for (Map.Entry<Caracteristic, Serializable> entry : row.getDefaultCaracteristics().entrySet()) {
            Serializable value = entry.getValue();
            if (value != null) {
                caracteristics.put(entry.getKey(), value);
            }
        }

        FishingOperation fishingOperation = getModel().getFishingOperation();
        entityToSave.setFishingOperation(fishingOperation);

        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(entityToSave)) {

            entityToSave = persistenceService.createIndividualObservationBatch(entityToSave);
            row.setId(entityToSave.getId());
        } else {
            persistenceService.saveIndividualObservationBatch(entityToSave);
        }

        getModel().fireBatchUpdated(row);
    }
}
