package fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesFrequencyUIHandler.java 855 2013-04-22 17:03:12Z kmorin $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-2.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/frequency/SpeciesFrequencyUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.SpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyCellComponent.FrequencyCellEditor;
import fr.ifremer.tutti.ui.swing.util.Cancelable;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.JComponent;
import javax.swing.JTextField;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesFrequencyUIHandler extends AbstractTuttiTableUIHandler<SpeciesFrequencyRowModel, SpeciesFrequencyUIModel, SpeciesFrequencyUI> implements Cancelable {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesFrequencyUIHandler.class);

    private FrequencyCellEditor frequencyEditor;

    private TuttiProtocol protocol;

    private Map<Integer, SpeciesProtocol> speciesProtocol;

    private Map<String, Caracteristic> lengthStepCaracteristics;

    protected Set<SpeciesFrequencyRowModel> withWeightRows = Sets.newHashSet();

    public SpeciesFrequencyUIHandler(TuttiUIContext context,
                                     SpeciesFrequencyUI ui) {
        super(context, ui,
              SpeciesFrequencyRowModel.PROPERTY_LENGTH_STEP,
              SpeciesFrequencyRowModel.PROPERTY_NUMBER,
              SpeciesFrequencyRowModel.PROPERTY_WEIGHT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public SpeciesFrequencyTableModel getTableModel() {
        return (SpeciesFrequencyTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SpeciesFrequencyRowModel row) {
        return row.getLengthStepCaracteristic() != null &&
               row.getLengthStep() != null &&
               row.getNumber() != null && row.getNumber() > 0 &&
               (withWeightRows.isEmpty() || row.getWeight() != null && row.getWeight() > 0);
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SpeciesFrequencyRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        boolean recomputeAllRows;

        if (row.getWeight() != null) {
            // check if no row had a weight, then if one of them now has a weight,
            // the other ones must have one too to be valid
            recomputeAllRows = withWeightRows.isEmpty();
            withWeightRows.add(row);

        } else {
            withWeightRows.remove(row);
            // check if no row has a weight, then if none of them now has a weight,
            // the other ones do not need to have a weight to be valid
            recomputeAllRows = withWeightRows.isEmpty();
        }
        if (recomputeAllRows) {
            List<SpeciesFrequencyRowModel> rows = getModel().getRows();
            for (SpeciesFrequencyRowModel r : rows) {
                recomputeRowValidState(r);
            }
        } else {
            recomputeRowValidState(row);
        }
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SpeciesFrequencyRowModel> rowMonitor,
                                             SpeciesFrequencyRowModel row) {
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          SpeciesFrequencyRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);
        ui.getValidator().doValidate();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<SpeciesFrequencyUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        SpeciesFrequencyUIModel model = new SpeciesFrequencyUIModel();

        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        initUI(ui);

        List<Caracteristic> lengthStepCaracterics =
                Lists.newArrayList(getDataContext().getLengthStepCaracteristics());

        lengthStepCaracteristics = TuttiEntities.splitById(lengthStepCaracterics);

        SpeciesFrequencyUIModel model = getModel();

        if (context.isProtocolFilled()) {

            // get loaded protocol
            protocol = getDataContext().getProtocol();
            Preconditions.checkNotNull(protocol,
                                       "Could not find protocol in ui context");

            // FIXME 20130128 kmorin: the species have no technical id
            speciesProtocol = Maps.newHashMap();
            for (SpeciesProtocol sp : protocol.getSpecies()) {
                speciesProtocol.put(sp.getSpeciesReferenceTaxonId(), sp);
            }

        }

        Caracteristic modelCaracteristic = model.getLengthStepCaracteristic();
        initBeanFilterableComboBox(ui.getLengthStepCaracteristicComboBox(),
                                   lengthStepCaracterics,
                                   modelCaracteristic);

        // get precision from the pmfm
        Float precision = null;
        if (modelCaracteristic != null) {
            precision = model.getLengthStepCaracteristic().getPrecision();
        }
        if (precision == null) {
            precision = 1f;
        }
        model.setStep(precision);

        model.setMinStep(null);
        model.setMaxStep(null);

        ui.getRafaleStepField().getTextField().addKeyListener(new KeyAdapter() {

            @Override
            public void keyReleased(KeyEvent e) {
                if (e.getKeyCode() == KeyEvent.VK_ENTER) {
                    e.consume();
                    Float step = (Float) ui.getRafaleStepField().getModel();

                    applyRafaleStep(step);

                    //select text
                    JTextField field = (JTextField) e.getSource();
                    field.selectAll();
                }
            }
        });

        // when lengthStepCaracteristic changed, let's updates all row with the new value
        model.addPropertyChangeListener(SpeciesFrequencyUIModel.PROPERTY_LENGHT_STEP_CARACTERISTIC, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                Caracteristic newValue = (Caracteristic) evt.getNewValue();
                for (SpeciesFrequencyRowModel rowModel : getModel().getRows()) {
                    rowModel.setLengthStepCaracteristic(newValue);
                    // get precision from the pmfm
                    Float precision = null;
                    if (newValue != null) {
                        precision = newValue.getPrecision();
                    }
                    if (precision == null) {
                        precision = .5f;
                    }
                    getModel().setStep(precision);
                    recomputeRowValidState(rowModel);
                }
                ui.getValidator().doValidate();
            }
        });

        // create table column model
        DefaultTableColumnModelExt columnModel = new DefaultTableColumnModelExt();

        { // LengthStep

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.LENGTH_STEP,
                                  TuttiUI.DECIMAL1_PATTERN);
        }

        { // Number

            addIntegerColumnToModel(columnModel,
                                    SpeciesFrequencyTableModel.NUMBER,
                                    TuttiUI.INT_6_DIGITS_PATTERN);
        }

        { // Weight

            addFloatColumnToModel(columnModel,
                                  SpeciesFrequencyTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        // create table model
        SpeciesFrequencyTableModel tableModel =
                new SpeciesFrequencyTableModel(columnModel, model);

        JXTable table = getTable();

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);

        installTableKeyListener(columnModel, table);

        listenValidatorValid(ui.getValidator(), model);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getLengthStepCaracteristicComboBox();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        frequencyEditor = null;

        // evict model from validator
        ui.getValidator().setBean(null);

        // when canceling always invalid model (in that way)
        getModel().setValid(false);

        getModel().setSimpleCount(null);

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getHandler().setSpeciesSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    //------------------------------------------------------------------------//
    //-- Cancelable methods                                                 --//
    //------------------------------------------------------------------------//

    @Override
    public void cancel() {

        if (log.isInfoEnabled()) {
            log.info("Cancel UI " + ui);
        }

        // close dialog
        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void generateLengthSteps() {

        SpeciesFrequencyUIModel model = getModel();
        SpeciesFrequencyTableModel tableModel = getTableModel();

        Map<Float, SpeciesFrequencyRowModel> rowsByStep =
                getTableModel().getRowCache();

        Float minStep = model.getLengthStep(model.getMinStep());
        Float maxStep = model.getLengthStep(model.getMaxStep());
        Caracteristic lengthStepCaracteristic = model.getLengthStepCaracteristic();

        for (float i = minStep, step = model.getStep(); i <= maxStep; i += step) {
            if (!rowsByStep.containsKey(i)) {

                // add it
                SpeciesFrequencyRowModel newRow = tableModel.createNewRow();
                newRow.setLengthStep(i);
                newRow.setLengthStepCaracteristic(lengthStepCaracteristic);
                rowsByStep.put(i, newRow);
            }
        }

        List<SpeciesFrequencyRowModel> rows =
                Lists.newArrayList(rowsByStep.values());

        model.setRows(rows);
    }

    public void applyRafaleStep(Float step) {

        if (log.isInfoEnabled()) {
            log.info("Will apply rafale step: " + step);
        }
        SpeciesFrequencyUIModel model = getModel();
        SpeciesFrequencyTableModel tableModel = getTableModel();

        Map<Float, SpeciesFrequencyRowModel> rowsByStep = tableModel.getRowCache();

        float aroundLengthStep = model.getLengthStep(step);

        SpeciesFrequencyRowModel row = rowsByStep.get(aroundLengthStep);

        int rowIndex;

        if (row != null) {

            // increments current row
            Integer number = row.getNumber();
            row.setNumber((number == null ? 0 : number) + 1);
            rowIndex = tableModel.updateRow(row);

        } else {

            // create a new row

            row = tableModel.createNewRow();
            row.setLengthStep(aroundLengthStep);
            row.setNumber(1);
            row.setValid(true);

            // get new index
            List<Float> steps = Lists.newArrayList(rowsByStep.keySet());
            steps.add(aroundLengthStep);

            Collections.sort(steps);

            rowIndex = steps.indexOf(aroundLengthStep);

            tableModel.addNewRow(rowIndex, row);
        }

        getTable().scrollRowToVisible(rowIndex);
    }

    public void editBatch(SpeciesBatchRowModel speciesBatch, FrequencyCellEditor editor) {
        withWeightRows.clear();

        frequencyEditor = editor;

        Caracteristic lengthStepCaracteristic = null;
        Float lengthStep = 1f;

        List<SpeciesFrequencyRowModel> editFrequency = Lists.newArrayList();

        if (speciesBatch != null) {

            List<SpeciesFrequencyRowModel> frequency =
                    speciesBatch.getFrequency();

            // try to load existing frequency

            if (CollectionUtils.isNotEmpty(frequency)) {

                SpeciesFrequencyTableModel tableModel = getTableModel();

                for (SpeciesFrequencyRowModel rowModel : frequency) {

                    SpeciesFrequencyRowModel newRow = tableModel.createNewRow();
                    newRow.setLengthStepCaracteristic(rowModel.getLengthStepCaracteristic());
                    newRow.setLengthStep(rowModel.getLengthStep());
                    newRow.setNumber(rowModel.getNumber());
                    newRow.setWeight(rowModel.getWeight());
                    editFrequency.add(newRow);

                    if (newRow.getWeight() != null) {
                        withWeightRows.add(newRow);
                    }
                }

                // use first frequency row length step caracteristics

                SpeciesFrequencyRowModel rowModel = frequency.get(0);
                lengthStepCaracteristic = rowModel.getLengthStepCaracteristic();
                lengthStep = rowModel.getLengthStep();

                if (log.isInfoEnabled()) {
                    log.info("Use existing lengthStep " +
                             "caracteristic / step " +
                             decorate(lengthStepCaracteristic) + " / " +
                             lengthStep);
                }
            }

            if (lengthStepCaracteristic == null && protocol != null) {

                Species species = speciesBatch.getSpecies();

                SpeciesProtocol sProtocol =
                        speciesProtocol.get(species.getReferenceTaxonId());

                if (sProtocol != null) {


                    String lengthStepPmfmId = sProtocol.getLengthStepPmfmId();

                    lengthStepCaracteristic =
                            this.lengthStepCaracteristics.get(lengthStepPmfmId);
                    lengthStep = sProtocol.getLengthStep();

                    if (log.isInfoEnabled()) {
                        log.info("Use existing from protocol lengthStep " +
                                 "caracteristic / step " +
                                 decorate(lengthStepCaracteristic) + " / " +
                                 lengthStep);
                    }
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Will edit batch row: " + speciesBatch + " with " +
                      editFrequency.size() + " frequency");
        }

        SpeciesFrequencyUIModel model = getModel();

        SpeciesFrequencyUIModel.ConfigurationMode mode = SpeciesFrequencyUIModel.ConfigurationMode.SIMPLE;
        if (lengthStepCaracteristic == null && protocol != null) {
            Integer taxonId = speciesBatch.getSpecies().getReferenceTaxonId();
            SpeciesProtocol speciesProtocol = this.speciesProtocol.get(taxonId);
            if (speciesProtocol == null || speciesProtocol.getLengthStepPmfmId() == null) {
                mode = SpeciesFrequencyUIModel.ConfigurationMode.SIMPLE_COUNTING;
            }
        }
        if (speciesBatch.getNumber() != null && editFrequency.isEmpty()) {
            mode = SpeciesFrequencyUIModel.ConfigurationMode.SIMPLE_COUNTING;
            model.setSimpleCount(speciesBatch.getNumber());
        }
        model.setConfigurationMode(mode);

        // connect model to validator
        ui.getValidator().setBean(model);

        model.setRows(editFrequency);
        model.setLengthStepCaracteristic(lengthStepCaracteristic);

        // keep batch (will be used to push back editing entry)
        model.setBatch(speciesBatch);
    }

    public void save() {

        if (log.isInfoEnabled()) {
            log.info("Save UI " + ui);
        }

        frequencyEditor.validateEdition(getModel());

        closeUI(ui);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

}
