package fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.caracteristics;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.CaracteristicMap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.IndividualObservationBatch;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.service.DecoratorService;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.IndividualObservationBatchRowModel;
import fr.ifremer.tutti.ui.swing.util.Cancelable;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.editor.CaracteristicValueEditor;
import fr.ifremer.tutti.ui.swing.util.editor.CaracteristicValueRenderer;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableUIHandler;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.editor.bean.BeanFilterableComboBox;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;

import javax.swing.JComponent;
import java.util.List;

import static fr.ifremer.tutti.ui.swing.content.operation.catches.individualobservation.caracteristics.CaracteristicMapCellComponent.CaracteristicMapCellEditor;

/**
 * @author kmorin <kmorin@codelutin.com>
 * @since 1.4
 */
public class CaracteristicMapEditorUIHandler
        extends AbstractTuttiTableUIHandler<CaracteristicMapEditorRowModel, CaracteristicMapEditorUIModel, CaracteristicMapEditorUI>
        implements Cancelable {

    private final static Log log =
            LogFactory.getLog(CaracteristicMapEditorUIHandler.class);

    protected CaracteristicMapCellEditor caracteristicMapCellEditor;

    public CaracteristicMapEditorUIHandler(TuttiUI parentUi,
                                           CaracteristicMapEditorUI ui) {
        super(parentUi.getHandler().getContext(), ui, CaracteristicMapEditorRowModel.PROPERTY_VALUE);
    }

    @Override
    public CaracteristicMapEditorTableModel getTableModel() {
        return (CaracteristicMapEditorTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getCaracteristicMapEditorTable();
    }

    @Override
    protected boolean isRowValid(CaracteristicMapEditorRowModel row) {
        return row.getKey() != null && row.getValue() != null;
    }

    @Override
    protected void onRowModified(int rowIndex, CaracteristicMapEditorRowModel row,
                                 String propertyName, Object oldValue, Object newValue) {
        recomputeRowValidState(row);
        super.onRowModified(rowIndex, row, propertyName, oldValue, newValue);
        saveSelectedRowIfNeeded();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<CaracteristicMapEditorRowModel> rowMonitor,
                                             CaracteristicMapEditorRowModel row) {
        if (row.isValid()) {
            // there is a valid bean attached to the monitor

            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                saveRow(row);

            }
        }
    }

    @Override
    public void beforeInitUI() {
        CaracteristicMapEditorUIModel model = createModel();
        getUI().setContextValue(model);
    }

    @Override
    public void afterInitUI() {
        initUI(getUI());

        initBeanFilterableComboBox(getKeyCombo(), Lists.<Caracteristic>newArrayList(), null);
        getModel().setAvailableCaracteristics(getDataContext().getCaracteristics());

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        {

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Caracteristic.class, DecoratorService.CARACTERISTIC_WITH_UNIT),
                             CaracteristicMapEditorTableModel.KEY);
        }

        {

            addColumnToModel(columnModel,
                             new CaracteristicValueEditor(context),
                             new CaracteristicValueRenderer(context),
                             CaracteristicMapEditorTableModel.VALUE);
        }

        // create table model
        CaracteristicMapEditorTableModel tableModel =
                new CaracteristicMapEditorTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initTable(table);
    }

    @Override
    protected JComponent getComponentToFocus() {
        return getUI().getNewRowKey();
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }

        // when canceling always invalid model (in that way)
        getModel().setValid(false);

        EditCatchesUI parent = SwingUtil.getParentContainer(ui, EditCatchesUI.class);
        parent.getHandler().setIndividualObservationSelectedCard(EditCatchesUIHandler.MAIN_CARD);
    }

    @Override
    public SwingValidator<CaracteristicMapEditorUIModel> getValidator() {
        return null;
    }

    @Override
    public void cancel() {

        if (log.isInfoEnabled()) {
            log.info("Cancel UI " + ui);
        }

        // close dialog
        closeUI(ui);
    }

    protected BeanFilterableComboBox<Caracteristic> getKeyCombo() {
        return ui.getNewRowKey();
    }

    protected CaracteristicMapEditorUIModel createModel() {
        return new CaracteristicMapEditorUIModel();
    }

    protected CaracteristicMap getCaracteristics(IndividualObservationBatch individualObservation) {
        return individualObservation.getCaracteristics();
    }

    protected List<String> getProtocolPmfmIds(TuttiProtocol protocol) {
        return protocol.getGearUseFeaturePmfmId();
    }

    protected void saveRow(CaracteristicMapEditorRowModel row) {

        if (row.isValid()) {
            CaracteristicMap caracteristics = getModel().getCaracteristicMap();
            Preconditions.checkNotNull(caracteristics);

            caracteristics.put(row.getKey(), row.getValue());
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    /** Adds a row with the parameter selected in the combo box */
    public void addRow() {
        BeanFilterableComboBox<Caracteristic> keyCombo = getKeyCombo();
        Caracteristic selectedItem = (Caracteristic) keyCombo.getSelectedItem();
        CaracteristicMapEditorTableModel tableModel = getTableModel();

        CaracteristicMapEditorRowModel row = tableModel.createNewRow();
        row.setKey(selectedItem);
//        tableModel.addNewRow(row);
        getModel().getRows().add(row);

        int rowIndex = tableModel.getRowCount() - 1;
        tableModel.fireTableRowsInserted(rowIndex, rowIndex);

        keyCombo.getHandler().removeItem(selectedItem);

        CaracteristicMapEditorUIModel model = getModel();
        model.setModify(true);
        recomputeRowValidState(row);
    }

    public void editBatch(IndividualObservationBatchRowModel individualObservation,
                          CaracteristicMapCellEditor caracteristicMapCellEditor) {

        this.caracteristicMapCellEditor = caracteristicMapCellEditor;

        CaracteristicMapEditorTableModel tableModel = getTableModel();
        CaracteristicMapEditorUIModel model = getModel();

        CaracteristicMap caracteristicMap = getCaracteristics(individualObservation);
        if (caracteristicMap == null) {
            caracteristicMap = new CaracteristicMap();
        }
        model.setCaracteristicMap(caracteristicMap);

        TuttiProtocol protocol = getDataContext().getProtocol();
        List<String> pmfmIds = Lists.newArrayList();
        if (protocol != null) {
            List<String> protocolPmfmId = getProtocolPmfmIds(protocol);
            if (protocolPmfmId != null) {
                pmfmIds.addAll(protocolPmfmId);
            }
        }

        List<CaracteristicMapEditorRowModel> rows = Lists.newArrayList();
        List<Caracteristic> caracteristics = Lists.newArrayList(caracteristicMap.keySet());

        List<Caracteristic> availableCaracteristics = model.getAvailableCaracteristics();
        for (String id : pmfmIds) {
            Caracteristic caracteristic = TuttiEntities.findById(availableCaracteristics, id);
            if (!caracteristics.contains(caracteristic)) {
                caracteristics.add(caracteristic);
            }
        }

        for (Caracteristic key : caracteristics) {
            CaracteristicMapEditorRowModel newRow = tableModel.createNewRow();
            newRow.setKey(key);
            newRow.setValue(caracteristicMap.get(key));
            rows.add(newRow);
        }

        model.setRows(rows);

        List<Caracteristic> caracteristicList = Lists.newArrayList();

        for (Caracteristic caracteristic : availableCaracteristics) {
            if (!caracteristics.contains(caracteristic)) {
                caracteristicList.add(caracteristic);
            }
        }

        BeanFilterableComboBox<Caracteristic> keyCombo = getKeyCombo();
        keyCombo.setData(caracteristicList);
//        selectFirstInCombo(keyCombo);
        keyCombo.getHandler().reset();
        model.setModify(false);
    }

    public void removeCaracteristic() {
        int rowIndex = getTable().getSelectedRow();

        Preconditions.checkState(
                rowIndex != -1,
                "Cant remove caracteristic if no caracteristic selected");

        CaracteristicMapEditorRowModel row = getTableModel().getEntry(rowIndex);

        CaracteristicMap caracteristicMap = getModel().getCaracteristicMap();
        if (caracteristicMap != null) {
            caracteristicMap.remove(row.getKey());
        }

        //add the row in the combo
        BeanFilterableComboBox keyCombo = getKeyCombo();
        keyCombo.addItem(row.getKey());
//        selectFirstInCombo(keyCombo);
        keyCombo.getHandler().reset();

        // remove the row from the model
        getModel().getRows().remove(rowIndex);

        // refresh all the table
        getTableModel().fireTableRowsDeleted(rowIndex, rowIndex);

        getModel().removeRowInError(row);
    }

    public void save() {

        if (log.isInfoEnabled()) {
            log.info("Save UI " + ui);
        }

        caracteristicMapCellEditor.validateEdition(getModel());

        closeUI(ui);
    }
}
