package fr.ifremer.tutti.ui.swing.content.operation.catches.macrowaste;

/*
 * #%L
 * Tutti :: UI
 * $Id: MacroWasteBatchUIHandler.java 602 2013-03-13 09:20:00Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/macrowaste/MacroWasteBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.MacroWasteBatch;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellEditor;
import fr.ifremer.tutti.ui.swing.util.attachment.AttachmentCellRenderer;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellEditor;
import fr.ifremer.tutti.ui.swing.util.comment.CommentCellRenderer;
import jaxx.runtime.validator.swing.SwingValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import java.util.List;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class MacroWasteBatchUIHandler extends AbstractTuttiBatchTableUIHandler<MacroWasteBatchRowModel, MacroWasteBatchUIModel, MacroWasteBatchUI> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(MacroWasteBatchUIHandler.class);

    public MacroWasteBatchUIHandler(TuttiUI<?, ?> parentUi, MacroWasteBatchUI ui) {
        super(parentUi, ui,
              MacroWasteBatchRowModel.PROPERTY_MACRO_WASTE_CATEGORY,
              MacroWasteBatchRowModel.PROPERTY_MACRO_WASTE_SIZE_CATEGORY,
              MacroWasteBatchRowModel.PROPERTY_WEIGHT,
              MacroWasteBatchRowModel.PROPERTY_NUMBER,
              MacroWasteBatchRowModel.PROPERTY_COMMENT);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        MacroWasteBatchUIModel model = getModel();

        List<MacroWasteBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get macroWaste batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {
                List<MacroWasteBatch> catches =
                        persistenceService.getAllMacroWasteBatch(bean.getId());
                for (MacroWasteBatch aBatch : catches) {
                    MacroWasteBatchRowModel entry =
                            new MacroWasteBatchRowModel(aBatch);
                    rows.add(entry);
                }
            }
        }
        model.setRows(rows);
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    public MacroWasteBatchTableModel getTableModel() {
        return (MacroWasteBatchTableModel) getTable().getModel();
    }

    @Override
    public JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(MacroWasteBatchRowModel row) {
        boolean result = row.getMacroWasteCategory() != null &&
                         row.getMacroWasteSizeCategory() != null &&
                         row.getWeight() != null;
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 MacroWasteBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {
        recomputeRowValidState(row);
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<MacroWasteBatchRowModel> rowMonitor,
                                             MacroWasteBatchRowModel row) {

        if (row != null) {

            if (row.isValid()) {
                // there is a valid bean attached to the monitor

                if (rowMonitor.wasModified()) {

                    // monitored bean was modified, save it
                    if (log.isInfoEnabled()) {
                        log.info("Row " + row + " was modified, will save it");
                    }

                    saveRow(row);

                    // clear modified flag on the monitor
                    rowMonitor.clearModified();
                }
            } else {

                // row is not valid can not save it

                MacroWasteBatch catchBean = row.toBean();

                if (!TuttiEntities.isNew(catchBean)) {

                    // remove this
                    persistenceService.deleteMacroWasteBatch(catchBean.getId());
                }
            }
        }
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    public SwingValidator<MacroWasteBatchUIModel> getValidator() {
        return ui.getValidator();
    }

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        MacroWasteBatchUIModel model = new MacroWasteBatchUIModel(catchesUIModel);
        ui.setContextValue(model);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        JXTable table = getTable();

        // create table column model
        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        {
            // MacroWaste Category column

            Decorator<CaracteristicQualitativeValue> decorator = getDecorator(
                    CaracteristicQualitativeValue.class, null);

            addComboDataColumnToModel(columnModel,
                                      MacroWasteBatchTableModel.MACRO_WASTE_CATEGORY,
                                      decorator, persistenceService.getMacroWasteCategoryCaracteristic().getQualitativeValue());
        }

        {
            // MacroWaste Size Category column

            Decorator<CaracteristicQualitativeValue> decorator = getDecorator(
                    CaracteristicQualitativeValue.class, null);

            addComboDataColumnToModel(columnModel,
                                      MacroWasteBatchTableModel.MACRO_WASTE_SIZE_CATEGORY,
                                      decorator, persistenceService.getMacroWasteSizeCategoryCaracteristic().getQualitativeValue());
        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  MacroWasteBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

        { // Number column

            addIntegerColumnToModel(columnModel,
                                    MacroWasteBatchTableModel.NUMBER,
                                    TuttiUI.INT_3_DIGITS_PATTERN);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             CommentCellEditor.newEditor(ui),
                             CommentCellRenderer.newRender(),
                             MacroWasteBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellEditor.newEditor(ui),
                             AttachmentCellRenderer.newRender(getDecorator(Attachment.class, null)),
                             MacroWasteBatchTableModel.ATTACHMENT);
        }

        // create table model
        MacroWasteBatchTableModel tableModel =
                new MacroWasteBatchTableModel(columnModel);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);
    }

    @Override
    public void onCloseUI() {
        if (log.isDebugEnabled()) {
            log.debug("closing: " + ui);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void saveRow(MacroWasteBatchRowModel row) {

        MacroWasteBatch catchBean = row.toBean();

        FishingOperation fishingOperation = getModel().getFishingOperation();
        catchBean.setFishingOperation(fishingOperation);
        if (log.isInfoEnabled()) {
            log.info("Selected fishingOperation: " + fishingOperation.getId());
        }

        if (TuttiEntities.isNew(catchBean)) {

            catchBean = persistenceService.createMacroWasteBatch(catchBean);
            row.setId(catchBean.getId());
        } else {
            persistenceService.saveMacroWasteBatch(catchBean);
        }
    }
}