package fr.ifremer.tutti.ui.swing.util.attachment;

/*
 * #%L
 * Tutti :: UI
 * $Id: AttachmentEditorUIHandler.java 567 2013-03-09 13:59:14Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/attachment/AttachmentEditorUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.TuttiUIContext;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.ComponentMover;
import jaxx.runtime.swing.ComponentResizer;
import jaxx.runtime.swing.HBox;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JRootPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.util.List;

import static org.nuiton.i18n.I18n._;

/**
 * @author kmorin <morin@codelutin.com>
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class AttachmentEditorUIHandler {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(AttachmentEditorUIHandler.class);

    public static final String CLOSE_DIALOG_ACTION = "closeDialog";

    public static final String SHOW_DIALOG_ACTION = "showDialog";

    protected final AttachmentEditorUI ui;

    protected final PersistenceService persistenceService;

    protected final TuttiUIContext context;

    public AttachmentEditorUIHandler(TuttiUIContext context,
                                     AttachmentEditorUI ui) {
        this.context = context;
        this.ui = ui;
        this.persistenceService = context.getPersistenceService();

//        this.ui.addPropertyChangeListener(new PropertyChangeListener() {
//
//            @Override
//            public void propertyChange(PropertyChangeEvent evt) {
//                if (log.isInfoEnabled()) {
//                    log.info("Init (property " + evt.getPropertyName() + " changed).");
//                }
//                init();
//            }
//        });
    }

    protected void beforeInitUI() {
    }

    protected void afterInitUI() {

        ui.getFile().setDialogOwner(ui);
        ui.pack();
        ui.setResizable(true);
//        ui.setSize(800, 300);

        ComponentResizer cr = new ComponentResizer();
        cr.registerComponent(ui);
        ComponentMover cm = new ComponentMover();
        cm.setDragInsets(cr.getDragInsets());
        cm.registerComponent(ui);

        JRootPane rootPane = ui.getRootPane();

        KeyStroke shortcutClosePopup = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);

        rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
                shortcutClosePopup, CLOSE_DIALOG_ACTION);

        closeAction = new AbstractAction() {
            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                ui.dispose();
                ui.setVisible(false);
            }
        };

        openAction = new AbstractAction() {

            private static final long serialVersionUID = 1L;

            @Override
            public void actionPerformed(ActionEvent e) {
                ui.setVisible(true);
            }
        };

        ImageIcon actionIcon = SwingUtil.createActionIcon("close-dialog");
        closeAction.putValue(Action.SMALL_ICON, actionIcon);
        closeAction.putValue(Action.LARGE_ICON_KEY, actionIcon);
        closeAction.putValue(Action.ACTION_COMMAND_KEY, "close");
        closeAction.putValue(Action.NAME, "close");
        closeAction.putValue(Action.SHORT_DESCRIPTION, _("tutti.action.closeAttachment.tip"));

        rootPane.getActionMap().put(CLOSE_DIALOG_ACTION, closeAction);
        rootPane.getActionMap().put(SHOW_DIALOG_ACTION, openAction);

        JButton closeButton = new JButton(closeAction);
        closeButton.setText(null);
        closeButton.setFocusPainted(false);
        closeButton.setRequestFocusEnabled(false);
        closeButton.setFocusable(false);

        JToolBar jToolBar = new JToolBar();
        jToolBar.setOpaque(false);
        jToolBar.add(closeAction);
        jToolBar.setBorderPainted(false);
        jToolBar.setFloatable(false);
        ui.getAttachmentBody().setRightDecoration(jToolBar);
    }

    protected Action closeAction;

    protected Action openAction;

    public void closeEditor() {

        closeAction.actionPerformed(null);
    }

    public void openEditor(JComponent component) {

        if (component != null) {
            place(component);
        }
        openAction.actionPerformed(null);
    }

    public void init() {
        resetFields();
        ui.getAttachments().removeAll();
        AttachmentModelAware bean = ui.getBean();
        if (bean != null) {
            List<Attachment> list = bean.getAttachment();
            if (list != null) {
                for (Attachment attachment : list) {
                    addAttachment(attachment);
                }
            }
        }
    }

    public void place(JComponent component) {
        // Computes the location of bottom left corner of the cell
        Component comp = component;
        int x = 0;
        int y = component.getHeight();
        while (comp != null) {
            x += comp.getX();
            y += comp.getY();
            comp = comp.getParent();
        }

        ui.pack();
        // if the editor is too big on the right,
        // then align its right side to the right side of the cell
        if (x + ui.getWidth() > ui.getOwner().getX() + ui.getOwner().getWidth()) {
            x = x - ui.getWidth() + component.getWidth();
        }
        ui.setLocation(x, y);
    }

    public void addAttachment() {
        AttachmentModelAware bean = ui.getBean();
        Attachment attachment = new Attachment();

        File file = ui.getFile().getSelectedFile();
        if (file != null) {
            String name = ui.getFileName().getText();
            if (StringUtils.isEmpty(name)) {
                name = file.getName();
            }
            attachment.setObjectType(bean.getObjectType());
            attachment.setObjectId(Integer.valueOf(bean.getObjectId()));
            attachment.setName(name);
            attachment.setComment(ui.getFileComment().getText());

            attachment = persistenceService.createAttachment(attachment, file);
            bean.addAttachment(attachment);

            resetFields();
            addAttachment(attachment);

            ui.pack();
        }
    }

    protected void addAttachment(final Attachment attachment) {
        final HBox hbox = new HBox();
        hbox.setVerticalAlignment(SwingConstants.CENTER);

        JButton deleteButton = new JButton(SwingUtil.createActionIcon("delete"));
        deleteButton.addActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                AttachmentModelAware bean = ui.getBean();

                persistenceService.deleteAttachment(attachment.getId());

                bean.removeAttachment(attachment);

                ui.getAttachments().remove(hbox);

                ui.pack();
            }
        });
        hbox.add(deleteButton);

        final JLabel label = new JLabel("<html><body><a href=''>"
                                        + attachment.getName() + "</a></body></html>");
        label.setToolTipText(attachment.getComment());
        label.addMouseListener(new MouseAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {
                File file = persistenceService.getAttachmentFile(attachment.getId());
                SwingUtil.openLink("file://" + file.getAbsolutePath());
            }

            @Override
            public void mouseEntered(MouseEvent e) {
                label.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
            }

            @Override
            public void mouseExited(MouseEvent e) {
                label.setCursor(Cursor.getDefaultCursor());
            }

        });
        hbox.add(label);

        ui.getAttachments().add(hbox);
    }

    protected void resetFields() {
        ui.getFile().setSelectedFile((File) null);
        ui.getFile().setSelectedFile((String) null);
        ui.getFileName().setText("");
        ui.getFileComment().setText("");
    }

}
