package fr.ifremer.tutti.ui.swing.util.attachment;

/*
 * #%L
 * Tutti :: UI
 * $Id: AttachmentCellEditor.java 534 2013-03-05 10:45:23Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/attachment/AttachmentCellEditor.java $
 * %%
 * Copyright (C) 2012 - 2013 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.persistence.entities.data.AttachementObjectTypeEnum;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.AbstractTuttiTableModel;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractCellEditor;
import javax.swing.JTable;
import javax.swing.border.LineBorder;
import javax.swing.table.TableCellEditor;
import java.awt.Color;
import java.awt.Component;
import java.awt.Frame;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import static org.nuiton.i18n.I18n._;

/**
 * To edit attachments from a table cell.
 *
 * @author tchemit <chemit@codelutin.com>
 * @author kmorin <morin@codelutin.com>
 * @since 1.0.2
 */
public class AttachmentCellEditor extends AbstractCellEditor implements TableCellEditor {

    private static final long serialVersionUID = 1L;

    /** Logger. */
    private static final Log log = LogFactory.getLog(AttachmentCellEditor.class);

    public static TableCellEditor newEditor(TuttiUI ui,
                                            AttachmentCellRenderer renderer,
                                            AttachementObjectTypeEnum objectType) {

        Frame frame = SwingUtil.getParentContainer(ui, Frame.class);

        renderer.setBorder(new LineBorder(Color.BLACK));

        return new AttachmentCellEditor(frame, renderer, objectType);
    }

    protected final AttachmentCellRenderer component;

    protected AttachmentEditor attachmentEditor;

    protected JTable table;

    protected AbstractTuttiTableModel<AbstractTuttiBeanUIModel> tableModel;

    protected ColumnIdentifier<AbstractTuttiBeanUIModel> columnIdentifier;

    protected AttachmentModelAware editRow;

    protected Integer rowIndex;

    protected Integer columnIndex;

    protected PropertyChangeListener reloadAttachmentEditorTextListener;

    public AttachmentCellEditor(Frame frame,
                                AttachmentCellRenderer component,
                                AttachementObjectTypeEnum objectTypeProperty) {

        this.component = component;

        reloadAttachmentEditorTextListener = new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {

                tableModel.fireTableCellUpdated(rowIndex, columnIndex);
//                List<Attachment> attachment = (List<Attachment>) evt.getNewValue();
//                AttachmentCellEditor.this.component.setText(ButtonAttachmentEditor.getButtonText(attachment));
            }
        };

        this.attachmentEditor = new AttachmentEditor(frame,
                                                     component,
                                                     objectTypeProperty,
                                                     _("tutti.table.accidental.batch.header.file")) {

            private static final long serialVersionUID = 1L;

            @Override
            protected AttachmentModelAware getBean() {
                return editRow;
            }

            @Override
            public void startEdit() {

                ((AbstractTuttiBeanUIModel) getBean()).addPropertyChangeListener(AttachmentModelAware.PROPERTY_ATTACHMENT, reloadAttachmentEditorTextListener);

                try {
                    super.startEdit();
                    int r = rowIndex;
                    int c = columnIndex;

                    // stop edition
                    stopCellEditing();

                    // refresh cell
                    tableModel.fireTableCellUpdated(r, c);

                    // reselect this cell
                    AbstractSelectTableAction.doSelectCell(table, r, c);

                    // focus on table
                    table.requestFocus();
                } finally {

                    ((AbstractTuttiBeanUIModel) getBean()).removePropertyChangeListener(AttachmentModelAware.PROPERTY_ATTACHMENT, reloadAttachmentEditorTextListener);
                }
            }
        };
    }

    @Override
    public Component getTableCellEditorComponent(JTable table,
                                                 Object value,
                                                 boolean isSelected,
                                                 int row,
                                                 int column) {
        this.table = table;
        this.tableModel = (AbstractTuttiTableModel<AbstractTuttiBeanUIModel>) table.getModel();
        columnIdentifier = tableModel.getPropertyName(column);

        rowIndex = row;
        columnIndex = column;

        editRow = (AttachmentModelAware) tableModel.getEntry(row);

        return component;
    }

    @Override
    public Object getCellEditorValue() {

        Preconditions.checkNotNull(editRow, "No editRow found in editor.");

        String propertyName = columnIdentifier.getPropertyName();
        Object result = TuttiUIUtil.getProperty(editRow, propertyName);
        if (log.isInfoEnabled()) {
            log.info("editor value (" + propertyName + "): " + result);
        }

        return result;
    }

    @Override
    public boolean stopCellEditing() {
        boolean b = super.stopCellEditing();
        if (b) {
            rowIndex = null;
            editRow = null;
            columnIndex = null;
        }
        return b;
    }

    @Override
    public void cancelCellEditing() {
        super.cancelCellEditing();
        rowIndex = null;
        columnIndex = null;
        editRow = null;
    }
}
