package fr.ifremer.tutti.ui.swing.util;

/*
 * #%L
 * Tutti :: UI
 * $Id: TuttiUIUtil.java 517 2013-03-02 16:44:34Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-1.0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/util/TuttiUIUtil.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.TuttiTechnicalException;
import fr.ifremer.tutti.ui.swing.content.MainUI;
import jaxx.runtime.JAXXUtil;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.Highlighter;
import org.nuiton.util.FileUtil;

import javax.swing.JOptionPane;
import javax.swing.UIManager;
import java.awt.Color;
import java.awt.Component;
import java.io.File;
import java.util.Date;

import static org.nuiton.i18n.I18n._;

/**
 * Created: 14/06/12
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public final class TuttiUIUtil {

    /** Logger. */
    private static final Log log = LogFactory.getLog(TuttiUIUtil.class);

    private TuttiUIUtil() {
        // never instanciate util class
    }

    /**
     * Choisir un fichier via un sélecteur graphique de fichiers.
     *
     * @param parent      le component swing appelant le controle
     * @param title       le titre du dialogue de sélection
     * @param buttonLabel le label du boutton d'acceptation
     * @param incoming    le fichier de base à utilier
     * @param filters     les filtres + descriptions sur le sélecteur de
     *                    fichiers
     * @return le fichier choisi ou le fichier incoming si l'opération a été
     *         annulée
     */
    public static File chooseFile(Component parent,
                                  String title,
                                  String buttonLabel,
                                  File incoming,
                                  String... filters) {
        File oldBasedir = FileUtil.getCurrentDirectory();
        if (incoming != null) {
            File basedir;
            if (incoming.isFile()) {
                basedir = incoming.getParentFile();
            } else {
                basedir = incoming;
            }
            if (basedir.exists()) {
                FileUtil.setCurrentDirectory(basedir);
            }
        }
        File file = FileUtil.getFile(title, buttonLabel, parent, filters);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        FileUtil.setCurrentDirectory(oldBasedir);
        File result = file == null ? incoming : file;
        return result;
    }

    /**
     * Choisir un répertoire via un sélecteur graphique de fichiers.
     *
     * @param parent      le component swing appelant le controle
     * @param title       le titre de la boite de dialogue de sléection
     * @param buttonLabel le label de l'action d'acceptation
     * @param incoming    le fichier de base à utiliser
     * @return le répertoire choisi ou le répertoire incoming si l'opération a
     *         été annulée
     */
    public static File chooseDirectory(Component parent,
                                       String title,
                                       String buttonLabel,
                                       File incoming) {
        File oldBasedir = FileUtil.getCurrentDirectory();
        if (incoming != null) {
            File basedir;
            if (incoming.isFile()) {
                basedir = incoming.getParentFile();
            } else {
                basedir = incoming;
            }
            if (basedir.exists()) {
                FileUtil.setCurrentDirectory(basedir);
            }
        }
        String file = FileUtil.getDirectory(parent, title, buttonLabel);
        if (log.isDebugEnabled()) {
            log.debug(title + " : " + file);
        }
        FileUtil.setCurrentDirectory(oldBasedir);
        return file == null ? incoming : new File(file);
    }

    public static void showSuccessMessage(Component ui,String title, String message) {
        JOptionPane.showMessageDialog(
                ui,
                message,
                title,
                JOptionPane.INFORMATION_MESSAGE,
                UIManager.getIcon("info")
        );
    }

    public static void showErrorMessage(Component ui, String message) {
        JOptionPane.showMessageDialog(
                ui,
                message,
                "Error",
                JOptionPane.ERROR_MESSAGE,
                UIManager.getIcon("error")
        );
    }

    public static boolean askQuestion(Component ui, String title, String message) {
        int i = JOptionPane.showConfirmDialog(
                ui,
                message,
                title,
                JOptionPane.YES_NO_OPTION);
        return i == JOptionPane.YES_OPTION;
    }

    public static void setProperty(Object bean, String property, Object value) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            PropertyUtils.setSimpleProperty(bean, property, value);
        } catch (Exception e) {
            throw new TuttiTechnicalException("Could not set property '" + property + "' on object of type '" + bean.getClass().getName() + "'", e);
        }
    }

    public static Object getProperty(Object bean, String property) {
        Preconditions.checkNotNull(bean);
        Preconditions.checkNotNull(property);
        try {
            return PropertyUtils.getSimpleProperty(bean, property);
        } catch (Exception e) {
            throw new TuttiTechnicalException("Could not get property '" + property + "' on object of type '" + bean.getClass().getName() + "'", e);
        }
    }

    public static Highlighter newBackgroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, false);
    }

    public static Highlighter newForegroundColorHighlighter(HighlightPredicate predicate, Color color) {
        return new TuttiColorHighlighter(predicate, color, true);
    }

    public static String getDuration(Date startDate, Date endDate) {
        String duration = "";
        if (startDate != null && endDate != null && !startDate.after(endDate)) {
            duration = DurationFormatUtils.formatPeriod(
                    startDate.getTime(),
                    endDate.getTime(),
                    _("tutti.duration.format"));
        }
        return duration;
    }

    public static int computeDistanceInMeters(Float startLatitude,
                                              Float startLongitude,
                                              Float endLatitude,
                                              Float endLongitude) {

        double sLat = startLatitude * Math.PI / 180;
        double sLong = startLongitude * Math.PI / 180;
        double eLat = endLatitude * Math.PI / 180;
        double eLong = endLongitude * Math.PI / 180;

        Double d = 6378288 *
                   (Math.PI / 2 - Math.asin(Math.sin(eLat) * Math.sin(sLat)
                                            + Math.cos(eLong - sLong) * Math.cos(eLat) * Math.cos(sLat)));
        return d.intValue();
    }

    public static String getDistanceInMilles(Float distance) {
        String distanceText;
        if (distance != null) {
            Float distanceInMilles = distance / 1852;
            distanceText = String.format("%.3f", distanceInMilles);

        } else {
            distanceText = "";
        }
        return distanceText;
    }

    public static String getWeightStringValue(Float weight) {
        String textValue;
        if (weight != null) {
            textValue = String.format("%.3f", weight);

        } else {
            textValue = JAXXUtil.getStringValue(weight);
        }
        return textValue;
    }

    public static boolean askOverwriteFile(MainUI ui, File file) {
        boolean result =
                !file.exists() ||
                askQuestion(ui,
                            _("tutti.dialog.askOverwriteFile.title"),
                            _("tutti.dialog.askOverwriteFile.message", file));
        return result;
    }
}
