package fr.ifremer.tutti.ui.swing.content.operation.catches.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchUIHandler.java 203 2013-01-15 11:33:29Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/SpeciesBatchUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.FishingOperation;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.ui.swing.TuttiUI;
import fr.ifremer.tutti.ui.swing.content.operation.AbstractTuttiBatchTableUIHandler;
import fr.ifremer.tutti.ui.swing.content.operation.catches.EditCatchesUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.FrequencyCellComponent;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.CreateSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.CreateSpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchRowModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchUI;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.split.SplitSpeciesBatchUIModel;
import fr.ifremer.tutti.ui.swing.util.TuttiBeanMonitor;
import fr.ifremer.tutti.ui.swing.util.editor.AttachmentCellComponent;
import fr.ifremer.tutti.ui.swing.util.editor.LongTextCellComponent;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.JAXXUtil;
import jaxx.runtime.context.JAXXContextEntryDef;
import jaxx.runtime.swing.ErrorDialogUI;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.table.DefaultTableColumnModelExt;
import org.nuiton.util.decorator.Decorator;

import javax.swing.RowFilter;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumnModel;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.Serializable;
import java.util.EnumMap;
import java.util.List;
import java.util.Set;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class SpeciesBatchUIHandler extends AbstractTuttiBatchTableUIHandler<SpeciesBatchRowModel, SpeciesBatchUIModel> {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(SpeciesBatchUIHandler.class);

    public static JAXXContextEntryDef<List<Caracteristic>> FREQUENCY_LENGTH_CONTEXT_ENTRY =
            JAXXUtil.newListContextEntryDef("frequencyLength");

    public static final Set<String> SAMPLING_PROPERTIES = Sets.newHashSet(
            SpeciesBatchRowModel.PROPERTY_SAMPLE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SPECIES,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY,
            SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
            SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT);

    /**
     * UI.
     *
     * @since 0.2
     */
    private final SpeciesBatchUI ui;

    private SplitSpeciesBatchUI splitSpeciesBatchEditor;

    private CreateSpeciesBatchUI createSpeciesBatchEditor;

    private SpeciesFrequencyUI speciesFrequencyEditor;

    private final EnumMap<TableViewMode, RowFilter<SpeciesBatchTableModel, Integer>> tableFilters;

    public SpeciesBatchUIHandler(TuttiUI<?, ?> parentUi,
                                 SpeciesBatchUI ui) {
        super(parentUi,
              SpeciesBatchRowModel.PROPERTY_SPECIES_TO_CONFIRM,
              SpeciesBatchRowModel.PROPERTY_SPECIES,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SIZE_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_SEX_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_MATURITY_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY,
              SpeciesBatchRowModel.PROPERTY_AGE_CATEGORY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_WEIGHT,
              SpeciesBatchRowModel.PROPERTY_NUMBER,
              SpeciesBatchRowModel.PROPERTY_COMMENT,
              SpeciesBatchRowModel.PROPERTY_ATTACHMENTS,
              SpeciesBatchRowModel.PROPERTY_FREQUENCY);
        this.ui = ui;
        tableFilters = new EnumMap<TableViewMode, RowFilter<SpeciesBatchTableModel, Integer>>(TableViewMode.class);

        tableFilters.put(TableViewMode.ALL, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                return true;
            }
        });

        tableFilters.put(TableViewMode.ROOT, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    SpeciesBatchTableModel model = entry.getModel();
                    SpeciesBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchRoot();
                }
                return result;
            }
        });

        tableFilters.put(TableViewMode.LEAF, new RowFilter<SpeciesBatchTableModel, Integer>() {
            @Override
            public boolean include(Entry<? extends SpeciesBatchTableModel, ? extends Integer> entry) {
                boolean result = false;
                Integer rowIndex = entry.getIdentifier();
                if (rowIndex != null) {
                    SpeciesBatchTableModel model = entry.getModel();
                    SpeciesBatchRowModel row = model.getEntry(rowIndex);
                    result = row != null && row.isBatchLeaf();
                }
                return result;
            }
        });
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiBatchTableUIHandler methods                           --//
    //------------------------------------------------------------------------//

    @Override
    public void selectFishingOperation(FishingOperation bean) {

        boolean empty = bean == null;

        SpeciesBatchUIModel model = getModel();

        List<SpeciesBatchRowModel> rows;

        if (empty) {
            rows = null;
        } else {

            if (log.isInfoEnabled()) {
                log.info("Get species batch for fishingOperation: " +
                         bean.getId());
            }
            rows = Lists.newArrayList();

            if (!TuttiEntities.isNew(bean)) {

                // get all batch species root (says the one with only a species sample category)
                List<SpeciesBatch> catches =
                        persistenceService.getAllRootSpeciesBatch(bean.getId());

                for (SpeciesBatch aBatch : catches) {

                    // root batch sample categroy is species
                    Preconditions.checkState(
                            aBatch.getSampleCategoryType() == SampleCategoryEnum.sortedUnsorted,
                            "Root species batch must be a sortedUnsorted sample " +
                            "category but was:" + aBatch.getSampleCategoryType());

                    SpeciesBatchRowModel rootRow =
                            loadSpeciesBatch(aBatch, null, rows);

                    if (log.isInfoEnabled()) {
                        log.info("Loaded root batch " +
                                 decorate(rootRow.getSpecies()) + " - " +
                                 decorate(rootRow.getSortedUnsortedCategoryValue()));
                    }
                }
            }
        }

        model.setRows(rows);
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiTableUIHandler methods                                --//
    //------------------------------------------------------------------------//

    @Override
    protected SpeciesBatchTableModel getTableModel() {
        return (SpeciesBatchTableModel) getTable().getModel();
    }

    @Override
    protected JXTable getTable() {
        return ui.getTable();
    }

    @Override
    protected boolean isRowValid(SpeciesBatchRowModel row) {

        // a row is valid if species category is not empty and valid
        // then if any of none empty category is valid
        boolean result = row.getSpecies() != null &&
                         !row.getSortedUnsortedCategory().isEmpty() &&
                         row.getSortedUnsortedCategory().isValid();

        result &= row.getSizeCategory().isEmptyOrValid();
        result &= row.getSexCategory().isEmptyOrValid();
        result &= row.getMaturityCategory().isEmptyOrValid();
        result &= row.getAgeCategory().isEmptyOrValid();
        return result;
    }

    @Override
    protected void onRowModified(int rowIndex,
                                 SpeciesBatchRowModel row,
                                 String propertyName,
                                 Object oldValue,
                                 Object newValue) {

        if (SAMPLING_PROPERTIES.contains(propertyName) ||
            SpeciesBatchRowModel.PROPERTY_WEIGHT.equals(propertyName)) {

            // species has changed, recompute valid property
            recomputeRowValidState(row);

            // recompute the totalUnsorted weight
            recomputeTotalUnsortedWeight();
        }

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void saveSelectedRowIfRequired(TuttiBeanMonitor<SpeciesBatchRowModel> rowMonitor,
                                             SpeciesBatchRowModel row) {

        if (row.isValid()) {
            // there is a valid bean attached to the monitor

            if (rowMonitor.wasModified()) {

                // monitored bean was modified, save it
                if (log.isInfoEnabled()) {
                    log.info("Row " + row + " was modified, will save it");
                }

                showInformationMessage(
                        "[ Captures - Espèces ] " +
                        "Sauvegarde des modifications de " + row + '.');

                saveRow(row);

                // clear modified flag on the monitor
                rowMonitor.clearModified();
            }
        } else {

            //FIXME See how to delete rows ? Or moreover how to save tehem...
            if (log.isWarnEnabled()) {
                log.warn("Will not remove not valid speciesBatch: " + row.getId());
            }

//            // row is not valid can not save it
//
//            SpeciesBatch catchBean = row.toBean();
//
//            if (!TuttiEntities.isNew(catchBean)) {
//
//                // remove this
//                persistenceService.deleteSpeciesBatch(catchBean.getId());
//                row.setId(null);
//            }
        }
    }

    @Override
    protected void onModelRowsChanged(List<SpeciesBatchRowModel> rows) {
        super.onModelRowsChanged(rows);

        // clear speciesUsed
        getModel().getSpeciesUsed().clear();

        for (SpeciesBatchRowModel row : rows) {
            updateTotalFromFrequencies(row);
            if (row.isBatchRoot()) {

                // update speciesUsed
                addToSpeciesUsed(row);
            }
        }
    }

    @Override
    protected void onRowValidStateChanged(int rowIndex,
                                          SpeciesBatchRowModel row,
                                          Boolean oldValue,
                                          Boolean newValue) {
        super.onRowValidStateChanged(rowIndex, row, oldValue, newValue);

        // when row valid state has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    @Override
    protected void onAfterSelectedRowChanged(int oldRowIndex,
                                             SpeciesBatchRowModel oldRow,
                                             int newRowIndex,
                                             SpeciesBatchRowModel newRow) {
        super.onAfterSelectedRowChanged(oldRowIndex, oldRow, newRowIndex, newRow);

        // when selected row has changed, recompute action enabled states
        recomputeBatchActionEnable();
    }

    //------------------------------------------------------------------------//
    //-- AbstractTuttiUIHandler methods                                     --//
    //------------------------------------------------------------------------//

    @Override
    protected SpeciesBatchUIModel getModel() {
        return ui.getModel();
    }

    @Override
    public void beforeInitUI() {

        if (log.isInfoEnabled()) {
            log.info("beforeInit: " + ui);
        }

        EditCatchesUIModel catchesUIModel =
                ui.getContextValue(EditCatchesUIModel.class);

        SpeciesBatchUIModel model = new SpeciesBatchUIModel(catchesUIModel);
        model.setTableViewMode(TableViewMode.ALL);
        ui.setContextValue(model);

        PersistenceService service =
                context.getService(PersistenceService.class);
        List<Caracteristic> lengthCaracterics =
                service.getAllSpeciesLengthStepCaracteristic();
        FREQUENCY_LENGTH_CONTEXT_ENTRY.setContextValue(ui, lengthCaracterics);
    }

    @Override
    public void afterInitUI() {

        if (log.isInfoEnabled()) {
            log.info("afterInit: " + ui);
        }

        initUI(ui);

        List<SampleCategoryType> samplingOrder;

        List<Species> allSpecies;

        TuttiProtocol protocol = ui.getContextValue(TuttiProtocol.class);

        Multimap<Species, SampleCategoryType> speciesSampleCategories =
                HashMultimap.create();

        if (protocol != null) {

            // fill sampling order from protocol

            List<SampleCategoryEnum> sampleCategoryOrder =
                    protocol.getSampleCategoryOrder();
            samplingOrder = Lists.newArrayList();
            for (SampleCategoryEnum sampleCategoryEnum : sampleCategoryOrder) {

                SampleCategoryType sampleCategoryType =
                        SampleCategoryType.valueOf(sampleCategoryEnum);
                samplingOrder.add(sampleCategoryType);
            }

            // fill available species from protocol

            allSpecies = Lists.newArrayList();
            List<SpeciesProtocol> protocolSpecies = protocol.getSpecies();
            for (SpeciesProtocol protocolSpecy : protocolSpecies) {
                Species species = persistenceService.getSpecies(
                        protocolSpecy.getSpeciesId());
                allSpecies.add(species);
                speciesSampleCategories.put(species, SampleCategoryType.sortedUnsorted);
                speciesSampleCategories.put(species, SampleCategoryType.size);
                speciesSampleCategories.put(species, SampleCategoryType.sex);
                speciesSampleCategories.put(species, SampleCategoryType.maturity);
                speciesSampleCategories.put(species, SampleCategoryType.age);
            }
        } else {

            // no protocol, use default values

            samplingOrder = Lists.newArrayList(
                    SampleCategoryType.sortedUnsorted,
                    SampleCategoryType.size,
                    SampleCategoryType.sex,
                    SampleCategoryType.maturity,
                    SampleCategoryType.age);

            allSpecies = Lists.newArrayList(
                    persistenceService.getAllSpecies());

            // each species can use any category
            for (Species species : allSpecies) {
                speciesSampleCategories.put(species, SampleCategoryType.sortedUnsorted);
                speciesSampleCategories.put(species, SampleCategoryType.size);
                speciesSampleCategories.put(species, SampleCategoryType.sex);
                speciesSampleCategories.put(species, SampleCategoryType.maturity);
                speciesSampleCategories.put(species, SampleCategoryType.age);
            }
        }

        if (log.isInfoEnabled()) {
            log.info("Will use sampling order: " + samplingOrder);
            log.info("Will use " + allSpecies.size() + " species.");
        }

        getModel().setSamplingOrder(samplingOrder);
        getModel().setAllSpecies(allSpecies);

        JXTable table = getTable();

        table.setColumnControlVisible(true);
        // create table column model
        TableCellRenderer defaultRenderer =
                table.getDefaultRenderer(Object.class);

        DefaultTableColumnModelExt columnModel =
                new DefaultTableColumnModelExt();

        Decorator<CaracteristicQualitativeValue> caracteristicDecorator =
                getDecorator(CaracteristicQualitativeValue.class, null);

        { // Species column

            addColumnToModel(columnModel,
                             null,
                             newTableCellRender(Species.class),
                             SpeciesBatchTableModel.SPECIES);
        }

        { // SortedUnsortedCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SORTED_UNSORTED_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // SizeCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SIZE_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // SexCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.SEX_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // MaturityCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.MATURITY_CATEGORY,
                                           caracteristicDecorator,
                                           defaultRenderer);
        }

        { // AgeCategory column

            addSampleCategoryColumnToModel(columnModel,
                                           SpeciesBatchTableModel.AGE_CATEGORY,
                                           getDecorator(Float.class, null),
                                           defaultRenderer);
        }

        { // Weight column

            addFloatColumnToModel(columnModel,
                                  SpeciesBatchTableModel.WEIGHT,
                                  TuttiUI.DECIMAL3_PATTERN);
        }

//        { // Computed weight column (from frequencies)
//
//            addColumnToModel(columnModel,
//                             FrequencyCellComponent.newEditor(ui),
//                             FrequencyCellComponent.newRender(),
//                             SpeciesBatchTableModel.COMPUTED_WEIGHT);
//        }

        { // Number column (from frequencies)

            addColumnToModel(columnModel,
                             FrequencyCellComponent.newEditor(ui),
                             FrequencyCellComponent.newRender(),
                             SpeciesBatchTableModel.COMPUTED_NUMBER);
        }

        { // Comment column

            addColumnToModel(columnModel,
                             LongTextCellComponent.newEditor(ui.getLongTextEditor()),
                             LongTextCellComponent.newRender(n_("tutti.tooltip.comment.none")),
                             SpeciesBatchTableModel.COMMENT);
        }

        { // File column

            addColumnToModel(columnModel,
                             AttachmentCellComponent.newEditor(ui.getAttachmentEditor()),
                             AttachmentCellComponent.newRender(
                                     getDecorator(Attachment.class, null),
                                     n_("tutti.tooltip.attachment.none")),
                             SpeciesBatchTableModel.ATTACHMENTS);
        }

        { // Species to confirm column

            addBooleanColumnToModel(columnModel,
                                    SpeciesBatchTableModel.SPECIES_TO_CONFIRM,
                                    getTable());
        }

        // create table model
        SpeciesBatchTableModel tableModel =
                new SpeciesBatchTableModel(columnModel, speciesSampleCategories);

        table.setModel(tableModel);
        table.setColumnModel(columnModel);

        initBatchTable(table, columnModel, tableModel);

        getModel().addPropertyChangeListener(SpeciesBatchUIModel.PROPERTY_TABLE_VIEW_MODE, new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                TableViewMode tableViewMode = (TableViewMode) evt.getNewValue();

                if (tableViewMode == null) {
                    tableViewMode = TableViewMode.ALL;
                }

                if (log.isDebugEnabled()) {
                    log.debug("Will use rowfilter for viewMode: " + tableViewMode);
                }
                RowFilter<SpeciesBatchTableModel, Integer> filter = tableFilters.get(tableViewMode);
                getTable().setRowFilter(filter);
            }
        });
    }


    @Override
    public void onCloseUI() {
        if (log.isInfoEnabled()) {
            log.info("Closing: " + ui);
        }

        if (splitSpeciesBatchEditor != null) {
            getContext().getSwingSession().add(splitSpeciesBatchEditor);
            closeUI(splitSpeciesBatchEditor);
        }

        if (createSpeciesBatchEditor != null) {
            getContext().getSwingSession().add(createSpeciesBatchEditor);
            closeUI(createSpeciesBatchEditor);
        }

        if (speciesFrequencyEditor != null) {
            getContext().getSwingSession().add(speciesFrequencyEditor);
            closeUI(speciesFrequencyEditor);
        }
    }

    //------------------------------------------------------------------------//
    //-- Public methods                                                     --//
    //------------------------------------------------------------------------//

    public void createSpeciesBatch() {

        CreateSpeciesBatchUI createBatchEditor =
                getCreateSpeciesBatchEditor();

        createBatchEditor.getHandler().openUI(getModel());

        openDialog(ui,
                   createBatchEditor,
                   _("tutti.title.createBatch"),
                   ui.getPreferredSize());

        CreateSpeciesBatchUIModel createModel = createBatchEditor.getModel();

        if (createModel.isValid()) {

            SpeciesBatchTableModel tableModel = getTableModel();

            SpeciesBatchRowModel newRow = tableModel.createNewRow();
            Species species = createModel.getSpecies();
            newRow.setSpecies(species);

            CaracteristicQualitativeValue sortedUnsortedCategory = createModel.getSortedUnsortedCategory();
            SampleCategory<CaracteristicQualitativeValue> category = newRow.getSortedUnsortedCategory();
            category.setCategoryValue(sortedUnsortedCategory);
            category.setCategoryWeight(createModel.getBatchWeight());
            newRow.setSampleCategory(category);

            recomputeRowValidState(newRow);

            tableModel.addNewRow(newRow);

            saveRow(newRow);

            // update speciesUsed
            addToSpeciesUsed(newRow);
        }

        // reset create ui
        createBatchEditor.getHandler().openUI(null);

        recomputeBatchActionEnable();
    }

    public void splitSpeciesBatch() {

        JXTable table = getTable();

        // get selected row
        int rowIndex = table.getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant split batch if no batch selected");

        SpeciesBatchTableModel tableModel = getTableModel();

        SpeciesBatchRowModel parentBatch = tableModel.getEntry(rowIndex);

        if (log.isInfoEnabled()) {
            log.info("Open split batch ui for row [" + rowIndex + ']');
        }
        SplitSpeciesBatchUI splitBatchEditor = getSplitSpeciesBatchEditor();
        splitBatchEditor.getHandler().editBatch(parentBatch);

        openDialog(ui,
                   splitBatchEditor,
                   _("tutti.title.splitBatch"),
                   ui.getPreferredSize());

        // at close, synch back batches ?
        SplitSpeciesBatchUIModel splitModel = splitBatchEditor.getModel();

        if (splitModel.isValid()) {

            // create batch rows

            int insertRow = rowIndex;

            SampleCategoryType selectedCategory = splitModel.getSelectedCategory();
            SampleCategoryEnum sampleCategoryEnum = selectedCategory.getType();

            // Create rows in batch table model

            List<SpeciesBatchRowModel> newBatches = Lists.newArrayList();
            for (SplitSpeciesBatchRowModel row : splitModel.getRows()) {
                if (row.isValid()) {

                    // can keep this row
                    SpeciesBatchRowModel newBatch = tableModel.createNewRow();

                    loadBatchRow(parentBatch,
                                 newBatch,
                                 sampleCategoryEnum,
                                 row.getCategoryValue(),
                                 row.getWeight());

                    recomputeRowValidState(newBatch);
                    newBatches.add(newBatch);

                    tableModel.addNewRow(++insertRow, newBatch);
                }
            }

            // add new batches to his parent
            parentBatch.setBatchChilds(newBatches);

            //TODO Should only save parentBatch (will persist all his childs)
            //saveRow(parentBatch);

            // save new batches
            saveRows(newBatches);
        }

        // reset split ui
        splitBatchEditor.getHandler().editBatch(null);

        recomputeBatchActionEnable();

//        // reselect this cell
//        AbstractSelectTableAction.doSelectCell(table, rowIndex, 0);
//        table.requestFocus();
    }

    public void removeSpeciesBatch() {

        JXTable table = getTable();

        int rowIndex = table.getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant remove batch if no batch selected");

        SpeciesBatchTableModel tableModel = getTableModel();
        SpeciesBatchRowModel selectedBatch = tableModel.getEntry(rowIndex);

        Preconditions.checkState(!TuttiEntities.isNew(selectedBatch),
                                 "Can't remove batch if batch is not persisted");

        if (!selectedBatch.isBatchRoot()) {

            // remove all sub batches of his parent
            SpeciesBatchRowModel parentBatch = selectedBatch.getBatchParent();

            // get parent row index
            int parentIndex = tableModel.getRowIndex(parentBatch);

            // select it
            table.setRowSelectionInterval(parentIndex, parentIndex);

            // remove all his children
            removeSpeciesSubBatch();

        } else {

            // remove selected batch and all his children

            try {

                // remove parent batch (will destroy all his childs from db)
                persistenceService.deleteSpeciesBatch(selectedBatch.getId());

                // update speciesUsed
                removeFromSpeciesUsed(selectedBatch);

                // collect of rows to remove from model
                Set<SpeciesBatchRowModel> rowToRemove =
                        Sets.newHashSet(selectedBatch);

                collectChilds(selectedBatch, rowToRemove);

                // remove all rows from the model
                getModel().getRows().removeAll(rowToRemove);

                // refresh table from parent batch row index to the end
                tableModel.fireTableDataChanged();

                if (tableModel.getRowCount() > 0) {

                    // select first row
                    AbstractSelectTableAction.doSelectCell(table, 0, 0);
                } else {

                    table.clearSelection();
                }

                if (table.isEditing()) {

                    // but no edit it
                    table.getCellEditor().stopCellEditing();
                }
            } catch (Exception e) {

                ErrorDialogUI.showError(e);
            }
        }
    }

    public void removeSpeciesSubBatch() {

        JXTable table = getTable();

        int rowIndex = table.getSelectedRow();

        Preconditions.checkState(rowIndex != -1,
                                 "Cant remove sub batch if no batch selected");

        SpeciesBatchRowModel parentBatch = getTableModel().getEntry(rowIndex);

        Preconditions.checkState(!TuttiEntities.isNew(parentBatch),
                                 "Can't remove sub batch if batch is not persisted");

        try {

            // save parent batch (will destroy all his childs from db)
            persistenceService.deleteSpeciesSubBatch(parentBatch.getId());

            if (parentBatch.isBatchRoot()) {

                // update speciesUsed
                removeFromSpeciesUsed(parentBatch);
            }

            // collect of rows to remove from model
            Set<SpeciesBatchRowModel> rowToRemove = Sets.newHashSet();

            collectChilds(parentBatch, rowToRemove);

            // remove all rows from the model
            getModel().getRows().removeAll(rowToRemove);

            // remove childs from parent batch
            parentBatch.setBatchChilds(null);

            // refresh table from parent batch row index to the end
            getTableModel().fireTableDataChanged();

            // select parent batch row
            AbstractSelectTableAction.doSelectCell(table, rowIndex, 0);

            if (table.isEditing()) {

                // but no edit it
                table.getCellEditor().stopCellEditing();
            }
        } catch (Exception e) {

            ErrorDialogUI.showError(e);
        }

    }

    public SplitSpeciesBatchUI getSplitSpeciesBatchEditor() {
        if (splitSpeciesBatchEditor == null) {
            splitSpeciesBatchEditor = new SplitSpeciesBatchUI(ui);
        }
        return splitSpeciesBatchEditor;
    }

    public CreateSpeciesBatchUI getCreateSpeciesBatchEditor() {
        if (createSpeciesBatchEditor == null) {
            createSpeciesBatchEditor = new CreateSpeciesBatchUI(ui);
        }
        return createSpeciesBatchEditor;
    }

    public SpeciesFrequencyUI getSpeciesFrequencyEditor() {
        if (speciesFrequencyEditor == null) {
            speciesFrequencyEditor = new SpeciesFrequencyUI(ui);
        }
        return speciesFrequencyEditor;
    }

    public void updateTotalFromFrequencies(SpeciesBatchRowModel row) {
        Float totalNumber = null;
        Float totalWeight = null;
        List<SpeciesFrequencyRowModel> frequency = row.getFrequency();

        if (CollectionUtils.isNotEmpty(frequency)) {
            totalNumber = 0f;
            totalWeight = 0f;
            for (SpeciesFrequencyRowModel frequencyModel : frequency) {
                totalNumber += frequencyModel.getNumber();
                Float w = frequencyModel.getWeight();
                if (w == null) {

                    // can't sum when a null value appears
                    totalWeight = null;
                } else if (totalWeight != null) {

                    // still can sum weights
                    totalWeight += w;
                }
            }
        }

        row.setComputedNumber(totalNumber);
        row.setComputedWeight(totalWeight);
    }

    //------------------------------------------------------------------------//
    //-- Internal methods                                                   --//
    //------------------------------------------------------------------------//

    protected void saveRows(Iterable<SpeciesBatchRowModel> rows) {
        for (SpeciesBatchRowModel row : rows) {
            saveRow(row);
        }
    }

    protected void saveRow(SpeciesBatchRowModel row) {

        FishingOperation fishingOperation = getModel().getFishingOperation();
        Preconditions.checkNotNull(fishingOperation);

        Preconditions.checkNotNull(row.getSpecies());
        SampleCategory<?> sampleCategory = row.getSampleCategory();
        Preconditions.checkNotNull(sampleCategory);
        Preconditions.checkNotNull(sampleCategory.getCategoryType());
        Preconditions.checkNotNull(sampleCategory.getCategoryValue());
        Preconditions.checkNotNull(sampleCategory.getCategoryWeight());

        SpeciesBatch catchBean = row.toBean();
        catchBean.setFishingOperation(fishingOperation);

        // apply sample category
        catchBean.setSampleCategoryType(sampleCategory.getCategoryType().getType());
        catchBean.setSampleCategoryValue(sampleCategory.getCategoryValue());
        catchBean.setSampleCategoryWeight(sampleCategory.getCategoryWeight());

        if (TuttiEntities.isNew(catchBean)) {

            SpeciesBatchRowModel batchParent = row.getBatchParent();
            String parentBatchId = null;

            if (batchParent != null) {
                parentBatchId = batchParent.getId();
            }

            if (log.isInfoEnabled()) {
                log.info("Persist new species batch with parentId: " +
                         parentBatchId);
            }
            catchBean = persistenceService.createSpeciesBatch(catchBean,
                                                              parentBatchId);
            row.setId(catchBean.getId());
        } else {
            if (log.isInfoEnabled()) {
                log.info("Persist existing species batch: " + catchBean.getId());
            }
            persistenceService.saveSpeciesBatch(catchBean);
        }

        List<SpeciesFrequencyRowModel> frequencyRows = row.getFrequency();

        List<SpeciesBatchFrequency> frequency =
                SpeciesFrequencyRowModel.toBeans(frequencyRows, catchBean);

        if (log.isInfoEnabled()) {
            log.info("Will save " + frequency.size() + " frequencies.");
        }
        frequency = persistenceService.saveSpeciesBatchFrequency(
                catchBean.getId(), frequency);

        // push it back to row model
        frequencyRows = SpeciesFrequencyRowModel.fromBeans(frequency);
        row.setFrequency(frequencyRows);
    }

    protected void recomputeTotalUnsortedWeight() {

        // recompute total hors vrac
        Float totalHorsVrac = 0f;
        Float totalVrac = 0f;

        for (SpeciesBatchRowModel batch : getModel().getRows()) {
            CaracteristicQualitativeValue vracHorsVrac =
                    batch.getSortedUnsortedCategoryValue();

            if (vracHorsVrac == null) {

                // can't sum anything
            } else {
                boolean unsorted = "unsorted".equals(vracHorsVrac.getName());
                if (unsorted) {
                    Float weight = batch.getWeight();
                    if (weight != null) {
                        totalHorsVrac += weight;
                    }
                } else {
                    Float weight = batch.getWeight();
                    if (weight != null) {
                        totalVrac += weight;
                    }
                }
            }
        }
        if (log.isInfoEnabled()) {
            log.info("New total vrac / hors vrac: " +
                     totalVrac + " / " + totalHorsVrac);
        }
        getModel().setSpeciesTotalUnsortedWeight(totalHorsVrac);
        //TODO Should we also set the total vrac weight ?
//        getModel().setTotalVracWeight(totalVrac);
    }

    protected void recomputeBatchActionEnable() {

        int rowIndex = getTable().getSelectedRow();

        //TODO Improve this test
        boolean enableAdd = true;
//                CollectionUtils.isNotEmpty(getModel().getAvailableSpecies());

        boolean enableSplit = false;
        boolean enableRemove = false;
        boolean enableRemoveSub = false;

        if (rowIndex != -1) {

            // there is a selected row

            //TODO If there is some sub-batch, can remove them
            //TODO If there is no sub-batch, can split current batch

            SpeciesBatchRowModel row = getTableModel().getEntry(rowIndex);

            if (row.isValid()) {

                // must have at least species filled in row
                // otherwise nothing can be done

                enableSplit = true;
                enableRemove = true;
                enableRemoveSub = true;
            }

            if (enableSplit) {

                // can split if selected batch is a leaf
                enableSplit = row.isBatchLeaf();

            }

            if (enableRemove) {

                // can always remove the batch
                enableRemove = true;
            }

            if (enableRemoveSub) {

                // can remove sub batch if selected batch is not a leaf
                enableRemoveSub = !row.isBatchLeaf();
            }
        }
        getModel().setCreateSpeciesBatchEnabled(enableAdd);
        getModel().setSplitSpeciesBatchEnabled(enableSplit);
        getModel().setRemoveSpeciesBatchEnabled(enableRemove);
        getModel().setRemoveSpeciesSubBatchEnabled(enableRemoveSub);

    }

    protected void collectChilds(SpeciesBatchRowModel row,
                                 Set<SpeciesBatchRowModel> collectedRows) {

        if (!row.isBatchLeaf()) {

            for (SpeciesBatchRowModel batchChild : row.getBatchChild()) {
                collectedRows.add(batchChild);
                collectChilds(batchChild, collectedRows);
            }
        }
    }

    protected SpeciesBatchRowModel loadSpeciesBatch(SpeciesBatch aBatch,
                                                    SpeciesBatchRowModel parentRow,
                                                    List<SpeciesBatchRowModel> rows) {

        List<SpeciesBatchFrequency> frequencies =
                persistenceService.getAllSpeciesBatchFrequency(aBatch.getId());

        SpeciesBatchRowModel newRow =
                new SpeciesBatchRowModel(aBatch, frequencies);

        SampleCategoryEnum sampleCategoryEnum =
                aBatch.getSampleCategoryType();

        Preconditions.checkNotNull(
                sampleCategoryEnum,
                "Can't have a batch with no sample category, but was: " + aBatch);

        loadBatchRow(parentRow,
                     newRow,
                     sampleCategoryEnum,
                     aBatch.getSampleCategoryValue(),
                     aBatch.getSampleCategoryWeight());

        rows.add(newRow);

        if (!aBatch.isChildBatchsEmpty()) {

            // create batch childs rows

            List<SpeciesBatchRowModel> batchChilds = Lists.
                    newArrayListWithCapacity(aBatch.sizeChildBatchs());

            for (SpeciesBatch childBatch : aBatch.getChildBatchs()) {
                SpeciesBatchRowModel childRow = loadSpeciesBatch(childBatch, newRow, rows);
                batchChilds.add(childRow);
            }
            newRow.setBatchChilds(batchChilds);
        }

        return newRow;
    }

    protected void loadBatchRow(SpeciesBatchRowModel parentRow,
                                SpeciesBatchRowModel newRow,
                                SampleCategoryEnum sampleCategoryEnum,
                                Serializable categoryValue,
                                Float categoryWeight) {

        // get sample category from his type
        SampleCategory sampleCategory =
                newRow.getSampleCategory(sampleCategoryEnum);

        // fill it
        sampleCategory.setCategoryValue(categoryValue);
        sampleCategory.setCategoryWeight(categoryWeight);

        // push it back to row as his *main* sample category
        newRow.setSampleCategory(sampleCategory);

        if (parentRow != null) {

            // copy back parent data (mainly other sample categories)

            newRow.setSpecies(parentRow.getSpecies());
            newRow.setSpeciesToConfirm(parentRow.getSpeciesToConfirm());
            newRow.setBatchParent(parentRow);

            newRow.setSpecies(parentRow.getSpecies());
            if (sampleCategoryEnum != SampleCategoryEnum.sortedUnsorted) {
                newRow.setSortedUnsortedCategory(parentRow.getSortedUnsortedCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.size) {
                newRow.setSizeCategory(parentRow.getSizeCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.sex) {
                newRow.setSexCategory(parentRow.getSexCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.maturity) {
                newRow.setMaturityCategory(parentRow.getMaturityCategory());
            }
            if (sampleCategoryEnum != SampleCategoryEnum.age) {
                newRow.setAgeCategory(parentRow.getAgeCategory());
            }
        }
    }

    protected <C extends Serializable> void addSampleCategoryColumnToModel(TableColumnModel columnModel,
                                                                           ColumnIdentifier<SpeciesBatchRowModel> columnIdentifier,
                                                                           Decorator<C> decorator,
                                                                           TableCellRenderer defaultRenderer) {
        addColumnToModel(columnModel,
                         SampleCategoryComponent.newEditor(decorator),
                         SampleCategoryComponent.newRender(defaultRenderer, decorator),
                         columnIdentifier);
    }

    protected void removeFromSpeciesUsed(SpeciesBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isInfoEnabled()) {
            log.info("Remove from speciesUsed: " + decorate(row.getSortedUnsortedCategoryValue()) + " - " + decorate(row.getSpecies()));
        }
        getModel().getSpeciesUsed().remove(row.getSortedUnsortedCategoryValue(),
                                           row.getSpecies());
    }

    protected void addToSpeciesUsed(SpeciesBatchRowModel row) {
        Preconditions.checkNotNull(row);
        Preconditions.checkNotNull(row.getSpecies());
        Preconditions.checkNotNull(row.getSortedUnsortedCategoryValue());
        if (log.isInfoEnabled()) {
            log.info("Add to speciesUsed: " + decorate(row.getSortedUnsortedCategoryValue()) + " - " + decorate(row.getSpecies()));
        }
        getModel().getSpeciesUsed().put(row.getSortedUnsortedCategoryValue(),
                                        row.getSpecies());
    }

}
