package fr.ifremer.tutti.ui.swing.content.operation.catches.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: SpeciesBatchRowModel.java 173 2013-01-11 14:13:28Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.3/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/catches/species/SpeciesBatchRowModel.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.collect.Lists;
import fr.ifremer.tutti.persistence.entities.data.Attachment;
import fr.ifremer.tutti.persistence.entities.data.SampleCategoryEnum;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatch;
import fr.ifremer.tutti.persistence.entities.data.SpeciesBatchFrequency;
import fr.ifremer.tutti.persistence.entities.referential.CaracteristicQualitativeValue;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.AbstractTuttiBeanUIModel;
import fr.ifremer.tutti.ui.swing.content.operation.catches.species.frequency.SpeciesFrequencyRowModel;
import org.apache.commons.collections.CollectionUtils;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.Collections;
import java.util.List;

/**
 * Represents a species batch (i.e a row in the batch table).
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class SpeciesBatchRowModel extends AbstractTuttiBeanUIModel<SpeciesBatch, SpeciesBatchRowModel> {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_SPECIES_TO_CONFIRM = "speciesToConfirm";

    public static final String PROPERTY_SPECIES = "species";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY = "sortedUnsortedCategory";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_VALUE = "sortedUnsortedCategoryValue";

    public static final String PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT = "sortedUnsortedCategoryWeight";

    public static final String PROPERTY_SIZE_CATEGORY = "sizeCategory";

    public static final String PROPERTY_SIZE_CATEGORY_VALUE = "sizeCategoryValue";

    public static final String PROPERTY_SIZE_CATEGORY_WEIGHT = "sizeCategoryWeight";

    public static final String PROPERTY_SEX_CATEGORY = "sexCategory";

    public static final String PROPERTY_SEX_CATEGORY_VALUE = "sexCategoryValue";

    public static final String PROPERTY_SEX_CATEGORY_WEIGHT = "sexCategoryWeight";

    public static final String PROPERTY_MATURITY_CATEGORY = "maturityCategory";

    public static final String PROPERTY_MATURITY_CATEGORY_VALUE = "maturityCategoryValue";

    public static final String PROPERTY_MATURITY_CATEGORY_WEIGHT = "maturityCategoryWeight";

    public static final String PROPERTY_AGE_CATEGORY = "ageCategory";

    public static final String PROPERTY_AGE_CATEGORY_VALUE = "ageCategoryValue";

    public static final String PROPERTY_AGE_CATEGORY_WEIGHT = "ageCategoryWeight";

    public static final String PROPERTY_WEIGHT = "weight";

    public static final String PROPERTY_NUMBER = "number";

    public static final String PROPERTY_COMMENT = "comment";

    public static final String PROPERTY_FREQUENCY = "frequency";

    public static final String PROPERTY_COMPUTED_NUMBER = "computedNumber";

    public static final String PROPERTY_COMPUTED_WEIGHT = "computedWeight";

    public static final String PROPERTY_ATTACHMENTS = "attachments";

    public static final String PROPERTY_BATCH_PARENT = "batchParent";

    public static final String PROPERTY_BATCH_CHILD = "batchChild";

    public static final String PROPERTY_BATCH_LEAF = "batchLeaf";

    public static final String PROPERTY_BATCH_ROOT = "batchRoot";

    public static final String PROPERTY_SAMPLE_CATEGORY = "sampleCategory";

    /**
     * Species.
     *
     * @since 0.3
     */
    protected Species species;

    /**
     * Is catch is vrac or  horsVrac?.
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sortedUnsortedCategory;

    /**
     * Weight category (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sizeCategory;

    /**
     * Sex (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> sexCategory;

    /**
     * Maturity (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<CaracteristicQualitativeValue> maturityCategory;

    /**
     * Age (can be null).
     *
     * @since 0.2
     */
    protected SampleCategory<Float> ageCategory;

    /**
     * Sample category of this batch.
     * <p/>
     * This is one of the sample category befined above, used to know if the
     * table row which sample category affects this row exactly.
     *
     * @since 0.3
     */
    private SampleCategory<?> sampleCategory;

    /**
     * Is the species need to be confirmed?.
     *
     * @since 0.2
     */
    protected boolean speciesToConfirm;

    /**
     * Observed weight.
     *
     * @since 0.2
     */
    protected Float weight;

    /**
     * Observed number.
     *
     * @since 0.3
     */
    protected Integer number;

    /**
     * Total computed weight (from frequencies).
     *
     * @since 0.2
     */
    protected Float computedWeight;

    /**
     * Total computed number (from frequencies).
     *
     * @since 0.2
     */
    protected Float computedNumber;

    /**
     * Comment on this catches.
     *
     * @since 0.2
     */
    protected String comment;

    /**
     * Attachments
     *
     * @since 0.2
     */
    protected List<Attachment> attachments;

    /**
     * List of frequencies observed for this batch.
     *
     * @since 0.2
     */
    protected List<SpeciesFrequencyRowModel> frequency;

    /**
     * Parent of this batch (can be null if batch is root).
     *
     * @see #isBatchRoot()
     * @since 0.3
     */
    protected SpeciesBatchRowModel batchParent;

    /**
     * List of child batches (can be null or empty if batch is a leaf).
     *
     * @see #isBatchLeaf()
     * @since 0.3
     */
    protected List<SpeciesBatchRowModel> batchChild;

    protected static final Binder<SpeciesBatch, SpeciesBatchRowModel> fromBeanBinder =
            BinderFactory.newBinder(SpeciesBatch.class,
                                    SpeciesBatchRowModel.class);

    protected static final Binder<SpeciesBatchRowModel, SpeciesBatch> toBeanBinder =
            BinderFactory.newBinder(SpeciesBatchRowModel.class,
                                    SpeciesBatch.class);

    public SpeciesBatchRowModel() {
        super(SpeciesBatch.class, fromBeanBinder, toBeanBinder);
        frequency = Lists.newArrayList();
        sortedUnsortedCategory = SampleCategory.newSample(SampleCategoryType.sortedUnsorted);
        sizeCategory = SampleCategory.newSample(SampleCategoryType.size);
        sexCategory = SampleCategory.newSample(SampleCategoryType.sex);
        maturityCategory = SampleCategory.newSample(SampleCategoryType.maturity);
        ageCategory = SampleCategory.newSample(SampleCategoryType.age);
    }

    public SpeciesBatchRowModel(SpeciesBatch aBatch,
                                List<SpeciesBatchFrequency> frequencies) {
        this();
        fromBean(aBatch);
        List<SpeciesFrequencyRowModel> frequencyRows =
                SpeciesFrequencyRowModel.fromBeans(frequencies);
        frequency.addAll(frequencyRows);
        Collections.sort(frequency);
    }

    //------------------------------------------------------------------------//
    //-- Sample category                                                    --//
    //------------------------------------------------------------------------//

    public SampleCategory<?> getSampleCategory() {
        return sampleCategory;
    }

    public void setSampleCategory(SampleCategory<?> sampleCategory) {
        Object oldValue = getSampleCategory();
        this.sampleCategory = sampleCategory;
        firePropertyChange(PROPERTY_SAMPLE_CATEGORY, oldValue, sampleCategory);
    }

    public SampleCategory<?> getSampleCategory(SampleCategoryEnum sampleCategoryEnum) {
        switch (sampleCategoryEnum) {
            case sortedUnsorted:
                sampleCategory = getSortedUnsortedCategory();
                break;
            case size:
                sampleCategory = getSizeCategory();
                break;
            case sex:
                sampleCategory = getSexCategory();
                break;
            case maturity:
                sampleCategory = getMaturityCategory();
                break;
            case age:
                sampleCategory = getAgeCategory();
                break;
            default:
        }
        return sampleCategory;
    }

    //------------------------------------------------------------------------//
    //-- Species category                                                   --//
    //------------------------------------------------------------------------//

    public Species getSpecies() {
        return species;
    }

    public void setSpecies(Species species) {
        Object oldCategory = getSpecies();
        this.species = species;
        firePropertyChange(PROPERTY_SPECIES, oldCategory, species);
    }

    //------------------------------------------------------------------------//
    //-- SortedUnsorted category                                            --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSortedUnsortedCategory() {
        return sortedUnsortedCategory;
    }

    public void setSortedUnsortedCategory(SampleCategory<CaracteristicQualitativeValue> sortedUnsortedCategory) {
        Object oldCategory = getSortedUnsortedCategory();
        Object oldValue = getSortedUnsortedCategoryValue();
        Object oldWeight = getSortedUnsortedCategoryWeight();
        this.sortedUnsortedCategory = sortedUnsortedCategory;
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY, oldCategory, sortedUnsortedCategory);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_VALUE, oldValue, getSortedUnsortedCategoryValue());
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT, oldWeight, getSortedUnsortedCategoryWeight());
    }

    public CaracteristicQualitativeValue getSortedUnsortedCategoryValue() {
        return sortedUnsortedCategory.getCategoryValue();
    }

    public Float getSortedUnsortedCategoryWeight() {
        return sortedUnsortedCategory.getCategoryWeight();
    }

    public void setSortedUnsortedCategoryWeight(Float sortedUnsortedCategoryWeight) {
        Object oldValue = getSortedUnsortedCategoryWeight();
        sortedUnsortedCategory.setCategoryWeight(sortedUnsortedCategoryWeight);
        firePropertyChange(PROPERTY_SORTED_UNSORTED_CATEGORY_WEIGHT, oldValue, sortedUnsortedCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Size category                                                      --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSizeCategory() {
        return sizeCategory;
    }

    public void setSizeCategory(SampleCategory<CaracteristicQualitativeValue> sizeCategory) {
        Object oldCategory = getSizeCategory();
        Object oldValue = getSizeCategoryValue();
        Object oldWeight = getSizeCategoryWeight();
        this.sizeCategory = sizeCategory;
        firePropertyChange(PROPERTY_SIZE_CATEGORY, oldCategory, sizeCategory);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_VALUE, oldValue, getSizeCategoryValue());
        firePropertyChange(PROPERTY_SIZE_CATEGORY_WEIGHT, oldWeight, getSizeCategoryWeight());
    }

    public CaracteristicQualitativeValue getSizeCategoryValue() {
        return sizeCategory.getCategoryValue();
    }

    public Float getSizeCategoryWeight() {
        return sizeCategory.getCategoryWeight();
    }

    public void setSizeCategoryWeight(Float sizeCategoryWeight) {
        Object oldValue = getSizeCategoryWeight();
        this.sizeCategory.setCategoryWeight(sizeCategoryWeight);
        firePropertyChange(PROPERTY_SIZE_CATEGORY_WEIGHT, oldValue, sizeCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Sex category                                                       --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getSexCategory() {
        return sexCategory;
    }

    public void setSexCategory(SampleCategory<CaracteristicQualitativeValue> sexCategory) {
        Object oldCategory = getSexCategory();
        Object oldValue = getSexCategoryValue();
        Object oldWeight = getSexCategoryWeight();
        this.sexCategory = sexCategory;
        firePropertyChange(PROPERTY_SEX_CATEGORY, oldCategory, sexCategory);
        firePropertyChange(PROPERTY_SEX_CATEGORY_VALUE, oldValue, getSexCategoryValue());
        firePropertyChange(PROPERTY_SEX_CATEGORY_WEIGHT, oldWeight, getSexCategoryWeight());
    }

    public CaracteristicQualitativeValue getSexCategoryValue() {
        return sexCategory.getCategoryValue();
    }

    public Float getSexCategoryWeight() {
        return sexCategory.getCategoryWeight();
    }

    public void setSexCategoryWeight(Float sexCategoryWeight) {
        Object oldValue = getSexCategoryWeight();
        this.sexCategory.setCategoryWeight(sexCategoryWeight);
        firePropertyChange(PROPERTY_SEX_CATEGORY_WEIGHT, oldValue, sexCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Maturity category                                                  --//
    //------------------------------------------------------------------------//

    public SampleCategory<CaracteristicQualitativeValue> getMaturityCategory() {
        return maturityCategory;
    }

    public void setMaturityCategory(SampleCategory<CaracteristicQualitativeValue> maturityCategory) {
        Object oldCategory = getMaturityCategory();
        Object oldValue = getMaturityCategoryValue();
        Object oldWeight = getMaturityCategoryWeight();
        this.maturityCategory = maturityCategory;
        firePropertyChange(PROPERTY_MATURITY_CATEGORY, oldCategory, maturityCategory);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_VALUE, oldValue, getMaturityCategoryValue());
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_WEIGHT, oldWeight, getMaturityCategoryWeight());
    }

    public CaracteristicQualitativeValue getMaturityCategoryValue() {
        return maturityCategory.getCategoryValue();
    }

    public Float getMaturityCategoryWeight() {
        return maturityCategory.getCategoryWeight();
    }

    public void setMaturityCategoryWeight(Float maturityCategoryWeight) {
        Object oldValue = getMaturityCategoryWeight();
        this.maturityCategory.setCategoryWeight(maturityCategoryWeight);
        firePropertyChange(PROPERTY_MATURITY_CATEGORY_WEIGHT, oldValue, maturityCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Age category                                                       --//
    //------------------------------------------------------------------------//

    public SampleCategory<Float> getAgeCategory() {
        return ageCategory;
    }

    public void setAgeCategory(SampleCategory<Float> ageCategory) {
        Object oldCategory = getAgeCategory();
        Object oldValue = getAgeCategoryValue();
        Object oldWeight = getAgeCategoryWeight();
        this.ageCategory = ageCategory;

        firePropertyChange(PROPERTY_AGE_CATEGORY, oldCategory, ageCategory);
        firePropertyChange(PROPERTY_AGE_CATEGORY_VALUE, oldValue, getAgeCategoryValue());
        firePropertyChange(PROPERTY_AGE_CATEGORY_WEIGHT, oldWeight, getAgeCategoryWeight());
    }

    public Float getAgeCategoryValue() {
        return ageCategory.getCategoryValue();
    }

    public Float getAgeCategoryWeight() {
        return ageCategory.getCategoryWeight();
    }

    public void setAgeCategoryWeight(Float ageCategoryWeight) {
        Object oldValue = getAgeCategoryWeight();
        ageCategory.setCategoryWeight(ageCategoryWeight);
        firePropertyChange(PROPERTY_AGE_CATEGORY_WEIGHT, oldValue, ageCategoryWeight);
    }

    //------------------------------------------------------------------------//
    //-- Navigation properties                                              --//
    //------------------------------------------------------------------------//

    public SpeciesBatchRowModel getBatchParent() {
        return batchParent;
    }

    public void setBatchParent(SpeciesBatchRowModel batchParent) {
        Object oldValue = getBatchParent();
        this.batchParent = batchParent;
        firePropertyChange(PROPERTY_BATCH_PARENT, oldValue, batchParent);
        firePropertyChange(PROPERTY_BATCH_ROOT, null, isBatchRoot());
    }

    public List<SpeciesBatchRowModel> getBatchChild() {
        return batchChild;
    }

    public void setBatchChilds(List<SpeciesBatchRowModel> batchChild) {
        this.batchChild = batchChild;
        // force to propagate child changes
        firePropertyChange(PROPERTY_BATCH_CHILD, null, batchChild);
        firePropertyChange(PROPERTY_BATCH_LEAF, null, isBatchLeaf());
    }

    public boolean isBatchLeaf() {
        return CollectionUtils.isEmpty(batchChild);
    }

    public boolean isBatchRoot() {
        return batchParent == null;
    }

    //------------------------------------------------------------------------//
    //-- Other properties                                                   --//
    //------------------------------------------------------------------------//

    public Boolean getSpeciesToConfirm() {
        return speciesToConfirm;
    }

    public void setSpeciesToConfirm(Boolean speciesToConfirm) {
        Object oldValue = getSpeciesToConfirm();
        this.speciesToConfirm = speciesToConfirm;
        firePropertyChange(PROPERTY_SPECIES_TO_CONFIRM, oldValue, speciesToConfirm);
    }

    public Float getWeight() {
        return weight;
    }

    public void setWeight(Float weight) {
        Object oldValue = getWeight();
        this.weight = weight;
        firePropertyChange(PROPERTY_WEIGHT, oldValue, weight);
    }

    public Integer getNumber() {
        return number;
    }

    public void setNumber(Integer number) {
        Object oldValue = getNumber();
        this.number = number;
        firePropertyChange(PROPERTY_NUMBER, oldValue, number);
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        Object oldValue = getComment();
        this.comment = comment;
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    public List<Attachment> getAttachments() {
        return attachments;
    }

    public void setAttachments(List<Attachment> attachments) {
        Object oldValue = getAttachments();
        this.attachments = attachments;
        firePropertyChange(PROPERTY_ATTACHMENTS, oldValue, attachments);
    }

    public List<SpeciesFrequencyRowModel> getFrequency() {
        return frequency;
    }

    public void setFrequency(List<SpeciesFrequencyRowModel> frequency) {
        this.frequency = frequency;
        // force to propagate frequencies changes
        firePropertyChange(PROPERTY_FREQUENCY, null, frequency);
    }

    public Float getComputedWeight() {
        return computedWeight;
    }

    public void setComputedWeight(Float computedWeight) {
        Object oldValue = getComputedWeight();
        this.computedWeight = computedWeight;
        firePropertyChange(PROPERTY_COMPUTED_WEIGHT, oldValue,
                           computedWeight);
    }

    public Float getComputedNumber() {
        return computedNumber;
    }

    public void setComputedNumber(Float computedNumber) {
        Object oldValue = getComputedNumber();
        this.computedNumber = computedNumber;
        firePropertyChange(PROPERTY_COMPUTED_NUMBER, oldValue,
                           computedNumber);
    }

}
