package fr.ifremer.tutti.ui.swing.content.operation.species;

/*
 * #%L
 * Tutti :: UI
 * $Id: FrequencyCellComponent.java 81 2012-12-17 07:39:44Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/content/operation/species/FrequencyCellComponent.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import fr.ifremer.tutti.ui.swing.util.table.AbstractSelectTableAction;
import fr.ifremer.tutti.ui.swing.util.table.ColumnIdentifier;
import jaxx.runtime.SwingUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.swing.AbstractAction;
import javax.swing.AbstractCellEditor;
import javax.swing.JButton;
import javax.swing.JTable;
import javax.swing.border.LineBorder;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import java.awt.Color;
import java.awt.Component;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import static org.nuiton.i18n.I18n._;

/**
 * Component to render and edit frequency stuff from batch table.
 *
 * @author tchemit <chemit@codelutin.com>
 * @since 0.2
 */
public class FrequencyCellComponent extends JButton {

    private static final long serialVersionUID = 1L;

    public static final String ROW_INDEX = "rowIndex";

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(FrequencyCellComponent.class);

    public FrequencyCellComponent() {
        setContentAreaFilled(false);
        setOpaque(false);
        setIcon(SwingUtil.createActionIcon("show-frequency"));
    }

    public static TableCellRenderer newRender() {
        return new FrequencyCellRenderer();
    }

    public static TableCellEditor newEditor(SpeciesFrequencyUI ui) {
        return new FrequencyCellEditor(ui);
    }

    public static class FrequencyCellEditor extends AbstractCellEditor implements TableCellEditor {

        private static final long serialVersionUID = 1L;

        protected final FrequencyCellComponent component;

        protected final SpeciesFrequencyUI ui;

        protected Frame frame;

        protected JTable table;

        protected SpeciesBatchTableModel tableModel;

        protected ColumnIdentifier<SpeciesBatchRowModel> columnIdentifier;

        protected SpeciesBatchRowModel editRow;

        protected Integer rowIndex;

        protected Integer columnIndex;

        public FrequencyCellEditor(SpeciesFrequencyUI ui) {
            this.ui = ui;
            component = new FrequencyCellComponent();
            component.setBorder(new LineBorder(Color.BLACK));
            component.addKeyListener(new KeyAdapter() {
                @Override
                public void keyReleased(KeyEvent e) {
                    if (e.getKeyCode() == KeyEvent.VK_ENTER ||
                        e.getKeyCode() == KeyEvent.VK_SPACE) {
                        e.consume();
                        startEdit();
                    }
                }
            });

            component.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    e.consume();
                    startEdit();
                }
            });
        }

        protected void startEdit() {

            Preconditions.checkNotNull(tableModel, "No table model assigned.");

            // open frequency dialog

            Preconditions.checkNotNull(editRow, "No editRow found.");

            if (log.isInfoEnabled()) {
                log.info("Will edit frequencies for row: " + rowIndex);
            }

            if (frame == null) {
                frame = SwingUtil.getParentContainer(ui, Frame.class);
            }

            ui.editBatch(editRow);

            // open frequency dialog
            TuttiUIUtil.openInDialog(
                    ui, frame, _("tutti.title.frequency"), true,
                    new AbstractAction() {
                        private static final long serialVersionUID = 1L;

                        @Override
                        public void actionPerformed(ActionEvent e) {
                            ui.getHandler().close();
                        }
                    });

            // at close, synch back frequencies

            editRow.updateTotalFromFrequencies();

            int r = rowIndex;
            int c = columnIndex;

            // stop edition
            stopCellEditing();

            // reselect this cell
            AbstractSelectTableAction.doSelectCell(table, r, c);
        }

        @Override
        public Component getTableCellEditorComponent(JTable table,
                                                     Object value,
                                                     boolean isSelected,
                                                     int row,
                                                     int column) {
            if (tableModel == null) {
                tableModel = (SpeciesBatchTableModel) table.getModel();
                this.table = table;
                columnIdentifier = tableModel.getPropertyName(column);
            }
            String text;
            if (value == null) {
                text = " - ";
            } else {
                text = String.valueOf(value);
            }
            component.setText(text);

            rowIndex = row;
            columnIndex = column;

            editRow = tableModel.getEntry(row);

            return component;
        }

        @Override
        public Object getCellEditorValue() {

            Preconditions.checkNotNull(editRow, "No editRow found in editor.");

            Object result;
            if (columnIdentifier == SpeciesBatchTableModel.COMPUTED_NUMBER) {
                result = editRow.getComputedNumber();
            } else {
                result = editRow.getComputedWeight();
            }
            if (log.isInfoEnabled()) {
                log.info("editor value (" + columnIdentifier + "): " + result);
            }

            return result;
        }

        @Override
        public boolean stopCellEditing() {
            boolean b = super.stopCellEditing();
            if (b) {
                rowIndex = null;
                editRow = null;
                columnIndex = null;
            }
            return b;
        }

        @Override
        public void cancelCellEditing() {
            super.cancelCellEditing();
            rowIndex = null;
            columnIndex = null;
            editRow = null;
        }
    }

    public static class FrequencyCellRenderer implements TableCellRenderer {

        protected final FrequencyCellComponent component;

        public FrequencyCellRenderer() {
            component = new FrequencyCellComponent();
        }

        @Override
        public Component getTableCellRendererComponent(JTable table,
                                                       Object value,
                                                       boolean isSelected,
                                                       boolean hasFocus,
                                                       int row,
                                                       int column) {

            String text;
            if (value == null) {
                text = " - ";
            } else {
                text = String.valueOf(value);
            }
            boolean editable = table.isCellEditable(row, column);
            component.setEnabled(editable);
            component.setText(text);
            return component;
        }
    }
}
