package fr.ifremer.tutti.ui.swing;

/*
 * #%L
 * Tutti :: UI
 * $Id: MainUIHandler.java 55 2012-12-13 01:57:28Z tchemit $
 * $HeadURL: http://svn.forge.codelutin.com/svn/tutti/tags/tutti-0.2/tutti-ui-swing/src/main/java/fr/ifremer/tutti/ui/swing/MainUIHandler.java $
 * %%
 * Copyright (C) 2012 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import fr.ifremer.tutti.persistence.entities.data.Cruise;
import fr.ifremer.tutti.persistence.entities.data.Program;
import fr.ifremer.tutti.service.PersistenceService;
import fr.ifremer.tutti.service.config.TuttiServiceConfigOption;
import fr.ifremer.tutti.ui.swing.config.TuttiConfig;
import fr.ifremer.tutti.ui.swing.config.TuttiConfigOption;
import fr.ifremer.tutti.ui.swing.content.cruise.EditCruiseUI;
import fr.ifremer.tutti.ui.swing.content.home.SelectCruiseUI;
import fr.ifremer.tutti.ui.swing.content.operation.FishingOperationsUI;
import fr.ifremer.tutti.ui.swing.content.program.EditProgramUI;
import fr.ifremer.tutti.ui.swing.util.TuttiUIUtil;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.AboutPanel;
import jaxx.runtime.swing.ErrorDialogUI;
import jaxx.runtime.swing.editor.config.ConfigUIHelper;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.ApplicationConfig;
import org.nuiton.widget.SwingSession;

import javax.swing.JComponent;
import java.awt.Desktop;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.util.Calendar;

import static org.nuiton.i18n.I18n._;
import static org.nuiton.i18n.I18n.n_;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class MainUIHandler extends AbstractTuttiUIHandler<TuttiUIContext> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(MainUIHandler.class);

    protected MainUI ui;

    protected JComponent currentBody;

    protected final PersistenceService persistenceService;

    protected MainUIHandler(TuttiUIContext context, MainUI ui) {
        super(context);
        this.ui = ui;
        this.persistenceService = context.getService(PersistenceService.class);
    }

    @Override
    public void beforeInitUI() {
        context.addPropertyChangeListener(new PropertyChangeListener() {
            @Override
            public void propertyChange(PropertyChangeEvent evt) {
                String propertyName = evt.getPropertyName();
                if (propertyName.equals(TuttiUIContext.PROPERTY_PROGRAM_ID) ||
                    propertyName.equals(TuttiUIContext.PROPERTY_CRUISE_ID)) {

                    // change the ui title
                    changeTitle();

                } else if (propertyName.equals(TuttiUIContext.PROPERTY_SCREEN)) {
                    setScreen((TuttiScreen) evt.getNewValue());
                }
            }
        });
    }

    @Override
    public void afterInitUI() {

        // Sync to error dialog
        ErrorDialogUI.init(ui);

        // Init SwingSession
        SwingSession swingSession = context.getSwingSession();
        swingSession.add(ui);
        swingSession.save();

        //SwingUtil.getLayer(ui.getBody()).setUI(new BetaLayerUI());

        changeTitle();

        TuttiScreen screen;

        if (context.isCruiseContextFilled()) {

            // direct fill catches screen
            screen = TuttiScreen.EDIT_FISHING_OPERATION;
        } else {

            // go to select cruise screen
            screen = TuttiScreen.SELECT_CRUISE;
        }

        context.setScreen(screen);
    }

    @Override
    public void onCloseUI() {

        closeCurrentBody();

        context.close();
    }

    @Override
    protected TuttiUIContext getModel() {
        return ui.getModel();
    }

    public void showConfig() {
        TuttiConfig config = context.getConfig();

        ConfigUIHelper helper = new ConfigUIHelper(config.getApplicationConfig());

        helper.registerCallBack(
                "ui", n_("tutti.action.reload.ui"),
                SwingUtil.createActionIcon("reload-ui"),
                new Runnable() {

                    @Override
                    public void run() {
                        reloadTutti();
                    }
                }
        ).registerCallBack(
                "home", n_("tutti.action.reload.home"),
                SwingUtil.createActionIcon("config"),
                new Runnable() {

                    @Override
                    public void run() {
                        if (TuttiScreen.SELECT_CRUISE == context.getScreen()) {
                            // Reselect home cruise
//                            SelectCampaignUI homeUI = (SelectCampaignUI) currentBody;
//                            homeUI.getHandler().selectCampaign();
                        }
                    }
                }
        ).registerCallBack(
                "actions", n_("tutti.action.reload.actions"),
                SwingUtil.createActionIcon("config"),
                new Runnable() {

                    @Override
                    public void run() {

                        //TODO Reload actions shortcuts
                    }
                }
        );

        // APPLICATION

        helper.addCategory(n_("tutti.config.category.applications"),
                           n_("tutti.config.category.applications.description"))
                .addOption(TuttiServiceConfigOption.DATA_DIRECTORY)
                .setOptionCallBack("ui");
//                .addOption(TuttiConfigOption.AUTO_COMMIT_DELAY) // milliseconds
//                .setOptionCallBack("ui");

        // SHORTCUT

//        helper.addCategory(n_("tutti.config.category.shortcuts"),
//                           n_("tutti.config.category.shortcuts.description"), "actions")
//                .addOption(TuttiConfigOption.KEY_START)
//                .addOption(TuttiConfigOption.KEY_STOP)
//                .addOption(TuttiConfigOption.KEY_BEGIN)
//                .addOption(TuttiConfigOption.KEY_END)
//                .addOption(TuttiConfigOption.KEY_NEXT)
//                .addOption(TuttiConfigOption.KEY_ADD)
//                .addOption(TuttiConfigOption.KEY_LEFT_OBSERVATION)
//                .addOption(TuttiConfigOption.KEY_CENTER_OBSERVATION)
//                .addOption(TuttiConfigOption.KEY_RIGHT_OBSERVATION)
//                .addOption(TuttiConfigOption.KEY_CIRCLE_BACK)
//                .addOption(TuttiConfigOption.KEY_VALID_FLIGHT)
//                .addOption(TuttiConfigOption.KEY_VALID_TRANSECT)
//                .addOption(TuttiConfigOption.KEY_VALID_OBSERVATION)
//                .addOption(TuttiConfigOption.KEY_VALID_ROUTE);

        // OTHER

        helper.addCategory(n_("tutti.config.category.other"),
                           n_("tutti.config.category.other.description"))
                .addOption(TuttiServiceConfigOption.SITE_URL)
                .addOption(TuttiConfigOption.UI_CONFIG_FILE)
                .setOptionCallBack("ui");


        helper.buildUI(ui, "tutti.config.category.applications");

        helper.displayUI(ui, false);
    }

    public void closeTutti() {
        closeTutti(true);
    }

    public void showManageProtocol() {
        context.setScreen(TuttiScreen.EDIT_PROTOCOL);
    }

    public void showSelectCampaign() {
        context.setScreen(TuttiScreen.SELECT_CRUISE);
    }

    public void showFillCatches() {
        context.setScreen(TuttiScreen.EDIT_FISHING_OPERATION);
    }

    public void showImportScreen() {

    }

    public void showExportScreen() {

    }

    public void showAllegroScreen() {

    }

    public void showAbout() {

        ApplicationConfig applicationConfig =
                context.getConfig().getApplicationConfig();

        String iconPath = applicationConfig.getOption("application.icon.path");
        String name = "tutti-ui-swing";
        String licensePath = "META-INF/" + name + "-LICENSE.txt";
        String thirdPartyPath = "META-INF/" + name + "-THIRD-PARTY.txt";

        AboutPanel about = new AboutPanel();
        about.setTitle(_("tutti.title.about"));
        about.setAboutText(_("tutti.about.message"));

        TuttiConfig config = context.getConfig();
        int currentYear = Calendar.getInstance().get(Calendar.YEAR);
        int inceptionYear = config.getInceptionYear();
        String years;
        if (currentYear != inceptionYear) {
            years = inceptionYear + "-" + currentYear;
        } else {
            years = inceptionYear + "";
        }

        about.setBottomText(_("tutti.about.bottomText",
                              config.getOrganizationName(),
                              years,
                              config.getVersion()));
        about.setIconPath(iconPath);
        about.setLicenseFile(licensePath);
        about.setThirdpartyFile(thirdPartyPath);
        about.buildTopPanel();
        about.init();
        about.showInDialog(ui, true);

        // register on swing session
        context.getSwingSession().add(about);
    }

    public void gotoSite() {
        TuttiConfig config = context.getConfig();

        URL siteURL = config.getSiteUrl();

        if (log.isDebugEnabled()) {
            log.debug("goto " + siteURL);
        }
        if (Desktop.isDesktopSupported() &&
            Desktop.getDesktop().isSupported(Desktop.Action.BROWSE)) {
            try {
                Desktop.getDesktop().browse(siteURL.toURI());
            } catch (Exception e) {
                log.error("Failed to open '" + siteURL + "' in browser", e);
                ErrorDialogUI.showError(e);
            }
        }
    }

    protected void setScreen(TuttiScreen screen) {

        try {

            // busy ui
            TuttiUIUtil.updateBusyState(ui, true);

            // remove any screen
//            ui.setScreen(null);

            closeCurrentBody();

            String surveyId = context.getProgramId();
            String campaignId = context.getCruiseId();

            JComponent screenUI;
            String screenTitle;

            switch (screen) {
                default:
                case SELECT_CRUISE:

                    screenUI = new SelectCruiseUI(context);
                    screenTitle = _("tutti.title.home");
                    break;

                case EDIT_PROGRAM:

                    if (surveyId == null) {
                        screenTitle = _("tutti.title.create.program");
                    } else {
                        screenTitle = _("tutti.title.edit.program");
                    }

                    screenUI = new EditProgramUI(context);
                    break;

                case EDIT_CRUISE:

                    if (campaignId == null) {
                        screenTitle = _("tutti.title.create.cruise");
                    } else {
                        screenTitle = _("tutti.title.edit.cruise");
                    }

                    screenUI = new EditCruiseUI(context);
                    break;

                case EDIT_FISHING_OPERATION:

                    screenTitle = _("tutti.title.edit.operations", getSelectedCruiseTitle());
                    screenUI = new FishingOperationsUI(context);
                    break;
            }

            changeScreen(screenUI, screenTitle);

            // set new screen only if everything is ok
//            ui.setScreen(screen);
        } catch (Exception e) {

            ErrorDialogUI.showError(e);

            // go back to home
            context.setScreen(TuttiScreen.SELECT_CRUISE);
        } finally {

            // free ui
            TuttiUIUtil.updateBusyState(ui, false);
        }
    }

    protected void reloadTutti() {

        // Close the application
        closeTutti(false);

        // Re-open the context
        try {
            context.open();
        } catch (Exception ex) {
            ErrorDialogUI.showError(ex);
        }

        MainUI mainUI = new MainUI(context);
        mainUI.setVisible(true);
    }

    protected void closeCurrentBody() {
        if (currentBody != null) {
            TuttiUI<?, ?> body = (TuttiUI<?, ?>) currentBody;
            body.getHandler().onCloseUI();

            context.getSwingSession().save();

            ui.getBody().remove(currentBody);

            currentBody = null;
        }
    }

    protected void changeScreen(JComponent newBody, String title) {
        currentBody = newBody;
        context.getSwingSession().add(currentBody);
        ui.getBody().setTitle(title);
        ui.getBody().add(currentBody);
    }

    protected void closeTutti(boolean exit) {

        context.getSwingSession().save();

        ui.setVisible(false);
        ui.dispose();

        onCloseUI();

        if (exit) {
            System.exit(0);
        }
    }

    protected void changeTitle() {

        String title = getSelectedCruiseTitle();

        ui.setTitle("Tutti - v " + getConfig().getVersion() + " [ " + title + " ]");
    }

    protected String getSelectedCruiseTitle() {

        String surveyId = context.getProgramId();

        String title;

        if (surveyId == null) {

            // no program selected (so neither cruise)

            title = _("tutti.title.noSelectedProgram");

        } else {

            // selected program

            Program program = persistenceService.getProgram(surveyId);

            title = _("tutti.title.selectedProgram", program.getName()) + " / ";

            String campaignId = context.getCruiseId();

            if (campaignId == null) {

                // no selected cruise

                title += _("tutti.title.noSelectedCruise");


            } else {

                // selected cruise
                Cruise cruise;
                cruise = persistenceService.getCruise(campaignId);

                title += _("tutti.title.selectedCruise", cruise.getName());
            }
        }
        return title;
    }

}
